"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SpotEventPluginClient = exports.CollectionType = void 0;
/**
 * A type of collection to get/recive from Deadline.
 */
var CollectionType;
(function (CollectionType) {
    CollectionType["Pool"] = "pool";
    CollectionType["Group"] = "group";
})(CollectionType = exports.CollectionType || (exports.CollectionType = {}));
/**
 * Provides a simple interface to send requests to the Render Queue API related to the Deadline Spot Event Plugin.
 */
class SpotEventPluginClient {
    constructor(client) {
        this.deadlineClient = client;
    }
    async saveServerData(config) {
        console.log('Saving server data configuration:');
        console.log(config);
        try {
            // Get the concurrency token required to save server data
            const concurrencyToken = await this.concurrencyToken();
            await this.deadlineClient.PostRequest('/rcs/v1/putServerData', {
                ServerData: [
                    {
                        ID: SpotEventPluginClient.EVENT_PLUGIN_ID,
                        ServerDataDictionary: {
                            Config: config,
                        },
                        ConcurrencyToken: concurrencyToken,
                    },
                ],
            }, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            return true;
        }
        catch (e) {
            console.error(`Failed to save server data. Reason: ${e}`);
            return false;
        }
    }
    async configureSpotEventPlugin(configs) {
        console.log('Saving plugin configuration:');
        console.log(configs);
        try {
            await this.deadlineClient.PostRequest('/db/plugins/event/config/save', {
                ID: 'spot',
                DebugLogging: false,
                DlInit: configs,
                Icon: null,
                Limits: [],
                Meta: [],
                Name: 'Spot',
                PluginEnabled: 1,
            }, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            return true;
        }
        catch (e) {
            console.error(`Failed to save plugin configuration. Reason: ${e}`);
            return false;
        }
    }
    async addGroups(newGroups) {
        if (newGroups && newGroups.length) {
            const deadlineGroups = await this.getCollection(CollectionType.Group);
            if (deadlineGroups) {
                const newDeadlineGroups = deadlineGroups.Pools
                    .concat(newGroups
                    .filter(group => !deadlineGroups.Pools.includes(group)));
                return await this.saveCollection({
                    Pools: newDeadlineGroups,
                    ObsoletePools: deadlineGroups.ObsoletePools,
                }, CollectionType.Group);
            }
            return false;
        }
        return true;
    }
    async addPools(newPools) {
        if (newPools && newPools.length) {
            const deadlinePools = await this.getCollection(CollectionType.Pool);
            if (deadlinePools) {
                const newDeadlinePools = deadlinePools.Pools
                    .concat(newPools
                    .filter(pool => !deadlinePools.Pools.includes(pool)));
                return await this.saveCollection({
                    Pools: newDeadlinePools,
                    ObsoletePools: deadlinePools.ObsoletePools,
                }, CollectionType.Pool);
            }
            return false;
        }
        return true;
    }
    async getCollection(type) {
        console.log(`Getting ${type} collection:`);
        try {
            const response = await this.deadlineClient.GetRequest(`/db/settings/collections/${type}s?invalidateCache=true`, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            const deadlinePools = response.data;
            if (!deadlinePools.Pools || !Array.isArray(deadlinePools.Pools)) {
                console.error(`Failed to receive a ${type} collection. Invalid response: ${JSON.stringify(response.data)}.`);
                return undefined;
            }
            return deadlinePools;
        }
        catch (e) {
            console.error(`Failed to get ${type} collection. Reason: ${e.message}`);
            return undefined;
        }
    }
    async saveCollection(pools, type) {
        console.log(`Saving ${type} collection:`);
        console.log(pools);
        try {
            await this.deadlineClient.PostRequest(`/db/settings/collections/${type}s/save`, {
                Pools: pools.Pools,
                ObsoletePools: pools.ObsoletePools,
            }, {
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                },
            });
            return true;
        }
        catch (e) {
            console.error(`Failed to save ${type} collection. Reason: ${e.message}`);
            return false;
        }
    }
    async describeServerData() {
        return await this.deadlineClient.PostRequest('/rcs/v1/describeServerData', {
            ServerDataIds: [
                SpotEventPluginClient.EVENT_PLUGIN_ID,
            ],
        }, {
            headers: {
                'Content-Type': 'application/json; charset=utf-8',
            },
        });
    }
    /**
     * Requests a concurrencyToken required to save spot fleet request configuration.
     * If data already exists under the ID, an existing ConcurrencyToken has to be used.
     * First obtain the token and then save the data with the same ConcurrencyToken.
     * If there is no data under the ID, then real token is not required,
     * but the ConcurrencyToken property still has to be set.
     * NOTE:
     * saveServerData() will have a ConcurrencyToken in its response but we do not use it,
     * instead we always call this function to get a latest token.
     */
    async concurrencyToken() {
        var _a;
        const response = await this.describeServerData();
        const describedData = response.data;
        if (!describedData.ServerData || !Array.isArray(describedData.ServerData)) {
            throw new Error(`Failed to receive a ConcurrencyToken. Invalid response: ${describedData}.`);
        }
        const found = describedData.ServerData.find(element => element.ID === SpotEventPluginClient.EVENT_PLUGIN_ID);
        return (_a = found === null || found === void 0 ? void 0 : found.ConcurrencyToken) !== null && _a !== void 0 ? _a : '';
    }
}
exports.SpotEventPluginClient = SpotEventPluginClient;
SpotEventPluginClient.EVENT_PLUGIN_ID = 'event.plugin.spot';
//# sourceMappingURL=data:application/json;base64,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