"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const sinon = require("sinon");
const backoff_generator_1 = require("../backoff-generator");
describe('BackoffGenerator', () => {
    const base = 100;
    beforeEach(() => {
        jest.useFakeTimers();
    });
    afterAll(() => {
        jest.useRealTimers();
    });
    test('throws when jitterDivisor is invalid', () => {
        // GIVEN
        const jitterDivisor = 0;
        // WHEN
        expect(() => new backoff_generator_1.BackoffGenerator({
            jitterDivisor,
        }))
            // THEN
            .toThrow(`jitterDivisor must be greater than or equal to 1, got: ${jitterDivisor}`);
    });
    test('stops when max attempts is reached', async () => {
        // GIVEN
        const maxAttempts = 2;
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base,
            maxAttempts,
        });
        // WHEN
        let retvals = [];
        for (let i = 0; i < maxAttempts; i++) {
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base * Math.pow(2, i));
            const retval = await promise;
            retvals.push(retval);
        }
        // THEN
        retvals.slice(0, -1).forEach(retval => expect(retval).toBe(true));
        expect(retvals[retvals.length - 1]).toBe(false);
        expect(backoffGenerator.shouldContinue()).toBe(false);
    });
    test('stops when max cumulative backoff time is reached', async () => {
        // GIVEN
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base,
            maxCumulativeBackoffTimeMs: base,
        });
        // WHEN
        const promise = backoffGenerator.backoff();
        jest.advanceTimersByTime(base);
        const retval = await promise;
        // THEN
        expect(retval).toBe(false);
        expect(backoffGenerator.shouldContinue()).toBe(false);
    });
    test('respects max interval between backoffs', async () => {
        // GIVEN
        const maxIntervalMs = base / 2;
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base,
            maxIntervalMs,
        });
        // WHEN
        const promise = backoffGenerator.backoff();
        jest.advanceTimersByTime(maxIntervalMs);
        await promise;
        // THEN
        expect(maxIntervalMs).toBeLessThan(base);
        expect(setTimeout).toHaveBeenCalledTimes(1);
        expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), maxIntervalMs);
    });
    test('forces backoff', async () => {
        // GIVEN
        const backoffGenerator = new backoff_generator_1.BackoffGenerator({
            base,
            maxAttempts: 0,
        });
        if (backoffGenerator.shouldContinue() !== false) {
            throw new Error('BackoffGenerator.shouldContinue did not return false when it was expected to. Please fix this unit test.');
        }
        // WHEN
        const promise = backoffGenerator.backoff(true);
        jest.advanceTimersByTime(base);
        await promise;
        // THEN
        expect(setTimeout).toHaveBeenCalledTimes(1);
        expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), base);
    });
    describe('.restart()', () => {
        test('resets the number of attempts', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                maxAttempts: 1,
            });
            // This reaches maxAttempts, shouldContinue() will return false
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base);
            await promise;
            if (backoffGenerator.shouldContinue() !== false) {
                throw new Error('BackoffGenerator.shouldContinue did not return false when it was expected to. Please fix this unit test.');
            }
            // WHEN
            backoffGenerator.restart();
            // THEN
            expect(backoffGenerator.shouldContinue()).toBe(true);
        });
        test('resets the cumulative backoff time', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                maxCumulativeBackoffTimeMs: base,
            });
            // This reaches maxCumulativeBackoffTimeMs, shouldContinue() will return false
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base);
            await promise;
            if (backoffGenerator.shouldContinue() !== false) {
                throw new Error('BackoffGenerator.shouldContinue did not return false when it was expected to. Please fix this unit test.');
            }
            // WHEN
            backoffGenerator.restart();
            // THEN
            expect(backoffGenerator.shouldContinue()).toBe(true);
        });
    });
    describe('backs off and continues', () => {
        test('when there are remaining attempts', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                maxAttempts: 2,
            });
            // WHEN
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base);
            const retval = await promise;
            // THEN
            // We have one more attempt left, it should continue
            expect(retval).toBe(true);
            expect(backoffGenerator.shouldContinue()).toBe(true);
        });
        test('when there is remaining cumulative backoff time', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                maxCumulativeBackoffTimeMs: base + 1,
            });
            // WHEN
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(base);
            const retval = await promise;
            // THEN
            // We haven't reached max cumulative backoff time yet, it should continue
            expect(retval).toBe(true);
            expect(backoffGenerator.shouldContinue()).toBe(true);
        });
    });
    describe.each([
        0,
        0.25,
        0.5,
        0.75,
        1,
    ])('jitter (factor %d)', (factor) => {
        let randomStub;
        beforeAll(() => {
            randomStub = sinon.stub(Math, 'random').returns(factor);
        });
        afterAll(() => {
            randomStub.restore();
        });
        test('applies full jitter', async () => {
            // GIVEN
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                jitterDivisor: 1,
            });
            const interval = base * factor;
            // WHEN
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(interval);
            await promise;
            // THEN
            expect(setTimeout).toHaveBeenCalledTimes(1);
            expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), interval);
        });
        test('correctly calculates jitter with divisor', async () => {
            // GIVEN
            const jitterDivisor = 4;
            const backoffGenerator = new backoff_generator_1.BackoffGenerator({
                base,
                jitterDivisor,
            });
            const interval = (base - base / jitterDivisor) + Math.floor(base / jitterDivisor * factor);
            // WHEN
            const promise = backoffGenerator.backoff();
            jest.advanceTimersByTime(interval);
            await promise;
            // THEN
            expect(setTimeout).toHaveBeenCalledTimes(1);
            expect(setTimeout).toHaveBeenLastCalledWith(expect.any(Function), interval);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmFja29mZi1nZW5lcmF0b3IudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImJhY2tvZmYtZ2VuZXJhdG9yLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7R0FHRzs7QUFFSCwrQkFBK0I7QUFDL0IsNERBQXdEO0FBRXhELFFBQVEsQ0FBQyxrQkFBa0IsRUFBRSxHQUFHLEVBQUU7SUFDaEMsTUFBTSxJQUFJLEdBQUcsR0FBRyxDQUFDO0lBRWpCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7UUFDZCxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7SUFDdkIsQ0FBQyxDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsR0FBRyxFQUFFO1FBQ1osSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDO0lBQ3ZCLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtRQUNoRCxRQUFRO1FBQ1IsTUFBTSxhQUFhLEdBQUcsQ0FBQyxDQUFDO1FBRXhCLE9BQU87UUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxvQ0FBZ0IsQ0FBQztZQUNoQyxhQUFhO1NBQ2QsQ0FBQyxDQUFDO1lBRUgsT0FBTzthQUNKLE9BQU8sQ0FBQywwREFBMEQsYUFBYSxFQUFFLENBQUMsQ0FBQztJQUN4RixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxvQ0FBb0MsRUFBRSxLQUFLLElBQUksRUFBRTtRQUNwRCxRQUFRO1FBQ1IsTUFBTSxXQUFXLEdBQUcsQ0FBQyxDQUFDO1FBQ3RCLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxvQ0FBZ0IsQ0FBQztZQUM1QyxJQUFJO1lBQ0osV0FBVztTQUNaLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLE9BQU8sR0FBRyxFQUFFLENBQUM7UUFDakIsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLFdBQVcsRUFBRSxDQUFDLEVBQUUsRUFBRTtZQUNwQyxNQUFNLE9BQU8sR0FBRyxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUMzQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEQsTUFBTSxNQUFNLEdBQUcsTUFBTSxPQUFPLENBQUM7WUFDN0IsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUN0QjtRQUVELE9BQU87UUFDUCxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUNsRSxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDaEQsTUFBTSxDQUFDLGdCQUFnQixDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3hELENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLG1EQUFtRCxFQUFFLEtBQUssSUFBSSxFQUFFO1FBQ25FLFFBQVE7UUFDUixNQUFNLGdCQUFnQixHQUFHLElBQUksb0NBQWdCLENBQUM7WUFDNUMsSUFBSTtZQUNKLDBCQUEwQixFQUFFLElBQUk7U0FDakMsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sT0FBTyxHQUFHLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQzNDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvQixNQUFNLE1BQU0sR0FBRyxNQUFNLE9BQU8sQ0FBQztRQUU3QixPQUFPO1FBQ1AsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUMzQixNQUFNLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDeEQsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsd0NBQXdDLEVBQUUsS0FBSyxJQUFJLEVBQUU7UUFDeEQsUUFBUTtRQUNSLE1BQU0sYUFBYSxHQUFHLElBQUksR0FBRyxDQUFDLENBQUM7UUFDL0IsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLG9DQUFnQixDQUFDO1lBQzVDLElBQUk7WUFDSixhQUFhO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sT0FBTyxHQUFHLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQzNDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUN4QyxNQUFNLE9BQU8sQ0FBQztRQUVkLE9BQU87UUFDUCxNQUFNLENBQUMsYUFBYSxDQUFDLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3pDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM1QyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsd0JBQXdCLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsRUFBRSxhQUFhLENBQUMsQ0FBQztJQUNuRixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxLQUFLLElBQUksRUFBRTtRQUNoQyxRQUFRO1FBQ1IsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLG9DQUFnQixDQUFDO1lBQzVDLElBQUk7WUFDSixXQUFXLEVBQUUsQ0FBQztTQUNmLENBQUMsQ0FBQztRQUNILElBQUksZ0JBQWdCLENBQUMsY0FBYyxFQUFFLEtBQUssS0FBSyxFQUFFO1lBQy9DLE1BQU0sSUFBSSxLQUFLLENBQUMsMEdBQTBHLENBQUMsQ0FBQztTQUM3SDtRQUVELE9BQU87UUFDUCxNQUFNLE9BQU8sR0FBRyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO1FBQy9CLE1BQU0sT0FBTyxDQUFDO1FBRWQsT0FBTztRQUNQLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM1QyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsd0JBQXdCLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUMxRSxDQUFDLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxZQUFZLEVBQUUsR0FBRyxFQUFFO1FBQzFCLElBQUksQ0FBQywrQkFBK0IsRUFBRSxLQUFLLElBQUksRUFBRTtZQUMvQyxRQUFRO1lBQ1IsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLG9DQUFnQixDQUFDO2dCQUM1QyxJQUFJO2dCQUNKLFdBQVcsRUFBRSxDQUFDO2FBQ2YsQ0FBQyxDQUFDO1lBQ0gsK0RBQStEO1lBQy9ELE1BQU0sT0FBTyxHQUFHLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQzNDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUMvQixNQUFNLE9BQU8sQ0FBQztZQUNkLElBQUksZ0JBQWdCLENBQUMsY0FBYyxFQUFFLEtBQUssS0FBSyxFQUFFO2dCQUMvQyxNQUFNLElBQUksS0FBSyxDQUFDLDBHQUEwRyxDQUFDLENBQUM7YUFDN0g7WUFFRCxPQUFPO1lBQ1AsZ0JBQWdCLENBQUMsT0FBTyxFQUFFLENBQUM7WUFFM0IsT0FBTztZQUNQLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxvQ0FBb0MsRUFBRSxLQUFLLElBQUksRUFBRTtZQUNwRCxRQUFRO1lBQ1IsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLG9DQUFnQixDQUFDO2dCQUM1QyxJQUFJO2dCQUNKLDBCQUEwQixFQUFFLElBQUk7YUFDakMsQ0FBQyxDQUFDO1lBQ0gsOEVBQThFO1lBQzlFLE1BQU0sT0FBTyxHQUFHLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQzNDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUMvQixNQUFNLE9BQU8sQ0FBQztZQUNkLElBQUksZ0JBQWdCLENBQUMsY0FBYyxFQUFFLEtBQUssS0FBSyxFQUFFO2dCQUMvQyxNQUFNLElBQUksS0FBSyxDQUFDLDBHQUEwRyxDQUFDLENBQUM7YUFDN0g7WUFFRCxPQUFPO1lBQ1AsZ0JBQWdCLENBQUMsT0FBTyxFQUFFLENBQUM7WUFFM0IsT0FBTztZQUNQLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHlCQUF5QixFQUFFLEdBQUcsRUFBRTtRQUN2QyxJQUFJLENBQUMsbUNBQW1DLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDbkQsUUFBUTtZQUNSLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxvQ0FBZ0IsQ0FBQztnQkFDNUMsSUFBSTtnQkFDSixXQUFXLEVBQUUsQ0FBQzthQUNmLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUMzQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDL0IsTUFBTSxNQUFNLEdBQUcsTUFBTSxPQUFPLENBQUM7WUFFN0IsT0FBTztZQUNQLG9EQUFvRDtZQUNwRCxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxpREFBaUQsRUFBRSxLQUFLLElBQUksRUFBRTtZQUNqRSxRQUFRO1lBQ1IsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLG9DQUFnQixDQUFDO2dCQUM1QyxJQUFJO2dCQUNKLDBCQUEwQixFQUFFLElBQUksR0FBRyxDQUFDO2FBQ3JDLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLE9BQU8sR0FBRyxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUMzQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDL0IsTUFBTSxNQUFNLEdBQUcsTUFBTSxPQUFPLENBQUM7WUFFN0IsT0FBTztZQUNQLHlFQUF5RTtZQUN6RSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzFCLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLElBQUksQ0FBQztRQUNaLENBQUM7UUFDRCxJQUFJO1FBQ0osR0FBRztRQUNILElBQUk7UUFDSixDQUFDO0tBQ0YsQ0FBQyxDQUFDLG9CQUFvQixFQUFFLENBQUMsTUFBYyxFQUFFLEVBQUU7UUFDMUMsSUFBSSxVQUEyQixDQUFDO1FBQ2hDLFNBQVMsQ0FBQyxHQUFHLEVBQUU7WUFDYixVQUFVLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzFELENBQUMsQ0FBQyxDQUFDO1FBRUgsUUFBUSxDQUFDLEdBQUcsRUFBRTtZQUNaLFVBQVUsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUN2QixDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxxQkFBcUIsRUFBRSxLQUFLLElBQUksRUFBRTtZQUNyQyxRQUFRO1lBQ1IsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLG9DQUFnQixDQUFDO2dCQUM1QyxJQUFJO2dCQUNKLGFBQWEsRUFBRSxDQUFDO2FBQ2pCLENBQUMsQ0FBQztZQUNILE1BQU0sUUFBUSxHQUFHLElBQUksR0FBRyxNQUFNLENBQUM7WUFFL0IsT0FBTztZQUNQLE1BQU0sT0FBTyxHQUFHLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQzNDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUNuQyxNQUFNLE9BQU8sQ0FBQztZQUVkLE9BQU87WUFDUCxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDNUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLHdCQUF3QixDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDOUUsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsMENBQTBDLEVBQUUsS0FBSyxJQUFJLEVBQUU7WUFDMUQsUUFBUTtZQUNSLE1BQU0sYUFBYSxHQUFHLENBQUMsQ0FBQztZQUN4QixNQUFNLGdCQUFnQixHQUFHLElBQUksb0NBQWdCLENBQUM7Z0JBQzVDLElBQUk7Z0JBQ0osYUFBYTthQUNkLENBQUMsQ0FBQztZQUNILE1BQU0sUUFBUSxHQUFHLENBQUMsSUFBSSxHQUFHLElBQUksR0FBRyxhQUFhLENBQUMsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksR0FBRyxhQUFhLEdBQUcsTUFBTSxDQUFDLENBQUM7WUFFM0YsT0FBTztZQUNQLE1BQU0sT0FBTyxHQUFHLGdCQUFnQixDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQzNDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUNuQyxNQUFNLE9BQU8sQ0FBQztZQUVkLE9BQU87WUFDUCxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDNUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLHdCQUF3QixDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDOUUsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0ICogYXMgc2lub24gZnJvbSAnc2lub24nO1xuaW1wb3J0IHsgQmFja29mZkdlbmVyYXRvciB9IGZyb20gJy4uL2JhY2tvZmYtZ2VuZXJhdG9yJztcblxuZGVzY3JpYmUoJ0JhY2tvZmZHZW5lcmF0b3InLCAoKSA9PiB7XG4gIGNvbnN0IGJhc2UgPSAxMDA7XG5cbiAgYmVmb3JlRWFjaCgoKSA9PiB7XG4gICAgamVzdC51c2VGYWtlVGltZXJzKCk7XG4gIH0pO1xuXG4gIGFmdGVyQWxsKCgpID0+IHtcbiAgICBqZXN0LnVzZVJlYWxUaW1lcnMoKTtcbiAgfSk7XG5cbiAgdGVzdCgndGhyb3dzIHdoZW4gaml0dGVyRGl2aXNvciBpcyBpbnZhbGlkJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgaml0dGVyRGl2aXNvciA9IDA7XG5cbiAgICAvLyBXSEVOXG4gICAgZXhwZWN0KCgpID0+IG5ldyBCYWNrb2ZmR2VuZXJhdG9yKHtcbiAgICAgIGppdHRlckRpdmlzb3IsXG4gICAgfSkpXG5cbiAgICAvLyBUSEVOXG4gICAgICAudG9UaHJvdyhgaml0dGVyRGl2aXNvciBtdXN0IGJlIGdyZWF0ZXIgdGhhbiBvciBlcXVhbCB0byAxLCBnb3Q6ICR7aml0dGVyRGl2aXNvcn1gKTtcbiAgfSk7XG5cbiAgdGVzdCgnc3RvcHMgd2hlbiBtYXggYXR0ZW1wdHMgaXMgcmVhY2hlZCcsIGFzeW5jICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IG1heEF0dGVtcHRzID0gMjtcbiAgICBjb25zdCBiYWNrb2ZmR2VuZXJhdG9yID0gbmV3IEJhY2tvZmZHZW5lcmF0b3Ioe1xuICAgICAgYmFzZSxcbiAgICAgIG1heEF0dGVtcHRzLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGxldCByZXR2YWxzID0gW107XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCBtYXhBdHRlbXB0czsgaSsrKSB7XG4gICAgICBjb25zdCBwcm9taXNlID0gYmFja29mZkdlbmVyYXRvci5iYWNrb2ZmKCk7XG4gICAgICBqZXN0LmFkdmFuY2VUaW1lcnNCeVRpbWUoYmFzZSAqIE1hdGgucG93KDIsIGkpKTtcbiAgICAgIGNvbnN0IHJldHZhbCA9IGF3YWl0IHByb21pc2U7XG4gICAgICByZXR2YWxzLnB1c2gocmV0dmFsKTtcbiAgICB9XG5cbiAgICAvLyBUSEVOXG4gICAgcmV0dmFscy5zbGljZSgwLCAtMSkuZm9yRWFjaChyZXR2YWwgPT4gZXhwZWN0KHJldHZhbCkudG9CZSh0cnVlKSk7XG4gICAgZXhwZWN0KHJldHZhbHNbcmV0dmFscy5sZW5ndGggLSAxXSkudG9CZShmYWxzZSk7XG4gICAgZXhwZWN0KGJhY2tvZmZHZW5lcmF0b3Iuc2hvdWxkQ29udGludWUoKSkudG9CZShmYWxzZSk7XG4gIH0pO1xuXG4gIHRlc3QoJ3N0b3BzIHdoZW4gbWF4IGN1bXVsYXRpdmUgYmFja29mZiB0aW1lIGlzIHJlYWNoZWQnLCBhc3luYyAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBiYWNrb2ZmR2VuZXJhdG9yID0gbmV3IEJhY2tvZmZHZW5lcmF0b3Ioe1xuICAgICAgYmFzZSxcbiAgICAgIG1heEN1bXVsYXRpdmVCYWNrb2ZmVGltZU1zOiBiYXNlLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHByb21pc2UgPSBiYWNrb2ZmR2VuZXJhdG9yLmJhY2tvZmYoKTtcbiAgICBqZXN0LmFkdmFuY2VUaW1lcnNCeVRpbWUoYmFzZSk7XG4gICAgY29uc3QgcmV0dmFsID0gYXdhaXQgcHJvbWlzZTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QocmV0dmFsKS50b0JlKGZhbHNlKTtcbiAgICBleHBlY3QoYmFja29mZkdlbmVyYXRvci5zaG91bGRDb250aW51ZSgpKS50b0JlKGZhbHNlKTtcbiAgfSk7XG5cbiAgdGVzdCgncmVzcGVjdHMgbWF4IGludGVydmFsIGJldHdlZW4gYmFja29mZnMnLCBhc3luYyAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBtYXhJbnRlcnZhbE1zID0gYmFzZSAvIDI7XG4gICAgY29uc3QgYmFja29mZkdlbmVyYXRvciA9IG5ldyBCYWNrb2ZmR2VuZXJhdG9yKHtcbiAgICAgIGJhc2UsXG4gICAgICBtYXhJbnRlcnZhbE1zLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHByb21pc2UgPSBiYWNrb2ZmR2VuZXJhdG9yLmJhY2tvZmYoKTtcbiAgICBqZXN0LmFkdmFuY2VUaW1lcnNCeVRpbWUobWF4SW50ZXJ2YWxNcyk7XG4gICAgYXdhaXQgcHJvbWlzZTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QobWF4SW50ZXJ2YWxNcykudG9CZUxlc3NUaGFuKGJhc2UpO1xuICAgIGV4cGVjdChzZXRUaW1lb3V0KS50b0hhdmVCZWVuQ2FsbGVkVGltZXMoMSk7XG4gICAgZXhwZWN0KHNldFRpbWVvdXQpLnRvSGF2ZUJlZW5MYXN0Q2FsbGVkV2l0aChleHBlY3QuYW55KEZ1bmN0aW9uKSwgbWF4SW50ZXJ2YWxNcyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2ZvcmNlcyBiYWNrb2ZmJywgYXN5bmMgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYmFja29mZkdlbmVyYXRvciA9IG5ldyBCYWNrb2ZmR2VuZXJhdG9yKHtcbiAgICAgIGJhc2UsXG4gICAgICBtYXhBdHRlbXB0czogMCxcbiAgICB9KTtcbiAgICBpZiAoYmFja29mZkdlbmVyYXRvci5zaG91bGRDb250aW51ZSgpICE9PSBmYWxzZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdCYWNrb2ZmR2VuZXJhdG9yLnNob3VsZENvbnRpbnVlIGRpZCBub3QgcmV0dXJuIGZhbHNlIHdoZW4gaXQgd2FzIGV4cGVjdGVkIHRvLiBQbGVhc2UgZml4IHRoaXMgdW5pdCB0ZXN0LicpO1xuICAgIH1cblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBwcm9taXNlID0gYmFja29mZkdlbmVyYXRvci5iYWNrb2ZmKHRydWUpO1xuICAgIGplc3QuYWR2YW5jZVRpbWVyc0J5VGltZShiYXNlKTtcbiAgICBhd2FpdCBwcm9taXNlO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzZXRUaW1lb3V0KS50b0hhdmVCZWVuQ2FsbGVkVGltZXMoMSk7XG4gICAgZXhwZWN0KHNldFRpbWVvdXQpLnRvSGF2ZUJlZW5MYXN0Q2FsbGVkV2l0aChleHBlY3QuYW55KEZ1bmN0aW9uKSwgYmFzZSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCcucmVzdGFydCgpJywgKCkgPT4ge1xuICAgIHRlc3QoJ3Jlc2V0cyB0aGUgbnVtYmVyIG9mIGF0dGVtcHRzJywgYXN5bmMgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGJhY2tvZmZHZW5lcmF0b3IgPSBuZXcgQmFja29mZkdlbmVyYXRvcih7XG4gICAgICAgIGJhc2UsXG4gICAgICAgIG1heEF0dGVtcHRzOiAxLFxuICAgICAgfSk7XG4gICAgICAvLyBUaGlzIHJlYWNoZXMgbWF4QXR0ZW1wdHMsIHNob3VsZENvbnRpbnVlKCkgd2lsbCByZXR1cm4gZmFsc2VcbiAgICAgIGNvbnN0IHByb21pc2UgPSBiYWNrb2ZmR2VuZXJhdG9yLmJhY2tvZmYoKTtcbiAgICAgIGplc3QuYWR2YW5jZVRpbWVyc0J5VGltZShiYXNlKTtcbiAgICAgIGF3YWl0IHByb21pc2U7XG4gICAgICBpZiAoYmFja29mZkdlbmVyYXRvci5zaG91bGRDb250aW51ZSgpICE9PSBmYWxzZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0JhY2tvZmZHZW5lcmF0b3Iuc2hvdWxkQ29udGludWUgZGlkIG5vdCByZXR1cm4gZmFsc2Ugd2hlbiBpdCB3YXMgZXhwZWN0ZWQgdG8uIFBsZWFzZSBmaXggdGhpcyB1bml0IHRlc3QuJyk7XG4gICAgICB9XG5cbiAgICAgIC8vIFdIRU5cbiAgICAgIGJhY2tvZmZHZW5lcmF0b3IucmVzdGFydCgpO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICBleHBlY3QoYmFja29mZkdlbmVyYXRvci5zaG91bGRDb250aW51ZSgpKS50b0JlKHRydWUpO1xuICAgIH0pO1xuXG4gICAgdGVzdCgncmVzZXRzIHRoZSBjdW11bGF0aXZlIGJhY2tvZmYgdGltZScsIGFzeW5jICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBiYWNrb2ZmR2VuZXJhdG9yID0gbmV3IEJhY2tvZmZHZW5lcmF0b3Ioe1xuICAgICAgICBiYXNlLFxuICAgICAgICBtYXhDdW11bGF0aXZlQmFja29mZlRpbWVNczogYmFzZSxcbiAgICAgIH0pO1xuICAgICAgLy8gVGhpcyByZWFjaGVzIG1heEN1bXVsYXRpdmVCYWNrb2ZmVGltZU1zLCBzaG91bGRDb250aW51ZSgpIHdpbGwgcmV0dXJuIGZhbHNlXG4gICAgICBjb25zdCBwcm9taXNlID0gYmFja29mZkdlbmVyYXRvci5iYWNrb2ZmKCk7XG4gICAgICBqZXN0LmFkdmFuY2VUaW1lcnNCeVRpbWUoYmFzZSk7XG4gICAgICBhd2FpdCBwcm9taXNlO1xuICAgICAgaWYgKGJhY2tvZmZHZW5lcmF0b3Iuc2hvdWxkQ29udGludWUoKSAhPT0gZmFsc2UpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdCYWNrb2ZmR2VuZXJhdG9yLnNob3VsZENvbnRpbnVlIGRpZCBub3QgcmV0dXJuIGZhbHNlIHdoZW4gaXQgd2FzIGV4cGVjdGVkIHRvLiBQbGVhc2UgZml4IHRoaXMgdW5pdCB0ZXN0LicpO1xuICAgICAgfVxuXG4gICAgICAvLyBXSEVOXG4gICAgICBiYWNrb2ZmR2VuZXJhdG9yLnJlc3RhcnQoKTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KGJhY2tvZmZHZW5lcmF0b3Iuc2hvdWxkQ29udGludWUoKSkudG9CZSh0cnVlKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUoJ2JhY2tzIG9mZiBhbmQgY29udGludWVzJywgKCkgPT4ge1xuICAgIHRlc3QoJ3doZW4gdGhlcmUgYXJlIHJlbWFpbmluZyBhdHRlbXB0cycsIGFzeW5jICgpID0+IHtcbiAgICAgIC8vIEdJVkVOXG4gICAgICBjb25zdCBiYWNrb2ZmR2VuZXJhdG9yID0gbmV3IEJhY2tvZmZHZW5lcmF0b3Ioe1xuICAgICAgICBiYXNlLFxuICAgICAgICBtYXhBdHRlbXB0czogMixcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBwcm9taXNlID0gYmFja29mZkdlbmVyYXRvci5iYWNrb2ZmKCk7XG4gICAgICBqZXN0LmFkdmFuY2VUaW1lcnNCeVRpbWUoYmFzZSk7XG4gICAgICBjb25zdCByZXR2YWwgPSBhd2FpdCBwcm9taXNlO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICAvLyBXZSBoYXZlIG9uZSBtb3JlIGF0dGVtcHQgbGVmdCwgaXQgc2hvdWxkIGNvbnRpbnVlXG4gICAgICBleHBlY3QocmV0dmFsKS50b0JlKHRydWUpO1xuICAgICAgZXhwZWN0KGJhY2tvZmZHZW5lcmF0b3Iuc2hvdWxkQ29udGludWUoKSkudG9CZSh0cnVlKTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ3doZW4gdGhlcmUgaXMgcmVtYWluaW5nIGN1bXVsYXRpdmUgYmFja29mZiB0aW1lJywgYXN5bmMgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGJhY2tvZmZHZW5lcmF0b3IgPSBuZXcgQmFja29mZkdlbmVyYXRvcih7XG4gICAgICAgIGJhc2UsXG4gICAgICAgIG1heEN1bXVsYXRpdmVCYWNrb2ZmVGltZU1zOiBiYXNlICsgMSxcbiAgICAgIH0pO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBwcm9taXNlID0gYmFja29mZkdlbmVyYXRvci5iYWNrb2ZmKCk7XG4gICAgICBqZXN0LmFkdmFuY2VUaW1lcnNCeVRpbWUoYmFzZSk7XG4gICAgICBjb25zdCByZXR2YWwgPSBhd2FpdCBwcm9taXNlO1xuXG4gICAgICAvLyBUSEVOXG4gICAgICAvLyBXZSBoYXZlbid0IHJlYWNoZWQgbWF4IGN1bXVsYXRpdmUgYmFja29mZiB0aW1lIHlldCwgaXQgc2hvdWxkIGNvbnRpbnVlXG4gICAgICBleHBlY3QocmV0dmFsKS50b0JlKHRydWUpO1xuICAgICAgZXhwZWN0KGJhY2tvZmZHZW5lcmF0b3Iuc2hvdWxkQ29udGludWUoKSkudG9CZSh0cnVlKTtcbiAgICB9KTtcbiAgfSk7XG5cbiAgZGVzY3JpYmUuZWFjaChbXG4gICAgMCxcbiAgICAwLjI1LFxuICAgIDAuNSxcbiAgICAwLjc1LFxuICAgIDEsXG4gIF0pKCdqaXR0ZXIgKGZhY3RvciAlZCknLCAoZmFjdG9yOiBudW1iZXIpID0+IHtcbiAgICBsZXQgcmFuZG9tU3R1Yjogc2lub24uU2lub25TdHViO1xuICAgIGJlZm9yZUFsbCgoKSA9PiB7XG4gICAgICByYW5kb21TdHViID0gc2lub24uc3R1YihNYXRoLCAncmFuZG9tJykucmV0dXJucyhmYWN0b3IpO1xuICAgIH0pO1xuXG4gICAgYWZ0ZXJBbGwoKCkgPT4ge1xuICAgICAgcmFuZG9tU3R1Yi5yZXN0b3JlKCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdhcHBsaWVzIGZ1bGwgaml0dGVyJywgYXN5bmMgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGJhY2tvZmZHZW5lcmF0b3IgPSBuZXcgQmFja29mZkdlbmVyYXRvcih7XG4gICAgICAgIGJhc2UsXG4gICAgICAgIGppdHRlckRpdmlzb3I6IDEsXG4gICAgICB9KTtcbiAgICAgIGNvbnN0IGludGVydmFsID0gYmFzZSAqIGZhY3RvcjtcblxuICAgICAgLy8gV0hFTlxuICAgICAgY29uc3QgcHJvbWlzZSA9IGJhY2tvZmZHZW5lcmF0b3IuYmFja29mZigpO1xuICAgICAgamVzdC5hZHZhbmNlVGltZXJzQnlUaW1lKGludGVydmFsKTtcbiAgICAgIGF3YWl0IHByb21pc2U7XG5cbiAgICAgIC8vIFRIRU5cbiAgICAgIGV4cGVjdChzZXRUaW1lb3V0KS50b0hhdmVCZWVuQ2FsbGVkVGltZXMoMSk7XG4gICAgICBleHBlY3Qoc2V0VGltZW91dCkudG9IYXZlQmVlbkxhc3RDYWxsZWRXaXRoKGV4cGVjdC5hbnkoRnVuY3Rpb24pLCBpbnRlcnZhbCk7XG4gICAgfSk7XG5cbiAgICB0ZXN0KCdjb3JyZWN0bHkgY2FsY3VsYXRlcyBqaXR0ZXIgd2l0aCBkaXZpc29yJywgYXN5bmMgKCkgPT4ge1xuICAgICAgLy8gR0lWRU5cbiAgICAgIGNvbnN0IGppdHRlckRpdmlzb3IgPSA0O1xuICAgICAgY29uc3QgYmFja29mZkdlbmVyYXRvciA9IG5ldyBCYWNrb2ZmR2VuZXJhdG9yKHtcbiAgICAgICAgYmFzZSxcbiAgICAgICAgaml0dGVyRGl2aXNvcixcbiAgICAgIH0pO1xuICAgICAgY29uc3QgaW50ZXJ2YWwgPSAoYmFzZSAtIGJhc2UgLyBqaXR0ZXJEaXZpc29yKSArIE1hdGguZmxvb3IoYmFzZSAvIGppdHRlckRpdmlzb3IgKiBmYWN0b3IpO1xuXG4gICAgICAvLyBXSEVOXG4gICAgICBjb25zdCBwcm9taXNlID0gYmFja29mZkdlbmVyYXRvci5iYWNrb2ZmKCk7XG4gICAgICBqZXN0LmFkdmFuY2VUaW1lcnNCeVRpbWUoaW50ZXJ2YWwpO1xuICAgICAgYXdhaXQgcHJvbWlzZTtcblxuICAgICAgLy8gVEhFTlxuICAgICAgZXhwZWN0KHNldFRpbWVvdXQpLnRvSGF2ZUJlZW5DYWxsZWRUaW1lcygxKTtcbiAgICAgIGV4cGVjdChzZXRUaW1lb3V0KS50b0hhdmVCZWVuTGFzdENhbGxlZFdpdGgoZXhwZWN0LmFueShGdW5jdGlvbiksIGludGVydmFsKTtcbiAgICB9KTtcbiAgfSk7XG59KTtcbiJdfQ==