"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('Test WorkerInstanceConfiguration for Linux', () => {
    let stack;
    let vpc;
    let instance;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
        });
        const userData = stack.resolve(instance.userData.render());
        // // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`),
                ],
            ],
        });
    });
    test('custom listener port', () => {
        const otherListenerPort = 55555;
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                listenerPort: otherListenerPort,
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${otherListenerPort} /tmp/`),
                ],
            ],
        });
    });
    test('groups, pools, region setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                groups: ['g1', 'g2'],
                pools: ['p1', 'p2'],
                region: 'r1',
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxConfigureWorkerScriptBoilerplate(`\' \'g1,g2\' \'p1,p2\' \'r1\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`),
                ],
            ],
        });
    });
    test('log setup', () => {
        // GIVEN
        const logGroupProps = {
            logGroupPrefix: '/test-prefix/',
        };
        // WHEN
        const config = new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            cloudWatchLogSettings: logGroupProps,
        });
        const logGroup = config.node.findChild('ConfigLogGroupWrapper');
        const logGroupName = stack.resolve(logGroup.logGroupName);
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxCloudWatchScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`),
                ],
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            Value: {
                'Fn::Join': [
                    '',
                    [
                        '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"WorkerLogs-{instance_id}\",\"file_path\":\"/var/log/Thinkbox/Deadline10/deadlineslave*.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"LauncherLogs-{instance_id}\",\"file_path\":\"/var/log/Thinkbox/Deadline10/deadlinelauncher*.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ],
                ],
            },
        }));
    });
});
describe('Test WorkerInstanceConfiguration for Windows', () => {
    let stack;
    let vpc;
    let instance;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/`),
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
    });
    test('groups, pools, region setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                groups: ['g1', 'g2'],
                pools: ['p1', 'p2'],
                region: 'r1',
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsConfigureWorkerScriptBoilerplate(`\' \'g1,g2\' \'p1,p2\' \'r1\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/`),
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
    });
    test('custom listner port', () => {
        const otherListenerPort = 55555;
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                listenerPort: otherListenerPort,
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${otherListenerPort} C:/temp/`),
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
    });
    test('log setup', () => {
        // GIVEN
        const logGroupProps = {
            logGroupPrefix: '/test-prefix/',
        };
        // WHEN
        const config = new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            cloudWatchLogSettings: logGroupProps,
        });
        const logGroup = config.node.findChild('ConfigLogGroupWrapper');
        const logGroupName = stack.resolve(logGroup.logGroupName);
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsCloudWatchScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/`),
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            Value: {
                'Fn::Join': [
                    '',
                    [
                        '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"UserdataExecution-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Amazon\\\\EC2-Windows\\\\Launch\\\\Log\\\\UserdataExecution.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"WorkerLogs-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Thinkbox\\\\Deadline10\\\\logs\\\\deadlineslave*.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"LauncherLogs-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Thinkbox\\\\Deadline10\\\\logs\\\\deadlinelauncher*.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ],
                ],
            },
        }));
    });
});
describe('Test WorkerInstanceConfiguration connect to RenderQueue', () => {
    let stack;
    let vpc;
    let renderQueue;
    let renderQueueSGId;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        const rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            version,
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
        });
        const rqSecGrp = renderQueue.connections.securityGroups[0];
        renderQueueSGId = stack.resolve(rqSecGrp.securityGroupId);
    });
    test('For Linux', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            renderQueue,
        });
        const instanceSG = instance.connections.securityGroups[0];
        const instanceSGId = stack.resolve(instanceSG.securityGroupId);
        // THEN
        // Open-box testing. We know that we invoked the connection method on the
        // render queue if the security group for the instance has an ingress rule to the RQ.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            ToPort: 8080,
            SourceSecurityGroupId: instanceSGId,
            GroupId: renderQueueSGId,
        }));
    });
    test('For Windows', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            renderQueue,
        });
        const instanceSG = instance.connections.securityGroups[0];
        const instanceSGId = stack.resolve(instanceSG.securityGroupId);
        // THEN
        // Open-box testing. We know that we invoked the connection method on the
        // render queue if the security group for the instance has an ingress rule to the RQ.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            ToPort: 8080,
            SourceSecurityGroupId: instanceSGId,
            GroupId: renderQueueSGId,
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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