"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/* eslint-disable dot-notation */
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_1 = require("../lib");
let app;
let stack;
let spotFleetStack;
let vpc;
let renderQueue;
let rcsImage;
const groupName = 'Group_Name';
const deadlineGroups = [
    groupName,
];
const workerMachineImage = new aws_ec2_1.GenericLinuxImage({
    'us-east-1': 'ami-any',
});
const instanceTypes = [
    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
];
const maxCapacity = 1;
describe('SpotEventPluginFleet', () => {
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'infraStack', {
            env: {
                region: 'us-east-1',
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'VersionQuery');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        spotFleetStack = new core_1.Stack(app, 'SpotFleetStack', {
            env: {
                region: 'us-east-1',
            },
        });
    });
    describe('created with default values', () => {
        test('creates a security group', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.securityGroups).toBeDefined();
            expect(fleet.securityGroups.length).toBe(1);
            assert_1.expect(spotFleetStack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
        });
        test('allows connection to the render queue', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
                SourceSecurityGroupId: spotFleetStack.resolve(fleet.connections.securityGroups[0].securityGroupId),
            }));
        });
        test('creates a spot fleet instance role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn)),
            }));
        });
        test('creates an instance profile', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.instanceProfile).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::InstanceProfile', {
                Roles: assert_1.arrayWith({
                    Ref: spotFleetStack.getLogicalId(fleet.fleetInstanceRole.node.defaultChild),
                }),
            }));
        });
        test('creates a spot fleet role', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn)),
            }));
        });
        test('adds group names to user data', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBeDefined();
            expect(renderedUserData).toMatch(groupName.toLocaleLowerCase());
        });
        test('adds RFDK tags', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            expect(fleet.tags).toBeDefined();
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                Tags: assert_1.arrayWith(assert_1.objectLike({
                    Key: rfdkTag.name,
                    Value: rfdkTag.value,
                })),
            }));
        });
        test('uses default LogGroup prefix %s', () => {
            // GIVEN
            const id = 'SpotFleet';
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: '/renderfarm/' + id,
            }));
        });
        test('sets default allocation strategy', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(lib_1.SpotFleetAllocationStrategy.LOWEST_PRICE);
        });
        test('does not set valid until property', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.validUntil).toBeUndefined();
        });
        test('does not set valid block devices', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.blockDevices).toBeUndefined();
        });
        test('does not set ssh key', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.keyName).toBeUndefined();
        });
        test('.defaultSubnets is true', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.defaultSubnets).toBeTruthy();
        });
    });
    describe('created with custom values', () => {
        test('uses provided required properties', () => {
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const imageConfig = workerMachineImage.getImage(fleet);
            // THEN
            expect(fleet.deadlineGroups).toStrictEqual(deadlineGroups.map(group => group.toLocaleLowerCase()));
            expect(fleet.instanceTypes).toBe(instanceTypes);
            expect(fleet.imageId).toBe(imageConfig.imageId);
            expect(fleet.osType).toBe(imageConfig.osType);
            expect(fleet.maxCapacity).toBe(maxCapacity);
        });
        test('uses provided security group', () => {
            // GIVEN
            const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg,
                ],
            });
            // THEN
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
            expect(fleet.securityGroups.length).toBe(1);
            expect(fleet.securityGroups).toContainEqual(sg);
        });
        test('uses multiple provided security groups', () => {
            // GIVEN
            const sg1 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG1', 'sg-123456789', {
                allowAllOutbound: false,
            });
            const sg2 = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG2', 'sg-987654321', {
                allowAllOutbound: false,
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                securityGroups: [
                    sg1,
                    sg2,
                ],
            });
            // THEN
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup'));
            expect(fleet.securityGroups.length).toBe(2);
            expect(fleet.securityGroups).toContainEqual(sg1);
            expect(fleet.securityGroups).toContainEqual(sg2);
        });
        test('adds to provided user data', () => {
            // GIVEN
            const originalCommands = 'original commands';
            const originalUserData = workerMachineImage.getImage(spotFleetStack).userData;
            originalUserData.addCommands(originalCommands);
            const renderedOriginalUser = originalUserData.render();
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userData: originalUserData,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(fleet.userData).toBe(originalUserData);
            expect(renderedUserData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(originalCommands)));
            expect(renderedUserData).not.toEqual(renderedOriginalUser);
        });
        test('uses provided spot fleet instance role from the same stack', () => {
            // GIVEN
            const spotFleetInstanceRole = new aws_iam_1.Role(spotFleetStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                fleetInstanceRole: spotFleetInstanceRole,
            });
            // THEN
            expect(fleet.fleetInstanceRole).toBe(spotFleetInstanceRole);
            assert_1.expect(spotFleetStack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'ec2.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(spotFleetStack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy').managedPolicyArn)),
            }));
        });
        test('throws if provided spot fleet instance role is not from the same stack', () => {
            // GIVEN
            const otherStack = new core_1.Stack(app, 'OtherStack', {
                env: { region: 'us-east-1' },
            });
            const spotFleetInstanceRole = new aws_iam_1.Role(otherStack, 'SpotFleetInstanceRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginWorkerPolicy'),
                ],
            });
            // WHEN
            function createSpotEventPluginFleet() {
                new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    fleetInstanceRole: spotFleetInstanceRole,
                });
            }
            // THEN
            expect(createSpotEventPluginFleet).toThrowError('Fleet instance role should be created on the same stack as SpotEventPluginFleet to avoid circular dependencies.');
        });
        test('uses provided spot fleet role', () => {
            // GIVEN
            const fleetRole = new aws_iam_1.Role(stack, 'FleetRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('spotfleet.amazonaws.com'),
                managedPolicies: [
                    aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole'),
                ],
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                fleetRole: fleetRole,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            // THEN
            expect(fleet.fleetRole).toBe(fleetRole);
            assert_1.expect(spotFleetStack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                AssumeRolePolicyDocument: assert_1.objectLike({
                    Statement: [assert_1.objectLike({
                            Action: 'sts:AssumeRole',
                            Effect: 'Allow',
                            Principal: {
                                Service: 'spotfleet.amazonaws.com',
                            },
                        })],
                }),
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2SpotFleetTaggingRole').managedPolicyArn)),
            }));
        });
        test('tags resources deployed by CDK', () => {
            // GIVEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const tagName = 'name';
            const tagValue = 'tagValue';
            // WHEN
            core_1.Tags.of(fleet).add(tagName, tagValue);
            // THEN
            assert_1.expect(spotFleetStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                Tags: assert_1.arrayWith(assert_1.objectLike({
                    Key: tagName,
                    Value: tagValue,
                })),
            }));
        });
        test('uses provided subnets', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            const expectedSubnetId = stack.resolve(vpc.privateSubnets[0].subnetId);
            // THEN
            expect(stack.resolve(fleet.subnets.subnetIds)).toContainEqual(expectedSubnetId);
        });
        test('.defaultSubnets is false when subnets provided', () => {
            // GIVEN
            const privateSubnets = {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            };
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                vpcSubnets: privateSubnets,
            });
            // THEN
            expect(fleet.defaultSubnets).toBeFalsy();
        });
        test('uses provided allocation strategy', () => {
            // GIVEN
            const allocationStartegy = lib_1.SpotFleetAllocationStrategy.CAPACITY_OPTIMIZED;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                allocationStrategy: allocationStartegy,
            });
            // THEN
            expect(fleet.allocationStrategy).toEqual(allocationStartegy);
        });
        test('adds deadline region to user data', () => {
            // GIVEN
            const deadlineRegion = 'someregion';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlineRegion: deadlineRegion,
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(deadlineRegion);
        });
        test('adds deadline pools to user data', () => {
            // GIVEN
            const pool1 = 'Pool1';
            const pool2 = 'pool2';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                deadlinePools: [
                    pool1,
                    pool2,
                ],
            });
            const renderedUserData = fleet.userData.render();
            // THEN
            expect(renderedUserData).toMatch(pool1.toLocaleLowerCase());
            expect(renderedUserData).toMatch(pool2.toLocaleLowerCase());
        });
        test('uses provided ssh key name', () => {
            // GIVEN
            const keyName = 'test-key-name';
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                keyName: keyName,
            });
            // THEN
            expect(fleet.keyName).toEqual(keyName);
        });
        test('UserData is added by UserDataProvider', () => {
            // WHEN
            class UserDataProvider extends lib_1.InstanceUserDataProvider {
                preCloudWatchAgent(host) {
                    host.userData.addCommands('echo preCloudWatchAgent');
                }
                preRenderQueueConfiguration(host) {
                    host.userData.addCommands('echo preRenderQueueConfiguration');
                }
                preWorkerConfiguration(host) {
                    host.userData.addCommands('echo preWorkerConfiguration');
                }
                postWorkerLaunch(host) {
                    host.userData.addCommands('echo postWorkerLaunch');
                }
            }
            const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                userDataProvider: new UserDataProvider(spotFleetStack, 'UserDataProvider'),
            });
            const userData = fleet.userData.render();
            // THEN
            expect(userData).toContain('echo preCloudWatchAgent');
            expect(userData).toContain('echo preRenderQueueConfiguration');
            expect(userData).toContain('echo preWorkerConfiguration');
            expect(userData).toContain('echo postWorkerLaunch');
        });
        test.each([
            'test-prefix/',
            '',
        ])('uses custom LogGroup prefix %s', (testPrefix) => {
            // GIVEN
            const id = 'SpotFleet';
            const logGroupProps = {
                logGroupPrefix: testPrefix,
            };
            // WHEN
            new lib_1.SpotEventPluginFleet(stack, id, {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
                logGroupProps: logGroupProps,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
                RetentionInDays: 3,
                LogGroupName: testPrefix + id,
            }));
        });
    });
    describe('allowing remote control', () => {
        test('from CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlFrom(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            }));
        });
        test('to CIDR', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            fleet.allowRemoteControlTo(aws_ec2_1.Peer.ipv4('127.0.0.1/24'));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                SecurityGroupEgress: [{ CidrIp: '0.0.0.0/0' }],
                SecurityGroupIngress: [
                    {
                        CidrIp: '127.0.0.1/24',
                        Description: 'Worker remote command listening port',
                        FromPort: fromPort,
                        IpProtocol: 'tcp',
                        ToPort: fromPort + maxWorkersPerHost,
                    },
                ],
            }));
        });
        test('from SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('to SecurityGroup', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('from other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new core_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlFrom(securityGroup);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
        test('to other stack', () => {
            // GIVEN
            const fromPort = 56032;
            const maxWorkersPerHost = 8;
            const otherStack = new core_1.Stack(app, 'otherStack', {
                env: { region: 'us-east-1' },
            });
            // WHEN
            const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue,
                deadlineGroups,
                instanceTypes,
                workerMachineImage,
                maxCapacity,
            });
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(otherStack, 'SG', 'sg-123456789');
            fleet.allowRemoteControlTo(securityGroup);
            // THEN
            assert_1.expect(otherStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                FromPort: fromPort,
                IpProtocol: 'tcp',
                SourceSecurityGroupId: 'sg-123456789',
                ToPort: fromPort + maxWorkersPerHost,
            }));
        });
    });
    describe('validation with', () => {
        describe('instance types', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyInstanceTypes = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: emptyInstanceTypes,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one instance type is required for a Spot Fleet Request Configuration/);
            });
            test('passes with at least one', () => {
                // GIVEN
                const oneInstanceType = [aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL)];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes: oneInstanceType,
                        workerMachineImage,
                        maxCapacity,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('subnets', () => {
            test('error if no subnets provided', () => {
                // GIVEN
                const invalidSubnets = {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    availabilityZones: ['dummy zone'],
                };
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    vpcSubnets: invalidSubnets,
                });
                // THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.ERROR);
                expect(fleet.node.metadata[0].data).toMatch(/Did not find any subnets matching/);
            });
        });
        describe('groups', () => {
            test('throws with empty', () => {
                // GIVEN
                const emptyDeadlineGroups = [];
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: emptyDeadlineGroups,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/At least one Deadline Group is required for a Spot Fleet Request Configuration/);
            });
            test.each([
                'none',
                'with space',
                'group_*',
            ])('throws with %s', (invalidGroupName) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [invalidGroupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineGroups'/);
            });
            test('passes with valid group name', () => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineGroups: [groupName],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('region', () => {
            test.each([
                'none',
                'all',
                'unrecognized',
                'none@123',
                'None',
            ])('throws with %s', (deadlineRegion) => {
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/Invalid value: .+ for property 'deadlineRegion'/);
            });
            test('passes with reserved name as substring', () => {
                // GIVEN
                const deadlineRegion = 'none123';
                // WHEN
                function createSpotEventPluginFleet() {
                    new lib_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups,
                        instanceTypes,
                        workerMachineImage,
                        maxCapacity,
                        deadlineRegion: deadlineRegion,
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).not.toThrowError();
            });
        });
        describe('Block Device Tests', () => {
            test('Warning if no BlockDevices provided', () => {
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                });
                // THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadata[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
            });
            test('No Warnings if Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: '/dev/xvda',
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                        }],
                });
                //THEN
                expect(fleet.node.metadata).toHaveLength(0);
            });
            test('Warnings if non-Encrypted BlockDevices Provided', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                        }],
                });
                //THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('Warnings for BlockDevices without encryption specified', () => {
                const VOLUME_SIZE = 50;
                const DEVICE_NAME = '/dev/xvda';
                const id = 'SpotFleet';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, id, {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                        }],
                });
                //THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the spot-fleet ${id} is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
            });
            test('No warnings for Ephemeral blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                        }],
                });
                //THEN
                expect(fleet.node.metadata).toHaveLength(0);
            });
            test('No warnings for Suppressed blockDeviceVolumes', () => {
                const DEVICE_NAME = '/dev/xvda';
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(spotFleetStack, 'SpotFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups,
                    instanceTypes,
                    workerMachineImage,
                    maxCapacity,
                    blockDevices: [{
                            deviceName: DEVICE_NAME,
                            volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                        }],
                });
                //THEN
                expect(fleet.node.metadata).toHaveLength(0);
            });
            test('throws if block devices without iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.IO1;
                // WHEN
                function createSpotEventPluginFleet() {
                    return new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                        vpc,
                        renderQueue,
                        deadlineGroups: [
                            groupName,
                        ],
                        instanceTypes: [
                            aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                        ],
                        workerMachineImage,
                        maxCapacity: 1,
                        blockDevices: [{
                                deviceName,
                                volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                    volumeType,
                                }),
                            }],
                    });
                }
                // THEN
                expect(createSpotEventPluginFleet).toThrowError(/iops property is required with volumeType: EbsDeviceVolumeType.IO1/);
            });
            test('warning if block devices with iops and wrong volume type', () => {
                // GIVEN
                const deviceName = '/dev/xvda';
                const volumeSize = 50;
                const iops = 100;
                const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
                // WHEN
                const fleet = new lib_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                    vpc,
                    renderQueue,
                    deadlineGroups: [
                        groupName,
                    ],
                    instanceTypes: [
                        aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                    ],
                    workerMachineImage,
                    maxCapacity: 1,
                    blockDevices: [{
                            deviceName,
                            volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                                iops,
                                volumeType,
                            }),
                        }],
                });
                // THEN
                expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
                expect(fleet.node.metadata[0].data).toMatch('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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