"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const asset_constants_1 = require("../../deadline/test/asset-constants");
const lib_1 = require("../lib");
const asset_constants_2 = require("./asset-constants");
let app;
let stack;
let vpc;
let version;
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: ., $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack');
    vpc = new aws_ec2_1.Vpc(stack, 'VPC', {
        subnetConfiguration: [
            {
                name: 'Public',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            },
            {
                name: 'Private',
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            {
                name: 'Isolated',
                subnetType: aws_ec2_1.SubnetType.ISOLATED,
            },
        ],
    });
    class MockVersion extends lib_1.Version {
        constructor() {
            super(...arguments);
            this.linuxInstallers = {
                patchVersion: 0,
                repository: {
                    objectKey: 'testInstaller',
                    s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxInstallerBucket'),
                },
                client: {
                    objectKey: 'testClientInstaller',
                    s3Bucket: new aws_s3_1.Bucket(stack, 'LinuxClientInstallerBucket'),
                },
            };
        }
        linuxFullVersionString() {
            return this.toString();
        }
    }
    version = new MockVersion([10, 1, 19, 4]);
});
test('can create two repositories', () => {
    // GIVEN
    new lib_1.Repository(stack, 'Repo1', {
        vpc,
        version,
    });
    // THEN
    expect(() => {
        new lib_1.Repository(stack, 'Repo2', {
            vpc,
            version,
        });
    }).not.toThrow();
});
test('repository installer instance is created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        Properties: {
            MaxSize: '1',
            MinSize: '1',
            VPCZoneIdentifier: [
                {
                    Ref: 'VPCPrivateSubnet1Subnet8BCA10E0',
                },
                {
                    Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A',
                },
            ],
        },
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT15M',
            },
        },
        UpdatePolicy: {
            AutoScalingReplacingUpdate: {
                WillReplace: true,
            },
            AutoScalingScheduledAction: {
                IgnoreUnmodifiedGroupSizeProperties: true,
            },
        },
        DependsOn: [
            'repositoryInstallerDocumentDatabaseInstance11A6F8C8E',
        ],
    }, assert_1.ResourcePart.CompleteDefinition));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't3.large',
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
test('repository installer honors vpcSubnet', () => {
    // Note: Default is private subnets, so it's sufficient to test something other than
    // private subnets.
    // WHEN
    const isolatedSubnetIds = ['IsolatedSubnet1', 'IsolatedSubnet2'];
    const attrVpc = aws_ec2_1.Vpc.fromVpcAttributes(stack, 'TestVpc', {
        availabilityZones: ['us-east-1a', 'us-east-1b'],
        vpcId: 'vpcid',
        isolatedSubnetIds,
    });
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc: attrVpc,
        version,
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: isolatedSubnetIds,
    }));
});
test('repository installer security groups created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 2049,
        ToPort: 2049,
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerFileSystemEfsSecurityGroup289D043C',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        ToPort: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseA36CE7FE',
                'Port',
            ],
        },
        GroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerDocumentDatabaseSecurityGroupBEFDC58F',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'repositoryInstallerInstanceSecurityGroup0B5705D3',
                'GroupId',
            ],
        },
    }));
});
/*
IAM Policy document tests. The policy for the installer instance is:
  expectCDK(stack).to(haveResourceLike('AWS::IAM::Policy', {
    PolicyDocument: {
      Statement: [
        {}, // CloudWatch log group put
        {}, // cloudwatch agent install script
        {}, // cloudwatch agent string parameters
        {}, // cloudwatch agent get installer permissions
        {}, // gpg get installer permissions
        {}, // DocDB secret get
        {}, // filesystem mount script get
        {}, // installer get
        {}, // repository installation script asset get
        {}, // update autoscaling policy
        {}, // describe tags
      ],
    },
  }));

  We only explicitly test for the permissions we explicitly add:
    - docDB secret get
    - installer get
    - autoscaling policy (handled by: 'alidate instance self-termination' test)
    - describe tags (handled by: 'alidate instance self-termination' test)
  The others are side-effects of adding the cloudwatch agent, and mounting the filesystem.
  We make sure that we do those things in other ways, and trust to their constructs to be
  verifying proper permissions.
 */
test('repository installer iam permissions: db secret access', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:DescribeSecret',
                ],
                Effect: 'Allow',
                Resource: {
                    Ref: 'repositoryInstallerDocumentDatabaseSecretAttachment29753B7C',
                },
            }),
        },
    }));
});
test('repository installer iam permissions: installer get', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Effect: 'Allow',
                Resource: assert_1.arrayWith({
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                        ],
                    ],
                }, {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                Ref: 'AWS::Partition',
                            },
                            ':s3:::',
                            {
                                Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket,
                            },
                            '/*',
                        ],
                    ],
                }),
            }),
        },
    }));
});
test('default repository installer log group created correctly', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/repositoryInstaller',
    }));
});
test('repository installer logs all required files', () => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // THEN
    // The CloudWatchAgent stores the agent configuration in an SSM Parameter. Check it for the required setup.
    // Note: This would be better implemented using the tools in: https://github.com/aws/aws-cdk/pull/8444
    // but that is not yet available.
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
        Type: 'String',
        Value: {
            'Fn::Join': [
                '',
                [
                    '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                    {},
                    '\",\"log_stream_name\":\"deadlineRepositoryInstallationLogs-{instance_id}\",\"file_path\":\"/tmp/bitrock_installer.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                ],
            ],
        },
    }));
});
test('repository mounts repository filesystem', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
    });
    // WHEN
    const userData = repo.node.defaultChild.userData.render();
    // THEN
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/efs/fs1 false rw')));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors database removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            database: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test.each([
    [core_1.RemovalPolicy.DESTROY, 'Delete'],
    [core_1.RemovalPolicy.RETAIN, 'Retain'],
    [core_1.RemovalPolicy.SNAPSHOT, 'Snapshot'],
])('repository honors filesystem removal policy: %p', (policy, expected) => {
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: policy,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EFS::FileSystem', {
        DeletionPolicy: expected,
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository warns if removal policy for filesystem when filesystem provided', () => {
    // GIVEN
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct',
        }),
    ]));
});
test('repository warns if removal policy for database when database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        removalPolicy: {
            database: core_1.RemovalPolicy.DESTROY,
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'RemovalPolicy for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates deadlineDatabase if none provided', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBInstance'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBInstance', {
        AutoMinorVersionUpgrade: true,
    }));
});
test('disabling Audit logging does not enable Cloudwatch audit logs', () => {
    const testEFS = new aws_efs_1.FileSystem(stack, 'TestEfsFileSystem', {
        vpc,
    });
    const testAP = new aws_efs_1.AccessPoint(stack, 'TestAccessPoint', {
        fileSystem: testEFS,
    });
    const testFS = new core_2.MountableEfs(stack, {
        filesystem: testEFS,
        accessPoint: testAP,
    });
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        fileSystem: testFS,
        version,
        databaseAuditLogging: false,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DocDB::DBCluster'));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        EnableCloudwatchLogsExports: ['audit'],
    }, assert_1.ResourcePart.Properties));
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::DocDB::DBClusterParameterGroup', {
        Parameters: {
            audit_logs: 'enabled',
        },
    }, assert_1.ResourcePart.Properties));
});
test('repository warns if databaseAuditLogging defined and database is specified', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        removalPolicy: {
            filesystem: core_1.RemovalPolicy.DESTROY,
        },
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        databaseAuditLogging: true,
    });
    const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
        'Please ensure that the Database provided is configured correctly.';
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: warningMsg,
        }),
    ]));
});
test('honors subnet specification', () => {
    // GIVEN
    const dependencyStack = new core_1.Stack(app, 'DepStack');
    const dependencyVpc = new aws_ec2_1.Vpc(dependencyStack, 'DepVpc');
    const subnets = [
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
            subnetId: 'SubnetID1',
            availabilityZone: 'us-west-2a',
        }),
        aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
            subnetId: 'SubnetID2',
            availabilityZone: 'us-west-2b',
        }),
    ];
    const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
    // WHEN
    new lib_1.Repository(isolatedStack, 'repositoryInstaller', {
        vpc: dependencyVpc,
        version,
        vpcSubnets: {
            subnets,
        },
    });
    // THEN
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::DocDB::DBSubnetGroup', {
        SubnetIds: [
            'SubnetID1',
            'SubnetID2',
        ],
    }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID1' }));
    assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EFS::MountTarget', { SubnetId: 'SubnetID2' }));
});
test('repository honors database instance count', () => {
    // GIVEN
    const instanceCount = 2;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        documentDbInstanceCount: instanceCount,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::DocDB::DBInstance', instanceCount, {
        AutoMinorVersionUpgrade: true,
    }));
});
test('repository honors database retention period', () => {
    // GIVEN
    const period = 20;
    // WHEN
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(period),
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
        BackupRetentionPeriod: period,
    }));
});
test('warns if both retention period and database provided', () => {
    // GIVEN
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        },
    });
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
        backupOptions: {
            databaseRetention: core_1.Duration.days(20),
        },
    });
    // THEN
    expect(repo.node.metadata).toEqual(expect.arrayContaining([
        expect.objectContaining({
            type: 'aws:cdk:warning',
            data: 'Backup retention for database will not be applied since a database is not being created by this construct',
        }),
    ]));
});
test('repository creates filesystem if none provided', () => {
    const fsDatabase = new aws_docdb_1.DatabaseCluster(stack, 'TestDbCluster', {
        masterUser: {
            username: 'master',
        },
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
        vpc,
        vpcSubnets: {
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        },
        backup: {
            retention: core_1.Duration.days(15),
        },
    });
    if (!fsDatabase.secret) {
        throw new Error('secret cannot be null');
    }
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        database: lib_1.DatabaseConnection.forDocDB({ database: fsDatabase, login: fsDatabase.secret }),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::FileSystem'));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::MountTarget'));
    expect(repo.node.tryFindChild('PadEfsStorage')).toBeDefined();
    expect(repo.node.findChild('FileSystem').node.tryFindChild('PaddingAccessPoint')).toBeDefined();
});
test('default repository instance is created using user defined installation path prefix', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: 'xyz',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('default repository instance is created using user defined installation path prefix with extra slashes in path', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        version,
        repositoryInstallationPrefix: '/xyz//',
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toMatch(/\/mnt\/efs\/fs1\/xyz/);
});
test('repository instance is created with user defined timeout', () => {
    new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
        CreationPolicy: {
            ResourceSignal: {
                Count: 1,
                Timeout: 'PT30M',
            },
        },
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('repository instance is created with correct installer path version', () => {
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    // THEN
    const script = repo.node.defaultChild.userData;
    expect(script.render()).toEqual(expect.stringContaining(version.versionString));
});
test.each([
    'test-prefix/',
    '',
])('repository instance is created with correct LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'repositoryInstaller';
    // WHEN
    new lib_1.Repository(stack, id, {
        vpc,
        version,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        LogGroupName: testPrefix + id,
    }));
});
test('validate instance self-termination', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const asgLogicalId = stack.getLogicalId(repo.node.defaultChild.node.defaultChild);
    // THEN
    const regionToken = escapeTokenRegex('${Token[AWS.Region.\\d+]}');
    const expectedString = `function exitTrap\\(\\)\\{\nexitCode=\\$\\?\nsleep 1m\nTOKEN=\\$\\(curl -X PUT "http:\\/\\/169\\.254\\.169\\.254\\/latest\\/api\\/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> \\/dev\\/null\\)\nINSTANCE="\\$\\(curl -s -H "X-aws-ec2-metadata-token: \\$TOKEN" http:\\/\\/169\\.254\\.169\\.254\\/latest\\/meta-data\\/instance-id  2> \\/dev\\/null\\)"\nASG="\\$\\(aws --region ${regionToken} ec2 describe-tags --filters "Name=resource-id,Values=\\$\\{INSTANCE\\}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags\\[0\\]\\.Value" --output text\\)"\naws --region ${regionToken} autoscaling update-auto-scaling-group --auto-scaling-group-name \\$\\{ASG\\} --min-size 0 --max-size 0 --desired-capacity 0\n\\/opt\\/aws\\/bin\\/cfn-signal --stack ${stack.stackName} --resource ${asgLogicalId} --region ${regionToken} -e \\$exitCode \\|\\| echo 'Failed to send Cloudformation Signal'\n\\}`;
    expect(repo.node.defaultChild.userData.render()).toMatch(new RegExp(expectedString));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: 'autoscaling:UpdateAutoScalingGroup',
                Condition: {
                    StringEquals: {
                        'autoscaling:ResourceTag/resourceLogicalId': core_1.Names.uniqueId(repo),
                    },
                },
                Effect: 'Allow',
                Resource: '*',
            }, {
                Action: 'ec2:DescribeTags',
                Effect: 'Allow',
                Resource: '*',
            }),
        },
    }));
});
test('repository configure client instance', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instanceRole = (instance
        .node.findChild('InstanceRole')
        .node.defaultChild);
    const db = (repo
        .node.findChild('DocumentDatabase'));
    // WHEN
    repo.configureClientInstance({
        host: instance,
        mountPoint: '/mnt/repository',
    });
    const userData = instance.userData.render();
    // THEN
    // white-box testing. If we mount the filesystem, then we've called: setupDirectConnect()
    expect(userData).toMatch(new RegExp(escapeTokenRegex('mountEfs.sh ${Token[TOKEN.\\d+]} /mnt/repository false rw')));
    // Make sure we added the DB connection args
    expect(userData).toMatch(/.*export -f configure_deadline_database.*/);
    // Make sure we call the configureRepositoryDirectConnect script with appropriate argument.
    const regex = new RegExp(escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \\"/mnt/repository/DeadlineRepository\\"'));
    expect(userData).toMatch(regex);
    // Assert the IAM instance profile is given read access to the database credentials secret
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Action: [
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:DescribeSecret',
                ],
                Effect: 'Allow',
                Resource: stack.resolve(db.secret.secretArn),
            }),
        },
        Roles: [
            stack.resolve(instanceRole.ref),
        ],
    }));
});
test('configureClientInstance uses singleton for repo config script', () => {
    // Note: If this test fails, then check the asset hash for REPO_DC_ASSET -- it may have changed.
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance1 = new aws_ec2_1.Instance(stack, 'Instance1', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    const instance2 = new aws_ec2_1.Instance(stack, 'Instance2', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
    });
    // WHEN
    repo.configureClientInstance({
        host: instance1,
        mountPoint: '/mnt/repository',
    });
    repo.configureClientInstance({
        host: instance2,
        mountPoint: '/mnt/repository',
    });
    // THEN
    // Make sure that both instances have access to the same Asset for the configureRepositoryDirectConnect script
    assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Policy', 2, {
        PolicyDocument: {
            Statement: assert_1.arrayWith({
                Effect: 'Allow',
                Action: [
                    's3:GetObject*',
                    's3:GetBucket*',
                    's3:List*',
                ],
                Resource: [
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':s3:::',
                                {
                                    Ref: asset_constants_2.REPO_DC_ASSET.Bucket,
                                },
                            ],
                        ],
                    },
                    {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':s3:::',
                                {
                                    Ref: asset_constants_2.REPO_DC_ASSET.Bucket,
                                },
                                '/*',
                            ],
                        ],
                    },
                ],
            }),
        },
    }));
});
test('windows client cannot direct connect to repository', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'repositoryInstaller', {
        vpc,
        repositoryInstallationTimeout: core_1.Duration.minutes(30),
        version,
    });
    const instance = new aws_ec2_1.Instance(stack, 'Instance', {
        vpc,
        instanceType: new aws_ec2_1.InstanceType('t3.small'),
        machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
    });
    // THEN
    expect(() => {
        repo.configureClientInstance({
            host: instance,
            mountPoint: 'd:\\',
        });
    }).toThrowError('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'Repository',
        createConstruct: () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.Repository(isolatedStack, 'Repository', {
                vpc,
                version,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 3,
            'AWS::DocDB::DBClusterParameterGroup': 1,
            'AWS::DocDB::DBSubnetGroup': 1,
            'AWS::SecretsManager::Secret': 2,
            'AWS::DocDB::DBCluster': 1,
            'AWS::DocDB::DBInstance': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
describe('Security Groups', () => {
    let repositorySecurityGroup;
    beforeEach(() => {
        repositorySecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'AdditionalSecurityGroup', { vpc });
    });
    describe('DocDB', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    database: repositorySecurityGroup,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::DocDB::DBCluster', {
                VpcSecurityGroupIds: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
    describe('EFS', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    fileSystem: repositorySecurityGroup,
                },
            });
            // THEN
            // The EFS construct adds the security group to each mount target, and one mount target is generated per subnet.
            const numMountTargets = vpc.selectSubnets().subnets.length;
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::EFS::MountTarget', numMountTargets, {
                SecurityGroups: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
    describe('Installer', () => {
        test('adds security groups on construction', () => {
            // WHEN
            new lib_1.Repository(stack, 'Repository', {
                version,
                vpc,
                securityGroupsOptions: {
                    installer: repositorySecurityGroup,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                SecurityGroups: assert_1.arrayWith(stack.resolve(repositorySecurityGroup.securityGroupId)),
            }));
        });
    });
});
test('validates VersionQuery is not in a different stack', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    version = new lib_1.VersionQuery(stack, 'Version');
    new lib_1.Repository(newStack, 'Repository', {
        vpc,
        version,
    });
    // WHEN
    function synth() {
        assert_1.SynthUtils.synthesize(newStack);
    }
    // THEN
    expect(synth).toThrow('A VersionQuery can not be supplied from a different stack');
});
test('creates an EFS AccessPoint when no filesystem is supplied', () => {
    // WHEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
    });
    // THEN
    const efsResource = repo.node.findChild('FileSystem').node.defaultChild;
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EFS::AccessPoint', {
        FileSystemId: stack.resolve(efsResource.ref),
        PosixUser: {
            Gid: '0',
            Uid: '0',
        },
        RootDirectory: {},
    }));
});
test('throws an error if supplied a MountableEfs with no Access Point', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    const fs = new aws_efs_1.FileSystem(newStack, 'FileSystem', {
        vpc,
    });
    const mountableFs = new core_2.MountableEfs(newStack, {
        filesystem: fs,
    });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version,
            vpc,
            fileSystem: mountableFs,
        });
    }
    // THEN
    expect(when).toThrow('When using EFS with the Repository, you must provide an EFS Access Point');
});
test('disable Secrets Management by default when Deadline version is old', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    const oldVersion = new lib_1.VersionQuery(newStack, 'OldDeadlineVersion', { version: '10.0.0.0' });
    // WHEN
    const repository = new lib_1.Repository(newStack, 'Repo', {
        vpc,
        version: oldVersion,
    });
    // THEN
    expect(repository.secretsManagementSettings.enabled).toBeFalsy();
    expect(repository.secretsManagementSettings.credentials).toBeUndefined();
});
test('throws when Secrets Management is enabled but deadline version is too low', () => {
    // GIVEN
    const newStack = new core_1.Stack(app, 'NewStack');
    const oldVersion = new lib_1.VersionQuery(newStack, 'OldDeadlineVersion', { version: '10.0.0.0' });
    // WHEN
    function when() {
        new lib_1.Repository(newStack, 'Repo', {
            version: oldVersion,
            vpc,
            secretsManagementSettings: {
                enabled: true,
            },
        });
    }
    // THEN
    expect(when).toThrow(`The supplied Deadline version (${oldVersion.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${lib_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
});
test('imports repository settings', () => {
    // GIVEN
    const repositorySettings = new aws_s3_assets_1.Asset(stack, 'RepositorySettingsAsset', {
        path: __filename,
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        repositorySettings,
    });
    // THEN
    const installerGroup = repository.node.tryFindChild('Installer');
    expect(installerGroup.userData.render()).toContain(`aws s3 cp '${repositorySettings.s3ObjectUrl}'`);
});
test('IMountableLinuxFilesystem.usesUserPosixPermissions() = true changes ownership of repository files', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
        fileSystem: {
            mountToLinuxInstance: (_target, _mount) => { },
            usesUserPosixPermissions: () => true,
        },
    });
    // WHEN
    const script = repo.node.defaultChild.userData.render();
    // THEN
    expect(script).toMatch('-o 1000:1000');
});
test('IMountableLinuxFilesystem.usesUserPosixPermissions() = false does not change ownership of repository files', () => {
    // GIVEN
    const repo = new lib_1.Repository(stack, 'Repository', {
        version,
        vpc,
        fileSystem: {
            mountToLinuxInstance: (_target, _mount) => { },
            usesUserPosixPermissions: () => false,
        },
    });
    // WHEN
    const script = repo.node.defaultChild.userData.render();
    // THEN
    expect(script).not.toMatch('-o 1000:1000');
});
test('secret manager enabled', () => {
    // GIVEN
    const expectedCredentials = new aws_secretsmanager_1.Secret(stack, 'CustomSMAdminUser', {
        description: 'Custom admin credentials for the Secret Management',
        generateSecretString: {
            excludeCharacters: '\"$&\'()-/<>[\\]\`{|}',
            includeSpace: false,
            passwordLength: 24,
            requireEachIncludedType: true,
            generateStringKey: 'password',
            secretStringTemplate: JSON.stringify({ username: 'admin' }),
        },
    });
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        secretsManagementSettings: {
            enabled: true,
            credentials: expectedCredentials,
        },
    });
    // THEN
    expect(repository.secretsManagementSettings.credentials).toBe(expectedCredentials);
    const installerGroup = repository.node.tryFindChild('Installer');
    expect(installerGroup.userData.render()).toContain(`-r ${stack.region} -c ${expectedCredentials.secretArn}`);
});
test('secret manager is enabled by default', () => {
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
    });
    // THEN
    expect(repository.secretsManagementSettings.enabled).toBeTruthy();
    expect(repository.secretsManagementSettings.credentials).toBeDefined();
});
test('credentials are undefined when secrets management is disabled', () => {
    // WHEN
    const repository = new lib_1.Repository(stack, 'Repository', {
        vpc,
        version,
        secretsManagementSettings: {
            enabled: false,
        },
    });
    // THEN
    expect(repository.secretsManagementSettings.credentials).toBeUndefined();
});
test('throws an error if credentials are undefined and database is imported', () => {
    // GIVEN
    const sg = new aws_ec2_1.SecurityGroup(stack, 'SG', {
        vpc,
    });
    const secret = new aws_secretsmanager_1.Secret(stack, 'Secret');
    const database = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'DbCluster', {
        clusterEndpointAddress: '1.2.3.4',
        clusterIdentifier: 'foo',
        instanceEndpointAddresses: ['1.2.3.5'],
        instanceIdentifiers: ['i0'],
        port: 27001,
        readerEndpointAddress: '1.2.3.6',
        securityGroup: sg,
    });
    const databaseConnection = lib_1.DatabaseConnection.forDocDB({ database, login: secret });
    // WHEN
    function when() {
        new lib_1.Repository(stack, 'Repository', {
            vpc,
            version,
            database: databaseConnection,
        });
    }
    // THEN
    expect(when).toThrow('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
});
//# sourceMappingURL=data:application/json;base64,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