"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const lib_1 = require("../lib");
const spot_event_plugin_fleet_1 = require("../lib/spot-event-plugin-fleet");
describe('ConfigureSpotEventPlugin', () => {
    let stack;
    let vpc;
    let region;
    let renderQueue;
    let version;
    let app;
    let fleet;
    let groupName;
    const workerMachineImage = new aws_ec2_1.GenericWindowsImage({
        'us-east-1': 'ami-any',
    });
    beforeEach(() => {
        region = 'us-east-1';
        app = new core_1.App();
        stack = new core_1.Stack(app, 'stack', {
            env: {
                region,
            },
        });
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        groupName = 'group_name1';
        fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
            vpc,
            renderQueue: renderQueue,
            deadlineGroups: [
                groupName,
            ],
            instanceTypes: [
                aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
            ],
            workerMachineImage,
            maxCapacity: 1,
        });
    });
    describe('creates a custom resource', () => {
        test('with default spot event plugin properties', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotPluginConfigurations: assert_1.objectLike({
                    AWSInstanceStatus: 'Disabled',
                    DeleteInterruptedSlaves: false,
                    DeleteTerminatedSlaves: false,
                    IdleShutdown: 10,
                    Logging: 'Standard',
                    PreJobTaskMode: 'Conservative',
                    Region: core_1.Stack.of(renderQueue).region,
                    ResourceTracker: true,
                    StaggerInstances: 50,
                    State: 'Global Enabled',
                    StrictHardCap: false,
                }),
            })));
        });
        test('with custom spot event plugin properties', () => {
            // GIVEN
            const configuration = {
                awsInstanceStatus: lib_1.SpotEventPluginDisplayInstanceStatus.EXTRA_INFO_0,
                deleteEC2SpotInterruptedWorkers: true,
                deleteSEPTerminatedWorkers: true,
                idleShutdown: core_1.Duration.minutes(20),
                loggingLevel: lib_1.SpotEventPluginLoggingLevel.VERBOSE,
                preJobTaskMode: lib_1.SpotEventPluginPreJobTaskMode.NORMAL,
                region: 'us-west-2',
                enableResourceTracker: false,
                maximumInstancesStartedPerCycle: 10,
                state: lib_1.SpotEventPluginState.DISABLED,
                strictHardCap: true,
            };
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotPluginConfigurations: assert_1.objectLike({
                    AWSInstanceStatus: 'ExtraInfo0',
                    DeleteInterruptedSlaves: true,
                    DeleteTerminatedSlaves: true,
                    IdleShutdown: 20,
                    Logging: 'Verbose',
                    PreJobTaskMode: 'Normal',
                    Region: 'us-west-2',
                    ResourceTracker: false,
                    StaggerInstances: 10,
                    State: 'Disabled',
                    StrictHardCap: true,
                }),
            })));
        });
        test('without spot fleets', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', {
                spotFleetRequestConfigurations: assert_1.ABSENT,
            }));
        });
        test('provides RQ connection parameters to custom resource', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                connection: assert_1.objectLike({
                    hostname: stack.resolve(renderQueue.endpoint.hostname),
                    port: stack.resolve(renderQueue.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueue.endpoint.applicationProtocol.toString()),
                }),
            })));
        });
        test('with default spot fleet request configuration', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            const rfdkTag = runtime_info_1.tagFields(fleet);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        AllocationStrategy: fleet.allocationStrategy.toString(),
                        IamFleetRole: stack.resolve(fleet.fleetRole.roleArn),
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            IamInstanceProfile: {
                                Arn: stack.resolve(fleet.instanceProfile.attrArn),
                            },
                            ImageId: fleet.imageId,
                            SecurityGroups: assert_1.arrayWith(assert_1.objectLike({
                                GroupId: stack.resolve(fleet.securityGroups[0].securityGroupId),
                            })),
                            SubnetId: stack.resolve(core_1.Fn.join('', [vpc.privateSubnets[0].subnetId, ',', vpc.privateSubnets[1].subnetId])),
                            TagSpecifications: assert_1.arrayWith(assert_1.objectLike({
                                ResourceType: 'instance',
                                Tags: assert_1.arrayWith(assert_1.objectLike({
                                    Key: rfdkTag.name,
                                    Value: rfdkTag.value,
                                })),
                            })),
                            UserData: stack.resolve(core_1.Fn.base64(fleet.userData.render())),
                            InstanceType: fleet.instanceTypes[0].toString(),
                        })),
                        ReplaceUnhealthyInstances: true,
                        TargetCapacity: fleet.maxCapacity,
                        TerminateInstancesWithExpiration: true,
                        Type: 'maintain',
                        TagSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            ResourceType: 'spot-fleet-request',
                            Tags: assert_1.arrayWith(assert_1.objectLike({
                                Key: rfdkTag.name,
                                Value: rfdkTag.value,
                            })),
                        })),
                    }),
                }),
            })));
        });
        test('adds policies to the render queue', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn), stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: 'arn:aws:ec2:*:*:spot-fleet-request/*',
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('adds resource tracker policy even if rt disabled', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
                configuration: {
                    enableResourceTracker: false,
                },
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
        });
        test.each([
            undefined,
            [],
        ])('without spot fleet', (noFleets) => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: noFleets,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.ABSENT,
            })));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Role', {
                ManagedPolicyArns: assert_1.arrayWith(stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy').managedPolicyArn), stack.resolve(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy').managedPolicyArn)),
            }));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'iam:PassRole',
                            Condition: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                            Effect: 'Allow',
                            Resource: [
                                stack.resolve(fleet.fleetRole.roleArn),
                                stack.resolve(fleet.fleetInstanceRole.roleArn),
                            ],
                        },
                        {
                            Action: 'ec2:CreateTags',
                            Effect: 'Allow',
                            Resource: 'arn:aws:ec2:*:*:spot-fleet-request/*',
                        },
                    ],
                },
                Roles: [{
                        Ref: 'RQRCSTaskTaskRole00DC9B43',
                    }],
            }));
        });
        test('fleet with validUntil', () => {
            // GIVEN
            const validUntil = core_1.Expiration.atDate(new Date(2022, 11, 17));
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                validUntil,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        ValidUntil: validUntil.date.toISOString(),
                    }),
                }),
            })));
        });
        test('fleet with block devices', () => {
            // GIVEN
            const deviceName = '/dev/xvda';
            const volumeSize = 50;
            const encrypted = true;
            const deleteOnTermination = true;
            const iops = 100;
            const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                blockDevices: [{
                        deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.ebs(volumeSize, {
                            encrypted,
                            deleteOnTermination,
                            iops,
                            volumeType,
                        }),
                    }],
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                                DeviceName: deviceName,
                                Ebs: assert_1.objectLike({
                                    DeleteOnTermination: deleteOnTermination,
                                    Iops: iops,
                                    VolumeSize: volumeSize,
                                    VolumeType: volumeType,
                                    Encrypted: encrypted,
                                }),
                            })),
                        })),
                    }),
                }),
            })));
        });
        test('fleet with block devices with custom volume', () => {
            // GIVEN
            const deviceName = '/dev/xvda';
            const virtualName = 'name';
            const snapshotId = 'snapshotId';
            const volumeSize = 50;
            const deleteOnTermination = true;
            const iops = 100;
            const volumeType = aws_autoscaling_1.EbsDeviceVolumeType.STANDARD;
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                blockDevices: [{
                        deviceName: deviceName,
                        volume: {
                            ebsDevice: {
                                deleteOnTermination,
                                iops,
                                volumeSize,
                                volumeType,
                                snapshotId,
                            },
                            virtualName,
                        },
                    }],
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                                DeviceName: deviceName,
                                Ebs: assert_1.objectLike({
                                    SnapshotId: snapshotId,
                                    DeleteOnTermination: deleteOnTermination,
                                    Iops: iops,
                                    VolumeSize: volumeSize,
                                    VolumeType: volumeType,
                                    Encrypted: assert_1.ABSENT,
                                }),
                                VirtualName: virtualName,
                            })),
                        })),
                    }),
                }),
            })));
        });
        test('fleet with block devices with no device', () => {
            // GIVEN
            const deviceName = '/dev/xvda';
            const volume = aws_autoscaling_1.BlockDeviceVolume.noDevice();
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                blockDevices: [{
                        deviceName: deviceName,
                        volume,
                    }],
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                                DeviceName: deviceName,
                                NoDevice: '',
                            })),
                        })),
                    }),
                }),
            })));
        });
        test('fleet with deprecated mappingEnabled', () => {
            // GIVEN
            const deviceName = '/dev/xvda';
            const mappingEnabled = false;
            const fleetWithCustomProps = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotEventPluginFleet', {
                vpc,
                renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
                ],
                workerMachineImage,
                maxCapacity: 1,
                blockDevices: [{
                        deviceName: deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.ebs(50),
                        mappingEnabled,
                    }],
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleetWithCustomProps,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                spotFleetRequestConfigurations: assert_1.objectLike({
                    [groupName]: assert_1.objectLike({
                        LaunchSpecifications: assert_1.arrayWith(assert_1.objectLike({
                            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                                DeviceName: deviceName,
                                NoDevice: '',
                            })),
                        })),
                    }),
                }),
            })));
        });
    });
    test('only one object allowed per render queue', () => {
        // GIVEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/Only one ConfigureSpotEventPlugin construct is allowed per render queue./);
    });
    test('can create multiple objects with different render queues', () => {
        // GIVEN
        const renderQueue2 = new lib_1.RenderQueue(stack, 'RQ2', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(stack, 'Repository2', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
            vpc,
            renderQueue: renderQueue2,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('Custom::RFDK_ConfigureSpotEventPlugin', 2));
    });
    test('throws with not supported render queue', () => {
        // GIVEN
        const invalidRenderQueue = {};
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin2', {
                vpc,
                renderQueue: invalidRenderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(/The provided render queue is not an instance of RenderQueue class. Some functionality is not supported./);
    });
    test('tagSpecifications returns undefined if fleet does not have tags', () => {
        // GIVEN
        const mockFleet = {
            tags: {
                hasTags: jest.fn().mockReturnValue(false),
            },
        };
        const mockedFleet = mockFleet;
        const config = new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // WHEN
        // eslint-disable-next-line dot-notation
        const result = stack.resolve(config['tagSpecifications'](mockedFleet, lib_1.SpotFleetResourceType.INSTANCE));
        // THEN
        expect(result).toBeUndefined();
    });
    describe('with TLS', () => {
        let renderQueueWithTls;
        let caCert;
        beforeEach(() => {
            const host = 'renderqueue';
            const zoneName = 'deadline-test.internal';
            caCert = new core_2.X509CertificatePem(stack, 'RootCA', {
                subject: {
                    cn: 'SampleRootCA',
                },
            });
            renderQueueWithTls = new lib_1.RenderQueue(stack, 'RQ with TLS', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository2', {
                    vpc,
                    version,
                }),
                version,
                hostname: {
                    zone: new aws_route53_1.PrivateHostedZone(stack, 'DnsZone', {
                        vpc,
                        zoneName: zoneName,
                    }),
                    hostname: host,
                },
                trafficEncryption: {
                    externalTLS: {
                        rfdkCertificate: new core_2.X509CertificatePem(stack, 'RQCert', {
                            subject: {
                                cn: `${host}.${zoneName}`,
                            },
                            signingCertificate: caCert,
                        }),
                    },
                },
            });
        });
        test('Lambda role can get the ca secret', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(renderQueueWithTls.certChain.secretArn),
                        },
                    ],
                },
                Roles: [
                    {
                        Ref: 'ConfigureSpotEventPluginConfiguratorServiceRole341B4735',
                    },
                ],
            }));
        });
        test('creates a custom resource with connection', () => {
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueueWithTls,
                spotFleets: [
                    fleet,
                ],
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_ConfigureSpotEventPlugin', assert_1.objectLike({
                connection: assert_1.objectLike({
                    hostname: stack.resolve(renderQueueWithTls.endpoint.hostname),
                    port: stack.resolve(renderQueueWithTls.endpoint.portAsString()),
                    protocol: stack.resolve(renderQueueWithTls.endpoint.applicationProtocol.toString()),
                    caCertificateArn: stack.resolve(renderQueueWithTls.certChain.secretArn),
                }),
            })));
        });
    });
    test('throws with the same group name', () => {
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).toThrowError(`Bad Group Name: ${groupName}. Group names in Spot Fleet Request Configurations should be unique.`);
    });
    test('uses selected subnets', () => {
        // WHEN
        new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            renderQueue: renderQueue,
            spotFleets: [
                fleet,
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'configure-spot-event-plugin.configureSEP',
            VpcConfig: {
                SubnetIds: [
                    stack.resolve(vpc.privateSubnets[0].subnetId),
                ],
            },
        }));
    });
    describe('throws with wrong deadline version', () => {
        test.each([
            ['10.1.9'],
            ['10.1.10'],
        ])('%s', (versionString) => {
            // GIVEN
            const newStack = new core_1.Stack(app, 'NewStack');
            version = new lib_1.VersionQuery(newStack, 'OldVersion', {
                version: versionString,
            });
            renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(newStack, 'Repository', {
                    vpc,
                    version,
                    secretsManagementSettings: { enabled: false },
                }),
                trafficEncryption: { externalTLS: { enabled: false } },
                version,
            });
            // WHEN
            function createConfigureSpotEventPlugin() {
                new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                    vpc,
                    renderQueue: renderQueue,
                    spotFleets: [
                        fleet,
                    ],
                });
            }
            // THEN
            expect(createConfigureSpotEventPlugin).toThrowError(`Minimum supported Deadline version for ConfigureSpotEventPlugin is 10.1.12.0. Received: ${versionString}.`);
        });
    });
    test('does not throw with min deadline version', () => {
        // GIVEN
        const versionString = '10.1.12';
        const newStack = new core_1.Stack(app, 'NewStack');
        version = new lib_1.VersionQuery(newStack, 'OldVersion', {
            version: versionString,
        });
        renderQueue = new lib_1.RenderQueue(newStack, 'OldRenderQueue', {
            vpc,
            images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
            repository: new lib_1.Repository(newStack, 'Repository', {
                vpc,
                version,
                secretsManagementSettings: { enabled: false },
            }),
            trafficEncryption: { externalTLS: { enabled: false } },
            version,
        });
        // WHEN
        function createConfigureSpotEventPlugin() {
            new lib_1.ConfigureSpotEventPlugin(newStack, 'ConfigureSpotEventPlugin', {
                vpc,
                renderQueue: renderQueue,
                spotFleets: [
                    fleet,
                ],
            });
        }
        // THEN
        expect(createConfigureSpotEventPlugin).not.toThrow();
    });
    describe('secrets management enabled', () => {
        beforeEach(() => {
            region = 'us-east-1';
            app = new core_1.App();
            stack = new core_1.Stack(app, 'stack', {
                env: {
                    region,
                },
            });
            vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
            version = new lib_1.VersionQuery(stack, 'Version');
            renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
                vpc,
                images: { remoteConnectionServer: aws_ecs_1.ContainerImage.fromAsset(__dirname) },
                repository: new lib_1.Repository(stack, 'Repository', {
                    vpc,
                    version,
                }),
                version,
            });
            groupName = 'group_name1';
        });
        test('a fleet without vpcSubnets specified => warns about dedicated subnets', () => {
            // GIVEN
            fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleet', {
                vpc,
                renderQueue: renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
                ],
                workerMachineImage,
                maxCapacity: 1,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                renderQueue,
                vpc,
                spotFleets: [fleet],
            });
            // THEN
            expect(fleet.node.metadataEntry).toContainEqual(expect.objectContaining({
                type: 'aws:cdk:warning',
                data: 'Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components',
            }));
        });
        test('a fleet with vpcSubnets specified => does not warn about dedicated subnets', () => {
            // GIVEN
            fleet = new spot_event_plugin_fleet_1.SpotEventPluginFleet(stack, 'SpotFleetWithSubnets', {
                vpc,
                vpcSubnets: {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                },
                renderQueue: renderQueue,
                deadlineGroups: [
                    groupName,
                ],
                instanceTypes: [
                    aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.SMALL),
                ],
                workerMachineImage,
                maxCapacity: 1,
            });
            // WHEN
            new lib_1.ConfigureSpotEventPlugin(stack, 'ConfigureSpotEventPlugin', {
                renderQueue,
                vpc,
                spotFleets: [fleet],
            });
            // THEN
            expect(fleet.node.metadataEntry).not.toContainEqual(expect.objectContaining({
                type: 'aws:cdk:warning',
                data: expect.stringMatching(/dedicated subnet/i),
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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