/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { AutoScalingGroup, BlockDevice } from '@aws-cdk/aws-autoscaling';
import { IMetric } from '@aws-cdk/aws-cloudwatch';
import { Connections, IConnectable, IMachineImage, InstanceType, ISecurityGroup, IVpc, Port, SubnetSelection } from '@aws-cdk/aws-ec2';
import { IApplicationLoadBalancerTarget } from '@aws-cdk/aws-elasticloadbalancingv2';
import { IGrantable, IPolicy, IPrincipal, IRole } from '@aws-cdk/aws-iam';
import { Construct, IResource, ResourceEnvironment, Stack } from '@aws-cdk/core';
import { HealthCheckConfig, IHealthMonitor, IMonitorableFleet, LogGroupFactoryProps } from '../../core';
import { IRenderQueue } from './render-queue';
import { IInstanceUserDataProvider, WorkerSettings } from './worker-configuration';
/**
 * Interface for Deadline Worker Fleet.
 *
 * @stability stable
 */
export interface IWorkerFleet extends IResource, IConnectable, IGrantable {
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowListenerPortTo().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortFrom(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     */
    allowListenerPortFrom(other: IConnectable): void;
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowListenerPortFrom().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortTo(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     */
    allowListenerPortTo(other: IConnectable): void;
}
/**
 * Properties for the Deadline Worker Fleet.
 *
 * @stability stable
 */
export interface WorkerInstanceFleetProps extends WorkerSettings {
    /**
     * VPC to launch the worker fleet in.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
    /**
     * Security Group to assign to this fleet.
     *
     * @default - create new security group
     * @stability stable
     */
    readonly securityGroup?: ISecurityGroup;
    /**
     * An IAM role to associate with the instance profile assigned to its resources.
     *
     * The role must be assumable by the service principal `ec2.amazonaws.com`:
     *
     *     const role = new iam.Role(this, 'MyRole', {
     *       assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
     *     });
     *
     * @default - A role will automatically be created, it can be accessed via the `role` property
     * @stability stable
     */
    readonly role?: IRole;
    /**
     * AMI of the deadline worker to launch.
     *
     * @stability stable
     */
    readonly workerMachineImage: IMachineImage;
    /**
     * Type of instance to launch for executing repository installer.
     *
     * @default - a T2-Large type will be used.
     * @stability stable
     */
    readonly instanceType?: InstanceType;
    /**
     * Where to place the instance within the VPC.
     *
     * @default - Private subnets.
     * @stability stable
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * Name of SSH keypair to grant access to instance.
     *
     * @default - No SSH access will be possible.
     * @stability stable
     */
    readonly keyName?: string;
    /**
     * Initial amount of workers in the fleet.
     *
     * If this is set to a number, every deployment will reset the amount of
     * workers to this number. It is recommended to leave this value blank.
     *
     * @default minCapacity, and leave unchanged during deployment
     * @stability stable
     */
    readonly desiredCapacity?: number;
    /**
     * Minimum number of instances in the fleet.
     *
     * @default 1
     * @stability stable
     */
    readonly minCapacity?: number;
    /**
     * Maximum number of instances in the fleet.
     *
     * @default desiredCapacity, or minCapacity if desiredCapacity is not set
     * @stability stable
     */
    readonly maxCapacity?: number;
    /**
     * Endpoint for the RenderQueue, to which the worker fleet needs to be connected.
     *
     * @stability stable
     */
    readonly renderQueue: IRenderQueue;
    /**
     * Properties for setting up the Deadline Worker's LogGroup.
     *
     * @default - LogGroup will be created with all properties' default values and a prefix of "/renderfarm/".
     * @stability stable
     */
    readonly logGroupProps?: LogGroupFactoryProps;
    /**
     * Health Monitor component to monitor the health of instances.
     *
     * Note: The health-check feature is supported with Deadline Client v10.1.9 and later.
     *
     * @default - Health Monitoring is turned-off
     * @stability stable
     */
    readonly healthMonitor?: IHealthMonitor;
    /**
     * Properties for configuring a health check.
     *
     * Note: The health-check feature is supported with Deadline Client v10.1.9 and later.
     *
     * @default properties of HealthCheckConfig applies
     * @stability stable
     */
    readonly healthCheckConfig?: HealthCheckConfig;
    /**
     * The maximum hourly price($) to be paid for each Spot instance.
     *
     * min - 0.001; max - 255
     *
     * @default - launches on-demand EC2 instances.
     * @stability stable
     */
    readonly spotPrice?: number;
    /**
     * @stability stable
     */
    readonly blockDevices?: BlockDevice[];
    /**
     * An optional provider of user data commands to be injected at various points during the Worker configuration lifecycle.
     *
     * You can provide a subclass of InstanceUserDataProvider with the methods overridden as desired.
     *
     * @stability stable
     */
    readonly userDataProvider?: IInstanceUserDataProvider;
}
/**
 *  A new or imported Deadline Worker Fleet.
 */
declare abstract class WorkerInstanceFleetBase extends Construct implements IWorkerFleet, IMonitorableFleet {
    /**
     * The security groups/rules used to allow network connections to the file system.
     */
    abstract readonly connections: Connections;
    /**
     * The principal to grant permissions to.
     */
    abstract readonly grantPrincipal: IPrincipal;
    /**
     * The stack in which this worker fleet is defined.
     */
    abstract readonly stack: Stack;
    /**
     * The environment this resource belongs to.
     */
    abstract readonly env: ResourceEnvironment;
    /**
     * The ASG object created by the construct.
     */
    abstract readonly fleet: AutoScalingGroup;
    /**
     * This field expects the base capacity metric of the fleet against
     * which, the healthy percent will be calculated.
     *
     * eg.: GroupDesiredCapacity for an ASG
     */
    abstract readonly targetCapacityMetric: IMetric;
    /**
     * This field expects the component of type INetworkLoadBalancerTarget
     * which can be attached to Network Load Balancer for monitoring.
     *
     * eg. An AutoScalingGroup
     */
    abstract readonly targetToMonitor: IApplicationLoadBalancerTarget;
    /**
     * This field expects a policy which can be attached to the lambda
     * execution role so that it is capable of suspending the fleet.
     *
     * eg.: autoscaling:UpdateAutoScalingGroup permission for an ASG
     */
    abstract readonly targetUpdatePolicy: IPolicy;
    /**
     * This field expects the maximum instance count this fleet can have.
     */
    abstract readonly targetCapacity: number;
    /**
     * This field expects the scope in which to create the monitoring resource
     * like TargetGroups, Listener etc.
     */
    abstract readonly targetScope: Construct;
    /**
     * @inheritdoc
     */
    abstract allowListenerPortFrom(other: IConnectable): void;
    /**
     * @inheritdoc
     */
    abstract allowListenerPortTo(other: IConnectable): void;
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * ![architecture diagram](/diagrams/deadline/WorkerInstanceFleet.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An EC2 Auto Scaling Group to maintain the number of instances.
 * - An Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Worker, Deadline Launcher, and instance-startup logs for the instances
 *    in the fleet.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The data that is stored on your Worker's local EBS volume can include temporary working files from the applications
 *    that are rendering your jobs and tasks. That data can be sensitive or privileged, so we recommend that you encrypt
 *    the data volumes of these instances using either the provided option or by using an encrypted AMI as your source.
 * - The software on the AMI that is being used by this construct may pose a security risk. We recommend that you adopt a
 *    patching strategy to keep this software current with the latest security patches. Please see
 *    https://docs.aws.amazon.com/rfdk/latest/guide/patching-software.html for more information.
 *
 * @stability stable
 */
export declare class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    /**
     * The min limit for spot price.
     *
     * @stability stable
     */
    static readonly SPOT_PRICE_MIN_LIMIT = 0.001;
    /**
     * The max limit for spot price.
     *
     * @stability stable
     */
    static readonly SPOT_PRICE_MAX_LIMIT = 255;
    /**
     * This determines worker's health based on any hardware or software issues of EC2 instance.
     * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
     * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
     */
    private static readonly DEFAULT_HEALTH_CHECK_INTERVAL;
    /**
     * Default prefix for a LogGroup if one isn't provided in the props.
     */
    private static readonly DEFAULT_LOG_GROUP_PREFIX;
    /**
     * This is the current maximum for number of workers that can be started on a single host. Currently the
     * only thing using this limit is the configuration of the listener ports. More than 8 workers can be started,
     * but only the first 8 will have their ports opened in the workers' security group.
     */
    private static readonly MAX_WORKERS_PER_HOST;
    /**
     * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
     * to take for launch and execute the user-data for deadline worker configuration. As we are setting
     * failure signals in the user-data, any failure will terminate deployment immediately.
     */
    private static readonly RESOURCE_SIGNAL_TIMEOUT;
    /**
     * The ASG object created by the construct.
     *
     * @stability stable
     */
    readonly fleet: AutoScalingGroup;
    /**
     * The security groups/rules used to allow network connections to the file system.
     *
     * @stability stable
     */
    readonly connections: Connections;
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The stack in which this worker fleet is defined.
     *
     * @stability stable
     */
    readonly stack: Stack;
    /**
     * The environment this resource belongs to.
     *
     * @stability stable
     */
    readonly env: ResourceEnvironment;
    /**
     * The port workers listen on to share their logs.
     *
     * @stability stable
     */
    readonly listeningPorts: Port;
    /**
     * This field implements the base capacity metric of the fleet against which, the healthy percent will be calculated.
     *
     * eg.: GroupDesiredCapacity for an ASG
     *
     * @stability stable
     */
    readonly targetCapacityMetric: IMetric;
    /**
     * This field implements the component of type INetworkLoadBalancerTarget which can be attached to Network Load Balancer for monitoring.
     *
     * eg. An AutoScalingGroup
     *
     * @stability stable
     */
    readonly targetToMonitor: IApplicationLoadBalancerTarget;
    /**
     * This field implements a policy which can be attached to the lambda execution role so that it is capable of suspending the fleet.
     *
     * eg.: autoscaling:UpdateAutoScalingGroup permission for an ASG
     *
     * @stability stable
     */
    readonly targetUpdatePolicy: IPolicy;
    /**
     * This field implements the maximum instance count this fleet can have.
     *
     * @stability stable
     */
    readonly targetCapacity: number;
    /**
     * This field implements the scope in which to create the monitoring resource like TargetGroups, Listener etc.
     *
     * @stability stable
     */
    readonly targetScope: Construct;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: WorkerInstanceFleetProps);
    /**
     * Add the security group to all workers.
     *
     * @param securityGroup : The security group to add.
     * @stability stable
     */
    addSecurityGroup(securityGroup: ISecurityGroup): void;
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that depends on this stack.
     *
     * If this stack depends on the other stack, use allowListenerPortTo().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortFrom(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortFrom(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowListenerPortFrom(other: IConnectable): void;
    /**
     * Allow access to the worker's remote command listener port (configured as a part of the WorkerConfiguration) for an IConnectable that is either in this stack, or in a stack that this stack depends on.
     *
     * If the other stack depends on this stack, use allowListenerPortFrom().
     *
     * Common uses are:
     *
     *    Adding a SecurityGroup:
     *      `workerFleet.allowListenerPortTo(securityGroup)`
     *
     *    Adding a CIDR:
     *      `workerFleet.allowListenerPortTo(Peer.ipv4('10.0.0.0/24').connections)`
     *
     * @stability stable
     * @inheritdoc true
     */
    allowListenerPortTo(other: IConnectable): void;
    private validateProps;
    private validateSpotPrice;
    private validateRegion;
    private validateArrayGroupsPoolsSyntax;
    private validateBlockDevices;
    private configureHealthMonitor;
}
export {};
