"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
const version_1 = require("./version");
const version_query_1 = require("./version-query");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * ![architecture diagram](/diagrams/deadline/Repository.svg)
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon EFS Point - If no filesystem is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 * - An RFDK PadEfsStorage - If no filesystem is provided.
 * - An AWS Secrets Manager Secret - If no Secret with admin credentials for Deadline Secrets Management is provided.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *    system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *    Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *    Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *    require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *    submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *    that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *    to only those who require it.
 * - If no file-system is provided to the Repository, then the Repository creates an EFS access point with unrestricted
 *    access to the entire EFS file-system. If you would like a single EFS file-system that is used by the Deadline
 *    Repository and other agents, you should supply the file-system and a access-restricted EFS access point to the
 *    Repository construct instead.
 *
 * @stability stable
 */
class Repository extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0;
        super(scope, id);
        if (props.database && ((_b = props.backupOptions) === null || _b === void 0 ? void 0 : _b.databaseRetention)) {
            core_1.Annotations.of(this).addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.filesystem)) {
            core_1.Annotations.of(this).addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_d = props.removalPolicy) === null || _d === void 0 ? void 0 : _d.database)) {
            core_1.Annotations.of(this).addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem instanceof core_2.MountableEfs && !props.fileSystem.accessPoint) {
            throw new Error('When using EFS with the Repository, you must provide an EFS Access Point');
        }
        if (((_f = (_e = props.secretsManagementSettings) === null || _e === void 0 ? void 0 : _e.enabled) !== null && _f !== void 0 ? _f : true) && props.database && !props.database.databaseConstruct) {
            throw new Error('Admin credentials for Deadline Secrets Management cannot be generated when using an imported database. For setting up your own credentials, please refer to https://github.com/aws/aws-rfdk/tree/mainline/packages/aws-rfdk/lib/deadline#configuring-deadline-secrets-management-on-the-repository.');
        }
        this.version = props.version;
        const meetsMinSecretsVersion = !this.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION);
        const secretsManagementIsEnabled = (_h = (_g = props.secretsManagementSettings) === null || _g === void 0 ? void 0 : _g.enabled) !== null && _h !== void 0 ? _h : meetsMinSecretsVersion;
        if (secretsManagementIsEnabled && !meetsMinSecretsVersion) {
            throw new Error(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
        }
        this.secretsManagementSettings = {
            enabled: secretsManagementIsEnabled,
            credentials: (_k = (_j = props.secretsManagementSettings) === null || _j === void 0 ? void 0 : _j.credentials) !== null && _k !== void 0 ? _k : (secretsManagementIsEnabled ? new aws_secretsmanager_1.Secret(((_l = props.database) === null || _l === void 0 ? void 0 : _l.databaseConstruct) ? core_1.Stack.of((_m = props.database) === null || _m === void 0 ? void 0 : _m.databaseConstruct) : this, 'SMAdminUser', {
                description: 'Admin credentials for Deadline Secrets Management',
                generateSecretString: {
                    excludeCharacters: '\"$&\'()/<>[\\]\`{|}',
                    includeSpace: false,
                    passwordLength: 24,
                    requireEachIncludedType: true,
                    generateStringKey: 'password',
                    secretStringTemplate: JSON.stringify({ username: Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME }),
                },
                removalPolicy: (_p = (_o = props.secretsManagementSettings) === null || _o === void 0 ? void 0 : _o.credentialsRemovalPolicy) !== null && _p !== void 0 ? _p : core_1.RemovalPolicy.RETAIN,
            }) : undefined),
        };
        this.fileSystem = (_q = props.fileSystem) !== null && _q !== void 0 ? _q : (() => {
            var _b, _c, _d, _e, _f;
            const fs = new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_d = (_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.filesystem) !== null && _d !== void 0 ? _d : core_1.RemovalPolicy.RETAIN,
                securityGroup: (_e = props.securityGroupsOptions) === null || _e === void 0 ? void 0 : _e.fileSystem,
            });
            const paddingAccess = fs.addAccessPoint('PaddingAccessPoint', {
                createAcl: {
                    ownerGid: '0',
                    ownerUid: '0',
                    permissions: '744',
                },
                path: '/RFDK_PaddingFiles',
            });
            new core_2.PadEfsStorage(this, 'PadEfsStorage', {
                vpc: props.vpc,
                vpcSubnets: (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                accessPoint: paddingAccess,
                desiredPadding: core_1.Size.gibibytes(40),
            });
            const accessPoint = fs.addAccessPoint('AccessPoint', {
                posixUser: {
                    uid: '0',
                    gid: '0',
                },
            });
            return new core_2.MountableEfs(this, {
                filesystem: fs,
                accessPoint,
            });
        })();
        // Set up the Database of the repository
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
                    'Please ensure that the Database provided is configured correctly.';
                core_1.Annotations.of(this).addWarning(warningMsg);
            }
        }
        else {
            const databaseAuditLogging = (_r = props.databaseAuditLogging) !== null && _r !== void 0 ? _r : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_s = props.documentDbInstanceCount) !== null && _s !== void 0 ? _s : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                engineVersion: '3.6.0',
                instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                vpc: props.vpc,
                vpcSubnets: (_t = props.vpcSubnets) !== null && _t !== void 0 ? _t : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
                securityGroup: (_u = props.securityGroupsOptions) === null || _u === void 0 ? void 0 : _u.database,
                instances,
                backup: {
                    retention: (_w = (_v = props.backupOptions) === null || _v === void 0 ? void 0 : _v.databaseRetention) !== null && _w !== void 0 ? _w : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_y = (_x = props.removalPolicy) === null || _x === void 0 ? void 0 : _x.database) !== null && _y !== void 0 ? _y : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: (_z = props.vpcSubnets) !== null && _z !== void 0 ? _z : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.replacingUpdate(),
            signals: aws_autoscaling_1.Signals.waitForAll({
                timeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            }),
            securityGroup: (_0 = props.securityGroupsOptions) === null || _0 === void 0 ? void 0 : _0.installer,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version, props.repositorySettings, 
        // Change ownership of the Deadline repository files if-and-only-if the mounted file-system
        // uses the POSIX permissions based on the process' user UID/GID
        this.fileSystem.usesUserPosixPermissions() ? Repository.REPOSITORY_OWNER : undefined);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof version_query_1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(props) {
        var _b, _c;
        const hostMountPoint = (_b = props.containerInstances.filesystemMountPoint) !== null && _b !== void 0 ? _b : '/mnt/repo';
        const majorVersion = core_1.Token.isUnresolved(this.version.majorVersion) ?
            core_1.Token.asString(this.version.majorVersion) : this.version.majorVersion.toString();
        const containerMountPoint = (_c = props.containers.filesystemMountPoint) !== null && _c !== void 0 ? _c : `/opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Note: pathToFileURL messes up CDK Tokens like the one in majorVersion
        const containerMountPointURL = props.containers.filesystemMountPoint ?
            url_1.pathToFileURL(props.containers.filesystemMountPoint).toString() :
            `file:///opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: containerMountPointURL,
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *    - Configures Deadline to direct-connect to the Repository.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(props) {
        var _b;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        var _b;
        const prefix = (_b = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _b !== void 0 ? _b : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = core_1.Names.uniqueId(this);
        core_1.Tags.of(this.installerGroup).add(tagKey, core_1.Names.uniqueId(this));
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)');
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version, settings, owner) {
        var _b, _c;
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        const installerArgs = [
            '-i', `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
            '-p', `"${installPath}"`,
            '-v', version.linuxFullVersionString(),
        ];
        if (this.secretsManagementSettings.enabled) {
            installerArgs.push('-r', core_1.Stack.of((_b = this.secretsManagementSettings.credentials) !== null && _b !== void 0 ? _b : this).region);
            this.secretsManagementSettings.credentials.grantRead(installerGroup);
            installerArgs.push('-c', (_c = this.secretsManagementSettings.credentials.secretArn) !== null && _c !== void 0 ? _c : '');
        }
        if (settings) {
            const repositorySettingsFilePath = installerGroup.userData.addS3DownloadCommand({
                bucket: settings.bucket,
                bucketKey: settings.s3ObjectKey,
            });
            installerArgs.push('-s', repositorySettingsFilePath);
        }
        // We can ignore this in test coverage because we always use Repository.REPOSITORY_OWNER
        /* istanbul ignore next */
        if (owner) {
            installerArgs.push('-o', `${owner.uid}:${owner.gid}`);
        }
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: installerArgs,
        });
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "aws-rfdk.deadline.Repository", version: "0.38.0" };
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
/**
 * The Repository owner is 1000:1000 (ec2-user on AL2).
 */
Repository.REPOSITORY_OWNER = { uid: 1000, gid: 1000 };
/**
 * Default username for auto generated admin credentials in Secret Manager.
 */
Repository.DEFAULT_SECRETS_MANAGEMENT_USERNAME = 'admin';
//# sourceMappingURL=data:application/json;base64,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