"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const log_group_factory_1 = require("../lib/log-group-factory");
describe('log group', () => {
    test('created correctly with defaults', () => {
        const stack = new core_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: 'testLogGroup',
            RetentionInDays: 3,
        }));
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
    });
    test('created correctly with prefix', () => {
        const stack = new core_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            logGroupPrefix: 'prefix-',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: 'prefix-testLogGroup',
            RetentionInDays: 3,
        }));
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
    });
    test('created correctly with custom retention', () => {
        const stack = new core_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: 'testLogGroup',
            RetentionInDays: 7,
        }));
        assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
    });
});
describe('exporting log group', () => {
    test('created correctly with defaults', () => {
        const stack = new core_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            bucketName: 'test-bucket',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: 'testLogGroup',
            RetentionInDays: 3,
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            Targets: [
                {
                    Input: '{\"BucketName\":\"test-bucket\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"testLogGroup\",\"RetentionInHours\":72}',
                },
            ],
        }));
    });
    test('created correctly with prefix', () => {
        const stack = new core_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            bucketName: 'test-bucket',
            logGroupPrefix: 'prefix-',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: 'prefix-testLogGroup',
            RetentionInDays: 3,
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            Targets: [
                {
                    Input: '{\"BucketName\":\"test-bucket\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"prefix-testLogGroup\",\"RetentionInHours\":72}',
                },
            ],
        }));
    });
    test('created correctly with custom retention', () => {
        const stack = new core_1.Stack();
        // WHEN
        log_group_factory_1.LogGroupFactory.createOrFetch(stack, 'TestId', 'testLogGroup', {
            bucketName: 'test-bucket',
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: 'testLogGroup',
            RetentionInDays: 7,
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Role: {
                'Fn::GetAtt': [
                    'LogGroupExporter6382448ce4b242e9b14fa0a9ccdb198eServiceRoleB67C808B',
                    'Arn',
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
            Targets: [
                {
                    Input: '{\"BucketName\":\"test-bucket\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"testLogGroup\",\"RetentionInHours\":168}',
                },
            ],
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibG9nLWdyb3VwLWZhY3RvcnkudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImxvZy1ncm91cC1mYWN0b3J5LnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7R0FHRzs7QUFFSCw0Q0FBd0U7QUFDeEUsZ0RBQWtEO0FBQ2xELHdDQUFzQztBQUN0QyxnRUFBMkQ7QUFFM0QsUUFBUSxDQUFDLFdBQVcsRUFBRSxHQUFHLEVBQUU7SUFDekIsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtRQUMzQyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE9BQU87UUFDUCxtQ0FBZSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRS9ELE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLHNCQUFzQixFQUFFO1lBQzNELFlBQVksRUFBRSxjQUFjO1lBQzVCLGVBQWUsRUFBRSxDQUFDO1NBQ25CLENBQUMsQ0FBQyxDQUFDO1FBQ0osZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRztZQUNoRSxJQUFJLEVBQUU7Z0JBQ0osWUFBWSxFQUFFO29CQUNaLHFFQUFxRTtvQkFDckUsS0FBSztpQkFDTjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywrQkFBK0IsRUFBRSxHQUFHLEVBQUU7UUFDekMsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixPQUFPO1FBQ1AsbUNBQWUsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxjQUFjLEVBQUU7WUFDN0QsY0FBYyxFQUFFLFNBQVM7U0FDMUIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsc0JBQXNCLEVBQUU7WUFDM0QsWUFBWSxFQUFFLHFCQUFxQjtZQUNuQyxlQUFlLEVBQUUsQ0FBQztTQUNuQixDQUFDLENBQUMsQ0FBQztRQUNKLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxLQUFLLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUc7WUFDaEUsSUFBSSxFQUFFO2dCQUNKLFlBQVksRUFBRTtvQkFDWixxRUFBcUU7b0JBQ3JFLEtBQUs7aUJBQ047YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMseUNBQXlDLEVBQUUsR0FBRyxFQUFFO1FBQ25ELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsT0FBTztRQUNQLG1DQUFlLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsY0FBYyxFQUFFO1lBQzdELFNBQVMsRUFBRSx3QkFBYSxDQUFDLFFBQVE7U0FDbEMsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsc0JBQXNCLEVBQUU7WUFDM0QsWUFBWSxFQUFFLGNBQWM7WUFDNUIsZUFBZSxFQUFFLENBQUM7U0FDbkIsQ0FBQyxDQUFDLENBQUM7UUFDSixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDLHlCQUFnQixDQUFDLHVCQUF1QixFQUFHO1lBQ2hFLElBQUksRUFBRTtnQkFDSixZQUFZLEVBQUU7b0JBQ1oscUVBQXFFO29CQUNyRSxLQUFLO2lCQUNOO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztJQUNOLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxRQUFRLENBQUMscUJBQXFCLEVBQUUsR0FBRyxFQUFFO0lBQ25DLElBQUksQ0FBQyxpQ0FBaUMsRUFBRSxHQUFHLEVBQUU7UUFDM0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixPQUFPO1FBQ1AsbUNBQWUsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxjQUFjLEVBQUU7WUFDN0QsVUFBVSxFQUFFLGFBQWE7U0FDMUIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsc0JBQXNCLEVBQUU7WUFDM0QsWUFBWSxFQUFFLGNBQWM7WUFDNUIsZUFBZSxFQUFFLENBQUM7U0FDbkIsQ0FBQyxDQUFDLENBQUM7UUFFSixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLHVCQUF1QixFQUFHO1lBQzdELElBQUksRUFBRTtnQkFDSixZQUFZLEVBQUU7b0JBQ1oscUVBQXFFO29CQUNyRSxLQUFLO2lCQUNOO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsbUJBQW1CLEVBQUU7WUFDeEQsT0FBTyxFQUFFO2dCQUNQO29CQUNFLEtBQUssRUFBRSx5SEFBeUg7aUJBQ2pJO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztJQUNOLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtRQUN6QyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLE9BQU87UUFDUCxtQ0FBZSxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLGNBQWMsRUFBRTtZQUM3RCxVQUFVLEVBQUUsYUFBYTtZQUN6QixjQUFjLEVBQUUsU0FBUztTQUMxQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxzQkFBc0IsRUFBRTtZQUMzRCxZQUFZLEVBQUUscUJBQXFCO1lBQ25DLGVBQWUsRUFBRSxDQUFDO1NBQ25CLENBQUMsQ0FBQyxDQUFDO1FBRUosZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyx1QkFBdUIsRUFBRztZQUM3RCxJQUFJLEVBQUU7Z0JBQ0osWUFBWSxFQUFFO29CQUNaLHFFQUFxRTtvQkFDckUsS0FBSztpQkFDTjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG1CQUFtQixFQUFFO1lBQ3hELE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxLQUFLLEVBQUUsZ0lBQWdJO2lCQUN4STthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx5Q0FBeUMsRUFBRSxHQUFHLEVBQUU7UUFDbkQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixPQUFPO1FBQ1AsbUNBQWUsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxjQUFjLEVBQUU7WUFDN0QsVUFBVSxFQUFFLGFBQWE7WUFDekIsU0FBUyxFQUFFLHdCQUFhLENBQUMsUUFBUTtTQUNsQyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxzQkFBc0IsRUFBRTtZQUMzRCxZQUFZLEVBQUUsY0FBYztZQUM1QixlQUFlLEVBQUUsQ0FBQztTQUNuQixDQUFDLENBQUMsQ0FBQztRQUVKLGVBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMseUJBQWdCLENBQUMsdUJBQXVCLEVBQUc7WUFDN0QsSUFBSSxFQUFFO2dCQUNKLFlBQVksRUFBRTtvQkFDWixxRUFBcUU7b0JBQ3JFLEtBQUs7aUJBQ047YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxtQkFBbUIsRUFBRTtZQUN4RCxPQUFPLEVBQUU7Z0JBQ1A7b0JBQ0UsS0FBSyxFQUFFLDBIQUEwSDtpQkFDbEk7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCB7IGV4cGVjdCBhcyBleHBlY3RDREssIGhhdmVSZXNvdXJjZUxpa2UgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0IHsgUmV0ZW50aW9uRGF5cyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sb2dzJztcbmltcG9ydCB7IFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBMb2dHcm91cEZhY3RvcnkgfSBmcm9tICcuLi9saWIvbG9nLWdyb3VwLWZhY3RvcnknO1xuXG5kZXNjcmliZSgnbG9nIGdyb3VwJywgKCkgPT4ge1xuICB0ZXN0KCdjcmVhdGVkIGNvcnJlY3RseSB3aXRoIGRlZmF1bHRzJywgKCkgPT4ge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgTG9nR3JvdXBGYWN0b3J5LmNyZWF0ZU9yRmV0Y2goc3RhY2ssICdUZXN0SWQnLCAndGVzdExvZ0dyb3VwJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdDdXN0b206OkxvZ1JldGVudGlvbicsIHtcbiAgICAgIExvZ0dyb3VwTmFtZTogJ3Rlc3RMb2dHcm91cCcsXG4gICAgICBSZXRlbnRpb25JbkRheXM6IDMsXG4gICAgfSkpO1xuICAgIGV4cGVjdENESyhzdGFjaykubm90VG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywgIHtcbiAgICAgIFJvbGU6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0xvZ0dyb3VwRXhwb3J0ZXI2MzgyNDQ4Y2U0YjI0MmU5YjE0ZmEwYTljY2RiMTk4ZVNlcnZpY2VSb2xlQjY3QzgwOEInLFxuICAgICAgICAgICdBcm4nLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NyZWF0ZWQgY29ycmVjdGx5IHdpdGggcHJlZml4JywgKCkgPT4ge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgTG9nR3JvdXBGYWN0b3J5LmNyZWF0ZU9yRmV0Y2goc3RhY2ssICdUZXN0SWQnLCAndGVzdExvZ0dyb3VwJywge1xuICAgICAgbG9nR3JvdXBQcmVmaXg6ICdwcmVmaXgtJyxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0N1c3RvbTo6TG9nUmV0ZW50aW9uJywge1xuICAgICAgTG9nR3JvdXBOYW1lOiAncHJlZml4LXRlc3RMb2dHcm91cCcsXG4gICAgICBSZXRlbnRpb25JbkRheXM6IDMsXG4gICAgfSkpO1xuICAgIGV4cGVjdENESyhzdGFjaykubm90VG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywgIHtcbiAgICAgIFJvbGU6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0xvZ0dyb3VwRXhwb3J0ZXI2MzgyNDQ4Y2U0YjI0MmU5YjE0ZmEwYTljY2RiMTk4ZVNlcnZpY2VSb2xlQjY3QzgwOEInLFxuICAgICAgICAgICdBcm4nLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NyZWF0ZWQgY29ycmVjdGx5IHdpdGggY3VzdG9tIHJldGVudGlvbicsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIExvZ0dyb3VwRmFjdG9yeS5jcmVhdGVPckZldGNoKHN0YWNrLCAnVGVzdElkJywgJ3Rlc3RMb2dHcm91cCcsIHtcbiAgICAgIHJldGVudGlvbjogUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0N1c3RvbTo6TG9nUmV0ZW50aW9uJywge1xuICAgICAgTG9nR3JvdXBOYW1lOiAndGVzdExvZ0dyb3VwJyxcbiAgICAgIFJldGVudGlvbkluRGF5czogNyxcbiAgICB9KSk7XG4gICAgZXhwZWN0Q0RLKHN0YWNrKS5ub3RUbyhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCAge1xuICAgICAgUm9sZToge1xuICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAnTG9nR3JvdXBFeHBvcnRlcjYzODI0NDhjZTRiMjQyZTliMTRmYTBhOWNjZGIxOThlU2VydmljZVJvbGVCNjdDODA4QicsXG4gICAgICAgICAgJ0FybicsXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pKTtcbiAgfSk7XG59KTtcblxuZGVzY3JpYmUoJ2V4cG9ydGluZyBsb2cgZ3JvdXAnLCAoKSA9PiB7XG4gIHRlc3QoJ2NyZWF0ZWQgY29ycmVjdGx5IHdpdGggZGVmYXVsdHMnLCAoKSA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBMb2dHcm91cEZhY3RvcnkuY3JlYXRlT3JGZXRjaChzdGFjaywgJ1Rlc3RJZCcsICd0ZXN0TG9nR3JvdXAnLCB7XG4gICAgICBidWNrZXROYW1lOiAndGVzdC1idWNrZXQnLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQ3VzdG9tOjpMb2dSZXRlbnRpb24nLCB7XG4gICAgICBMb2dHcm91cE5hbWU6ICd0ZXN0TG9nR3JvdXAnLFxuICAgICAgUmV0ZW50aW9uSW5EYXlzOiAzLFxuICAgIH0pKTtcblxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywgIHtcbiAgICAgIFJvbGU6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0xvZ0dyb3VwRXhwb3J0ZXI2MzgyNDQ4Y2U0YjI0MmU5YjE0ZmEwYTljY2RiMTk4ZVNlcnZpY2VSb2xlQjY3QzgwOEInLFxuICAgICAgICAgICdBcm4nLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkV2ZW50czo6UnVsZScsIHtcbiAgICAgIFRhcmdldHM6IFtcbiAgICAgICAge1xuICAgICAgICAgIElucHV0OiAne1xcXCJCdWNrZXROYW1lXFxcIjpcXFwidGVzdC1idWNrZXRcXFwiLFxcXCJFeHBvcnRGcmVxdWVuY3lJbkhvdXJzXFxcIjoxLFxcXCJMb2dHcm91cE5hbWVcXFwiOlxcXCJ0ZXN0TG9nR3JvdXBcXFwiLFxcXCJSZXRlbnRpb25JbkhvdXJzXFxcIjo3Mn0nLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NyZWF0ZWQgY29ycmVjdGx5IHdpdGggcHJlZml4JywgKCkgPT4ge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgTG9nR3JvdXBGYWN0b3J5LmNyZWF0ZU9yRmV0Y2goc3RhY2ssICdUZXN0SWQnLCAndGVzdExvZ0dyb3VwJywge1xuICAgICAgYnVja2V0TmFtZTogJ3Rlc3QtYnVja2V0JyxcbiAgICAgIGxvZ0dyb3VwUHJlZml4OiAncHJlZml4LScsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdDdXN0b206OkxvZ1JldGVudGlvbicsIHtcbiAgICAgIExvZ0dyb3VwTmFtZTogJ3ByZWZpeC10ZXN0TG9nR3JvdXAnLFxuICAgICAgUmV0ZW50aW9uSW5EYXlzOiAzLFxuICAgIH0pKTtcblxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywgIHtcbiAgICAgIFJvbGU6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0xvZ0dyb3VwRXhwb3J0ZXI2MzgyNDQ4Y2U0YjI0MmU5YjE0ZmEwYTljY2RiMTk4ZVNlcnZpY2VSb2xlQjY3QzgwOEInLFxuICAgICAgICAgICdBcm4nLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG5cbiAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RXZlbnRzOjpSdWxlJywge1xuICAgICAgVGFyZ2V0czogW1xuICAgICAgICB7XG4gICAgICAgICAgSW5wdXQ6ICd7XFxcIkJ1Y2tldE5hbWVcXFwiOlxcXCJ0ZXN0LWJ1Y2tldFxcXCIsXFxcIkV4cG9ydEZyZXF1ZW5jeUluSG91cnNcXFwiOjEsXFxcIkxvZ0dyb3VwTmFtZVxcXCI6XFxcInByZWZpeC10ZXN0TG9nR3JvdXBcXFwiLFxcXCJSZXRlbnRpb25JbkhvdXJzXFxcIjo3Mn0nLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NyZWF0ZWQgY29ycmVjdGx5IHdpdGggY3VzdG9tIHJldGVudGlvbicsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIExvZ0dyb3VwRmFjdG9yeS5jcmVhdGVPckZldGNoKHN0YWNrLCAnVGVzdElkJywgJ3Rlc3RMb2dHcm91cCcsIHtcbiAgICAgIGJ1Y2tldE5hbWU6ICd0ZXN0LWJ1Y2tldCcsXG4gICAgICByZXRlbnRpb246IFJldGVudGlvbkRheXMuT05FX1dFRUssXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdDdXN0b206OkxvZ1JldGVudGlvbicsIHtcbiAgICAgIExvZ0dyb3VwTmFtZTogJ3Rlc3RMb2dHcm91cCcsXG4gICAgICBSZXRlbnRpb25JbkRheXM6IDcsXG4gICAgfSkpO1xuXG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCAge1xuICAgICAgUm9sZToge1xuICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAnTG9nR3JvdXBFeHBvcnRlcjYzODI0NDhjZTRiMjQyZTliMTRmYTBhOWNjZGIxOThlU2VydmljZVJvbGVCNjdDODA4QicsXG4gICAgICAgICAgJ0FybicsXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pKTtcblxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFdmVudHM6OlJ1bGUnLCB7XG4gICAgICBUYXJnZXRzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBJbnB1dDogJ3tcXFwiQnVja2V0TmFtZVxcXCI6XFxcInRlc3QtYnVja2V0XFxcIixcXFwiRXhwb3J0RnJlcXVlbmN5SW5Ib3Vyc1xcXCI6MSxcXFwiTG9nR3JvdXBOYW1lXFxcIjpcXFwidGVzdExvZ0dyb3VwXFxcIixcXFwiUmV0ZW50aW9uSW5Ib3Vyc1xcXCI6MTY4fScsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0pKTtcbiAgfSk7XG59KTtcbiJdfQ==