"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const deployment_instance_1 = require("../lib/deployment-instance");
const tag_helpers_1 = require("./tag-helpers");
const DEFAULT_CONSTRUCT_ID = 'DeploymentInstance';
/**
 * Machine image that spies on the following user data methods:
 *
 * * `.addOnExitCommands`
 * * `.addExecuteFileCommand`
 */
class AmazonLinuxWithUserDataSpy extends aws_ec2_1.AmazonLinuxImage {
    getImage(scope) {
        const result = super.getImage(scope);
        jest.spyOn(result.userData, 'addOnExitCommands');
        jest.spyOn(result.userData, 'addExecuteFileCommand');
        return result;
    }
}
describe('DeploymentInstance', () => {
    let app;
    let depStack;
    let stack;
    let vpc;
    let target;
    beforeAll(() => {
        // GIVEN
        app = new cdk.App();
        depStack = new cdk.Stack(app, 'DepStack');
        vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
    });
    describe('defaults', () => {
        beforeAll(() => {
            // GIVEN
            stack = new cdk.Stack(app, 'DefaultsStack');
            target = new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
                vpc,
            });
        });
        describe('Auto-Scaling Group', () => {
            // Only one ASG is deployed. This is an anchor for the tests that follow. Each test is independent and not
            // guaranteed to match on the same resource in the CloudFormation template. Having a test that asserts a single
            // ASG makes these assertions linked
            test('deploys a single Auto-Scaling Group', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.countResources('AWS::AutoScaling::AutoScalingGroup', 1));
            });
            test('MaxSize is 1', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MaxSize: '1',
                }));
            });
            test('MinSize is 1', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: '1',
                }));
            });
            test('uses private subnets', () => {
                // GIVEN
                const privateSubnetIDs = vpc.selectSubnets({ subnetType: aws_ec2_1.SubnetType.PRIVATE }).subnetIds;
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    VPCZoneIdentifier: assert_1.arrayWith(...stack.resolve(privateSubnetIDs)),
                }));
            });
            test('waits 15 minutes for one signal', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    CreationPolicy: {
                        ResourceSignal: {
                            Count: 1,
                            Timeout: 'PT15M',
                        },
                    },
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('sets replacing update policy', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    UpdatePolicy: {
                        AutoScalingReplacingUpdate: {
                            WillReplace: true,
                        },
                        AutoScalingScheduledAction: {
                            IgnoreUnmodifiedGroupSizeProperties: true,
                        },
                    },
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('uses Launch Configuration', () => {
                // GIVEN
                const launchConfig = target.node.findChild('ASG').node.findChild('LaunchConfig');
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    LaunchConfigurationName: stack.resolve(launchConfig.ref),
                }));
            });
        });
        describe('Launch Configuration', () => {
            // Only one ASG is deployed. This is an anchor for the tests that follow. Each test is independent and not
            // guaranteed to match on the same resource in the CloudFormation template. Having a test that asserts a single
            // ASG makes these assertions linked
            test('deploys a single Launch Configuration', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.countResources('AWS::AutoScaling::LaunchConfiguration', 1));
            });
            test('uses latest Amazon Linux machine image', () => {
                // GIVEN
                const amazonLinux = aws_ec2_1.MachineImage.latestAmazonLinux();
                const imageId = stack.resolve(amazonLinux.getImage(stack)).imageId;
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    ImageId: imageId,
                }));
            });
            test('uses t3.small', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    InstanceType: 't3.small',
                }));
            });
            test('Uses created Security Group', () => {
                // GIVEN
                const securityGroup = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceSecurityGroup'));
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    SecurityGroups: [
                        stack.resolve(securityGroup.securityGroupId),
                    ],
                }));
            });
            test('depends on policy', () => {
                // GIVEN
                const policy = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceRole')
                    .node.findChild('DefaultPolicy')
                    .node.defaultChild);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
                    DependsOn: assert_1.arrayWith(stack.resolve(policy.logicalId)),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
        });
        describe('Security Group', () => {
            test('creates Security Group in the desired VPC', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
                    VpcId: stack.resolve(vpc.vpcId),
                }));
            });
        });
        describe('ASG IAM role', () => {
            let instanceRole;
            beforeAll(() => {
                // GIVEN
                instanceRole = (target
                    .node.findChild('ASG')
                    .node.findChild('InstanceRole')
                    .node.defaultChild);
            });
            test('creates an instance profile', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::InstanceProfile', {
                    Roles: [
                        { Ref: stack.getLogicalId(instanceRole) },
                    ],
                }));
            });
            test('creates a role that can be assumed by EC2', () => {
                // GIVEN
                const servicePrincipal = new iam.ServicePrincipal('ec2.amazonaws.com');
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Role', {
                    AssumeRolePolicyDocument: {
                        Statement: [
                            {
                                Action: 'sts:AssumeRole',
                                Effect: 'Allow',
                                Principal: {
                                    Service: stack.resolve(servicePrincipal.policyFragment.principalJson).Service[0],
                                },
                            },
                        ],
                    },
                }));
            });
            test('can signal to CloudFormation', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: 'cloudformation:SignalResource',
                            Effect: 'Allow',
                            Resource: { Ref: 'AWS::StackId' },
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can write to the log group', () => {
                // GIVEN
                const logGroup = target.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'logs:CreateLogStream',
                                'logs:PutLogEvents',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(logGroup.logGroupArn),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can fetch the CloudWatch Agent install script', () => {
                // GIVEN
                const cloudWatchAgentScriptAsset = (target
                    .node.findChild('CloudWatchConfigurationScriptAsset'));
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve([
                                cloudWatchAgentScriptAsset.bucket.bucketArn,
                                cloudWatchAgentScriptAsset.bucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can fetch the CloudWatch Agent configuration file SSM parameter', () => {
                // GIVEN
                const cloudWatchConfigSsmParam = (target
                    .node.findChild('StringParameter'));
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'ssm:DescribeParameters',
                                'ssm:GetParameters',
                                'ssm:GetParameter',
                                'ssm:GetParameterHistory',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve(cloudWatchConfigSsmParam.parameterArn),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can fetch the CloudWatch Agent installer from S3', () => {
                // GIVEN
                const cloudWatchAgentInstallerBucket = aws_s3_1.Bucket.fromBucketArn(depStack, 'CloudWatchAgentInstallerBucket', `arn:aws:s3:::amazoncloudwatch-agent-${stack.region}`);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve([
                                cloudWatchAgentInstallerBucket.bucketArn,
                                cloudWatchAgentInstallerBucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can fetch GPG installer from RFDK dependencies S3 bucket', () => {
                // GIVEN
                const rfdkExternalDepsBucket = aws_s3_1.Bucket.fromBucketArn(depStack, 'RfdkExternalDependenciesBucket', `arn:aws:s3:::rfdk-external-dependencies-${stack.region}`);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                            ],
                            Effect: 'Allow',
                            Resource: stack.resolve([
                                rfdkExternalDepsBucket.bucketArn,
                                rfdkExternalDepsBucket.arnForObjects('*'),
                            ]),
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
            test('can scale the Auto-Scaling Group', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: 'autoscaling:UpdateAutoScalingGroup',
                            Condition: {
                                // This tag is added by RFDK to scope down the permissions of the policy for least-privilege
                                StringEquals: { 'autoscaling:ResourceTag/resourceLogicalId': cdk.Names.uniqueId(target) },
                            },
                            Effect: 'Allow',
                            Resource: '*',
                        }, 
                        // The instance determines its Auto-Scaling Group by reading the tag created on the instance by the EC2
                        // Auto-Scaling service
                        {
                            Action: 'ec2:DescribeTags',
                            Effect: 'Allow',
                            Resource: '*',
                        }),
                    },
                    Roles: [
                        stack.resolve(instanceRole.ref),
                    ],
                }));
            });
        });
        describe('CloudWatch Agent config SSM parameter', () => {
            test('configures log group', () => {
                // GIVEN
                const logGroup = target.node.findChild(`${DEFAULT_CONSTRUCT_ID}LogGroup`);
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
                    Type: 'String',
                    Value: {
                        'Fn::Join': [
                            '',
                            assert_1.arrayWith('{"logs":{"logs_collected":{"files":{"collect_list":[{"log_group_name":"', stack.resolve(logGroup.logGroupName)),
                        ],
                    },
                }));
            });
            test('configures cloud-init log', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
                    Type: 'String',
                    Value: {
                        'Fn::Join': [
                            '',
                            assert_1.arrayWith(assert_1.stringLike('*"log_stream_name":"cloud-init-output-{instance_id}","file_path":"/var/log/cloud-init-output.log",*')),
                        ],
                    },
                }));
            });
        });
        describe('Tags resources with RFDK meta-data', () => {
            tag_helpers_1.testConstructTags({
                constructName: 'DeploymentInstance',
                createConstruct: () => {
                    return stack;
                },
                resourceTypeCounts: {
                    'AWS::EC2::SecurityGroup': 1,
                    'AWS::IAM::Role': 1,
                    'AWS::AutoScaling::AutoScalingGroup': 1,
                    'AWS::SSM::Parameter': 1,
                },
            });
        });
        // RFDK adds the resourceLogicalId tag to the Auto-Scaling Group in order to scope down the permissions of the
        // IAM policy given to the EC2 instance profile so that only that ASG can be scaled by the instance.
        test('Tagging for self-termination', () => {
            // THEN
            const matcher = tag_helpers_1.resourceTagMatcher('AWS::AutoScaling::AutoScalingGroup', 'resourceLogicalId', cdk.Names.uniqueId(target));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', matcher));
        });
    });
    describe('User Data', () => {
        beforeAll(() => {
            // GIVEN
            stack = new cdk.Stack(app, 'UserDataStack');
            // WHEN
            target = new deployment_instance_1.DeploymentInstance(stack, 'DeploymentInstanceNew', {
                vpc,
                // a hack to be able to spy on the user data's "addOnExitCommand" and "addExecuteFileCommand" methods.
                machineImage: new AmazonLinuxWithUserDataSpy(),
            });
        });
        test('configures self-termination', () => {
            // THEN
            expect(target.userData.addOnExitCommands).toHaveBeenCalledWith('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)', 'INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"', `ASG="$(aws --region ${stack.region} ec2 describe-tags --filters "Name=resource-id,Values=\${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"`, `aws --region ${stack.region} autoscaling update-auto-scaling-group --auto-scaling-group-name \${ASG} --min-size 0 --max-size 0 --desired-capacity 0`);
        });
        test('configures CloudWatch Agent', () => {
            // GIVEN
            const spy = target.userData.addExecuteFileCommand;
            const cloudWatchConfigSsmParam = (target
                .node.findChild('StringParameter'));
            // THEN
            // Should have been called
            expect(spy.mock.calls.length).toBeGreaterThanOrEqual(1);
            // The first call...
            const executeFileOptions = spy.mock.calls[0][0];
            // Should have been called with arguments
            const args = executeFileOptions.arguments;
            expect(args).not.toBeUndefined();
            const splitArgs = args.split(' ');
            // Should have three arguments
            expect(splitArgs).toHaveLength(3);
            // Specify the flag to install the CloudWatch Agent
            expect(splitArgs[0]).toEqual('-i');
            // Should pass the region
            expect(stack.resolve(splitArgs[1])).toEqual(stack.resolve(stack.region));
            // Should pass the SSM parameter containing the CloudWatch Agent configuration
            expect(stack.resolve(splitArgs[2])).toEqual(stack.resolve(cloudWatchConfigSsmParam.parameterName));
        });
    });
    describe('Custom::LogRetention.LogGroupName', () => {
        beforeEach(() => {
            // We need a clean construct tree, because the tests use the same construct ID
            app = new cdk.App();
            depStack = new cdk.Stack(app, 'DepStack');
            vpc = new aws_ec2_1.Vpc(depStack, 'VPC');
            stack = new cdk.Stack(app, 'Stack');
        });
        // GIVEN
        test.each([
            [
                {},
                // defaults expected final log group name
                `/renderfarm/${DEFAULT_CONSTRUCT_ID}`,
            ],
            [
                { logGroupName: 'foo' },
                // expected final log group name
                '/renderfarm/foo',
            ],
            [
                {
                    logGroupPrefix: 'logGroupPrefix',
                },
                // expected final log group name
                `logGroupPrefix${DEFAULT_CONSTRUCT_ID}`,
            ],
            [
                {
                    logGroupName: 'logGroupName',
                    logGroupPrefix: 'logGroupPrefix',
                },
                // expected final log group name
                'logGroupPrefixlogGroupName',
            ],
        ])('%s => %s', ({ logGroupName, logGroupPrefix }, expectedLogGroupName) => {
            // WHEN
            new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
                vpc,
                logGroupName,
                logGroupProps: logGroupPrefix ? { logGroupPrefix } : undefined,
            });
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
                LogGroupName: expectedLogGroupName,
            }));
        });
    });
    // GIVEN
    test('uses specified instance type', () => {
        // GIVEN
        const instanceType = new aws_ec2_1.InstanceType('c5.large');
        // We want an isolated stack to ensure expectCDK is only searching resources
        // synthesized by the specific DeploymentInstance stack
        stack = new cdk.Stack(app, 'InstanceTypeStack');
        // WHEN
        new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, {
            vpc,
            instanceType,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: instanceType.toString(),
        }));
    });
    describe('.selfTermination = false', () => {
        beforeAll(() => {
            // GIVEN
            stack = new cdk.Stack(app, 'SelfTerminationDisabledStack');
            // Spy on user data method calls
            const machineImage = new AmazonLinuxWithUserDataSpy();
            const deploymentInstanceProps = {
                vpc,
                selfTerminate: false,
                machineImage,
            };
            // WHEN
            target = new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, deploymentInstanceProps);
        });
        test('does not add on-exit commands', () => {
            // THEN
            expect(target.userData.addOnExitCommands).not.toHaveBeenCalledWith(expect.arrayContaining([
                expect.stringMatching(/\baws\s+.*\bautoscaling\s+update-auto-scaling-group/),
            ]));
        });
        test('is not granted IAM permissions to scale the Auto-Scaling Group', () => {
            // GIVEN
            const instanceRole = (target
                .node.findChild('ASG')
                .node.findChild('InstanceRole')
                .node.defaultChild);
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 'autoscaling:UpdateAutoScalingGroup',
                        Condition: {
                            // This tag is added by RFDK to scope down the permissions of the policy for least-privilege
                            StringEquals: { 'autoscaling:ResourceTag/resourceLogicalId': cdk.Names.uniqueId(target) },
                        },
                        Effect: 'Allow',
                        Resource: '*',
                    }, 
                    // The instance determines its Auto-Scaling Group by reading the tag created on the instance by the EC2
                    // Auto-Scaling service
                    {
                        Action: 'ec2:DescribeTags',
                        Effect: 'Allow',
                        Resource: '*',
                    }),
                },
                Roles: [
                    stack.resolve(instanceRole.ref),
                ],
            }));
        });
        test('does not tag for self-termination', () => {
            // THEN
            const matcher = tag_helpers_1.resourceTagMatcher('AWS::AutoScaling::AutoScalingGroup', 'resourceLogicalId', cdk.Names.uniqueId(target));
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', matcher));
        });
    });
    // GIVEN
    describe('.executionTimeout is specified', () => {
        const executionTimeout = cdk.Duration.minutes(30);
        beforeAll(() => {
            // GIVEN
            // Use a clean stack to not pollute other stacks with resources
            stack = new cdk.Stack(app, 'ExecutionTimeout');
            const deploymentInstanceProps = {
                vpc,
                executionTimeout,
            };
            // WHEN
            new deployment_instance_1.DeploymentInstance(stack, DEFAULT_CONSTRUCT_ID, deploymentInstanceProps);
        });
        // THEN
        test('AWS::AutoScaling::AutoScalingGroup creation policy signal timeout is set accordingly', () => {
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                CreationPolicy: {
                    ResourceSignal: {
                        Count: 1,
                        Timeout: executionTimeout.toIsoString(),
                    },
                },
            }, assert_1.ResourcePart.CompleteDefinition));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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