"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableFsxLustre = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon FSx for Lustre File System onto an instance.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class MountableFsxLustre {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const mountName = this.props.fileset ? path.posix.join(this.props.filesystem.mountName, this.props.fileset) : this.props.filesystem.mountName;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, 'bash ./installLustreClient.sh', `bash ./mountFsxLustre.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${mountName} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * Returns whether the mounted file-system evaluates the UID/GID of the system user accessing the file-system.
     *
     * Some network file-systems provide features to fix a UID/GID for all access to the mounted file-system and ignore
     * the system user accessing the file. If this is the case, an implementing class must indicate this in the return
     * value.
     *
     * @stability stable
     * @inheritdoc true
     */
    usesUserPosixPermissions() {
        return true;
    }
    /**
     * Fetch the Asset singleton for the FSx for Lustre mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton(scope) {
        var _b;
        const stack = core_1.Stack.of(scope);
        const uuid = '0db888da-5901-4948-aaa5-e71c541c8060';
        const uniqueId = 'MountableFsxLustreAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountFsxLustre.sh', '!installLustreClient.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableFsxLustre = MountableFsxLustre;
_a = JSII_RTTI_SYMBOL_1;
MountableFsxLustre[_a] = { fqn: "aws-rfdk.MountableFsxLustre", version: "0.38.0" };
//# sourceMappingURL=data:application/json;base64,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