"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountPermissionsHelper = void 0;
const mountable_filesystem_1 = require("./mountable-filesystem");
/**
 * This is an internal-only class that can be used to convert enum values from
 * {@link MountPermissions} into strings, or whatever, that are needed by OS-specific
 * mount utilities.
 */
class MountPermissionsHelper {
    /**
     * Convert the given permission into the appropriate mount option for Linux's mount command.
     *
     * @param permission The permission to convert. Defaults to {@link MountPermissions.READWRITE} if not defined.
     */
    static toLinuxMountOption(permission) {
        permission = permission !== null && permission !== void 0 ? permission : mountable_filesystem_1.MountPermissions.READWRITE;
        switch (permission) {
            case mountable_filesystem_1.MountPermissions.READONLY:
                return 'ro';
            case mountable_filesystem_1.MountPermissions.READWRITE:
                return 'rw';
        }
        throw new Error(`Unhandled MountPermission: ${permission}`);
    }
    /**
     * Convert the given permission into the appropriate list of IAM actions allowed on the EFS FileSystem required for
     * the mount.
     *
     * @param permission The permission to convert. Defaults to {@link MountPermissions.READWRITE} if not defined.
     */
    static toEfsIAMActions(permission) {
        permission = permission !== null && permission !== void 0 ? permission : mountable_filesystem_1.MountPermissions.READWRITE;
        const iamActions = [
            'elasticfilesystem:ClientMount',
        ];
        if (permission === mountable_filesystem_1.MountPermissions.READWRITE) {
            iamActions.push('elasticfilesystem:ClientWrite');
        }
        return iamActions;
    }
}
exports.MountPermissionsHelper = MountPermissionsHelper;
//# sourceMappingURL=data:application/json;base64,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