/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { ICertificate } from '@aws-cdk/aws-certificatemanager';
import { Metric, MetricOptions } from '@aws-cdk/aws-cloudwatch';
import { Table } from '@aws-cdk/aws-dynamodb';
import { IKey } from '@aws-cdk/aws-kms';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { Construct, ResourceEnvironment, Stack, Tag } from '@aws-cdk/core';
/**
 * Properties for importing a Certificate from Secrets into ACM.
 *
 * @stability stable
 */
export interface ImportedAcmCertificateProps {
    /**
     * A Secret that contains the Certificate data.
     *
     * @stability stable
     */
    readonly cert: ISecret;
    /**
     * A Secret that contains the encrypted Private Key data.
     *
     * @stability stable
     */
    readonly key: ISecret;
    /**
     * A Secret that contains the passphrase of the encrypted Private Key.
     *
     * @stability stable
     */
    readonly passphrase: ISecret;
    /**
     * A Secret that contains the chain of Certificates used to sign this Certificate.
     *
     * @default : No certificate chain is used, signifying a self-signed Certificate
     * @stability stable
     */
    readonly certChain?: ISecret;
    /**
     * The KMS Key used to encrypt the secrets.
     *
     * The Custom Resource to import the Certificate to ACM will be granted
     * permission to decrypt Secrets using this Key.
     *
     * @default : If the account's default CMK was used to encrypt the Secrets, no special permissions need to be given
     * @stability stable
     */
    readonly encryptionKey?: IKey;
}
/**
 * A Construct that creates an AWS CloudFormation Custom Resource that models a certificate that is imported into AWS Certificate Manager (ACM).
 *
 * It uses an AWS Lambda Function to extract the certificate from Secrets in AWS SecretsManager
 * and then import it into ACM. The interface is intended to be used with the {@link X509CertificatePem} Construct.
 *
 * ![architecture diagram](/diagrams/core/ImportedAcmCertificate.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - An AWS Lambda Function, with IAM Role - Used to create/update/delete the Custom Resource.
 * - AWS Certificate Manager Certificate - Created by the Custom Resource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda for this construct also has broad IAM permissions to delete any Certificate that is stored
 *    in AWS Certificate Manager. You should not grant any additional actors/principals the ability to modify or
 *    execute this Lambda.
 *
 * @stability stable
 */
export declare class ImportedAcmCertificate extends Construct implements ICertificate {
    private static IMPORTER_UUID;
    private static CERT_LOOKUP_CONSTRUCT_ID;
    /**
     * The ARN for the Certificate that was imported into ACM.
     *
     * @stability stable
     */
    readonly certificateArn: string;
    /**
     * The stack in which this resource is defined.
     *
     * @stability stable
     */
    readonly stack: Stack;
    /**
     * The environment this resource belongs to.
     *
     * For resources that are created and managed by the CDK
     * (generally, those created by creating new class instances like Role, Bucket, etc.),
     * this is always the same as the environment of the stack they belong to;
     * however, for imported resources
     * (those obtained from static methods like fromRoleArn, fromBucketName, etc.),
     * that might be different than the stack they were imported into.
     *
     * @stability stable
     */
    readonly env: ResourceEnvironment;
    /**
     * @stability stable
     */
    protected readonly database: Table;
    /**
     * @stability stable
     */
    protected readonly uniqueTag: Tag;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ImportedAcmCertificateProps);
    /**
     * Return the DaysToExpiry metric for this AWS Certificate Manager Certificate.
     *
     * By default, this is the minimum value over 1 day.
     *
     * This metric is no longer emitted once the certificate has effectively
     * expired, so alarms configured on this metric should probably treat missing
     * data as "breaching".
     *
     * @stability stable
     * @inheritdoc true
     */
    metricDaysToExpiry(props?: MetricOptions): Metric;
}
