"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentInstance = void 0;
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const runtime_info_1 = require("./runtime-info");
/**
 * Deploys an instance that runs its user data on deployment, waits for that user data to succeed, and optionally
 * terminates itself afterwards.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with max capacity of 1 instance.
 * - IAM instance profile, IAM role, and IAM policy
 * - An Amazon CloudWatch log group that contains the instance cloud-init logs
 * - A Lambda Function to fetch and existing Log Group or create a new one
 * - IAM role and policy for the Lambda Function
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 */
class DeploymentInstance extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        this.asg = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: (_a = props.instanceType) !== null && _a !== void 0 ? _a : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.SMALL),
            keyName: props.keyName,
            machineImage: (_b = props.machineImage) !== null && _b !== void 0 ? _b : aws_ec2_1.MachineImage.latestAmazonLinux(),
            vpc: props.vpc,
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            signals: aws_autoscaling_1.Signals.waitForAll({
                timeout: (_d = props.executionTimeout) !== null && _d !== void 0 ? _d : DeploymentInstance.DEFAULT_EXECUTION_TIMEOUT,
            }),
            updatePolicy: aws_autoscaling_1.UpdatePolicy.replacingUpdate(),
        });
        this.node.defaultChild = this.asg;
        this.connections = this.asg.connections;
        const logGroupName = (_e = props.logGroupName) !== null && _e !== void 0 ? _e : id;
        this.configureCloudWatchAgent(this.asg, logGroupName, props.logGroupProps);
        if ((_f = props.selfTerminate) !== null && _f !== void 0 ? _f : true) {
            this.configureSelfTermination();
        }
        this.asg.userData.addSignalOnExitCommand(this.asg);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Make the execution of the instance dependent upon another construct
     *
     * @param dependency The construct that should be dependended upon
     */
    addExecutionDependency(dependency) {
        if (core_1.Construct.isConstruct(dependency)) {
            this.asg.node.defaultChild.node.addDependency(dependency);
        }
    }
    /**
     * @inheritdoc
     */
    get osType() {
        return this.asg.osType;
    }
    /**
     * @inheritdoc
     */
    get userData() {
        return this.asg.userData;
    }
    /**
     * @inheritdoc
     */
    get grantPrincipal() {
        return this.asg.grantPrincipal;
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the deployment instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *
     * @param asg The auto-scaling group
     * @param groupName The name of the Log Group, or suffix of the Log Group if `logGroupProps.logGroupPrefix` is
     *                  specified
     * @param logGroupProps The properties for LogGroupFactory to create or fetch the log group
     */
    configureCloudWatchAgent(asg, groupName, logGroupProps) {
        var _a;
        const prefix = (_a = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _a !== void 0 ? _a : DeploymentInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'DeploymentInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(asg);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        new _1.CloudWatchAgent(this, 'CloudWatchAgent', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: asg,
        });
    }
    configureSelfTermination() {
        // Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources as it will cause
        // cyclic dependency. Hence, using Condition Keys
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${DeploymentInstance.ASG_TAG_KEY}`] = core_1.Names.uniqueId(this);
        core_1.Tags.of(this.asg).add(DeploymentInstance.ASG_TAG_KEY, core_1.Names.uniqueId(this));
        this.asg.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.asg.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.asg.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and ASG name and then setting its capacity to 0
        this.asg.userData.addOnExitCommands('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)', 'INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"', 'ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"', 'aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
}
exports.DeploymentInstance = DeploymentInstance;
/**
 * The tag key name used as an IAM condition to restrict autoscaling API grants
 */
DeploymentInstance.ASG_TAG_KEY = 'resourceLogicalId';
/**
 * How often the CloudWatch agent will flush its log files to CloudWatch
 */
DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The default timeout to wait for CloudFormation success signals before failing the resource create/update
 */
DeploymentInstance.DEFAULT_EXECUTION_TIMEOUT = core_1.Duration.minutes(15);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
DeploymentInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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