/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { AutoScalingGroup } from '@aws-cdk/aws-autoscaling';
import { Connections, IConnectable, IMachineImage, InstanceType, IVpc, SubnetSelection } from '@aws-cdk/aws-ec2';
import { Construct, Duration } from '@aws-cdk/core';
import { IScriptHost, LogGroupFactoryProps } from '.';
/**
 * Properties for constructing a `DeploymentInstance`
 */
export interface DeploymentInstanceProps {
    /**
     * The VPC that the instance should be launched in.
     */
    readonly vpc: IVpc;
    /**
     * The amount of time that CloudFormation should wait for the success signals before failing the create/update.
     *
     * @default 15 minutes
     */
    readonly executionTimeout?: Duration;
    /**
     * The instance type to deploy
     *
     * @default t3.small
     */
    readonly instanceType?: InstanceType;
    /**
     * An optional EC2 keypair name to associate with the instance
     *
     * @default no EC2 keypair is associated with the instance
     */
    readonly keyName?: string;
    /**
     * The log group name for streaming CloudWatch logs
     *
     * @default the construct ID is used
     */
    readonly logGroupName?: string;
    /**
     * Properties for setting up the DeploymentInstance's LogGroup in CloudWatch
     *
     * @default the LogGroup will be created with all properties' default values to the LogGroup: /renderfarm/<construct id>
     */
    readonly logGroupProps?: LogGroupFactoryProps;
    /**
     * The machine image to use.
     *
     * @default latest Amazon Linux 2 image
     */
    readonly machineImage?: IMachineImage;
    /**
     * Whether the instance should self-terminate after the deployment succeeds
     *
     * @default true
     */
    readonly selfTerminate?: boolean;
    /**
     * The subnets to deploy the instance to
     *
     * @default private subnets
     */
    readonly vpcSubnets?: SubnetSelection;
}
/**
 * Deploys an instance that runs its user data on deployment, waits for that user data to succeed, and optionally
 * terminates itself afterwards.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with max capacity of 1 instance.
 * - IAM instance profile, IAM role, and IAM policy
 * - An Amazon CloudWatch log group that contains the instance cloud-init logs
 * - A Lambda Function to fetch and existing Log Group or create a new one
 * - IAM role and policy for the Lambda Function
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 */
export declare class DeploymentInstance extends Construct implements IScriptHost, IConnectable {
    /**
     * The tag key name used as an IAM condition to restrict autoscaling API grants
     */
    private static readonly ASG_TAG_KEY;
    /**
     * How often the CloudWatch agent will flush its log files to CloudWatch
     */
    private static readonly CLOUDWATCH_LOG_FLUSH_INTERVAL;
    /**
     * The default timeout to wait for CloudFormation success signals before failing the resource create/update
     */
    private static readonly DEFAULT_EXECUTION_TIMEOUT;
    /**
     * Default prefix for a LogGroup if one isn't provided in the props.
     */
    private static readonly DEFAULT_LOG_GROUP_PREFIX;
    /**
     * @inheritdoc
     */
    readonly connections: Connections;
    /**
     * The auto-scaling group
     */
    protected readonly asg: AutoScalingGroup;
    constructor(scope: Construct, id: string, props: DeploymentInstanceProps);
    /**
     * Make the execution of the instance dependent upon another construct
     *
     * @param dependency The construct that should be dependended upon
     */
    addExecutionDependency(dependency: any): void;
    /**
     * @inheritdoc
     */
    get osType(): import("@aws-cdk/aws-ec2").OperatingSystemType;
    /**
     * @inheritdoc
     */
    get userData(): import("@aws-cdk/aws-ec2").UserData;
    /**
     * @inheritdoc
     */
    get grantPrincipal(): import("@aws-cdk/aws-iam").IPrincipal;
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the deployment instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *
     * @param asg The auto-scaling group
     * @param groupName The name of the Log Group, or suffix of the Log Group if `logGroupProps.logGroupPrefix` is
     *                  specified
     * @param logGroupProps The properties for LogGroupFactory to create or fetch the log group
     */
    private configureCloudWatchAgent;
    private configureSelfTermination;
}
