"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const fs_1 = require("fs");
const path = require("path");
/**
 * This is the regular expression that validates that the draw.io diagram is embedded within the SVG file.
 *
 * draw.io embeds the diagram as a "content" XML attribute of the <svg> element. This looks like:
 *
 * <svg ... content="...embedded SVG diagram...">
 *   ...
 * </svg>
 *
 * When you choose to not include a copy of the draw.io diagram, this attribute is not written to the file.
 *
 * This is a very simple regular expression that will match the opening <svg> tag and ensure it has a non-empty
 * XML attribute named "content". While this validation doesn't assert the validity of the content attribute,
 * it will catch the common mistake of forgetting to include an embedded copy of the draw.io diagram.
 */
const DRAW_IO_EMBEDDED_XML_ATTR_REGEX = /<svg( [^>]+)?content="[^"]+?"[^>]*?>/;
const DIAGRAM_ROOT = path.join(__dirname, '..');
/**
 * Returns all `.svg` files that are descendants of the `packages/aws-rfdk/docs/diagrams` directory.
 */
function findSvgDiagrams(dirPath) {
    if (!dirPath) {
        dirPath = DIAGRAM_ROOT;
    }
    let diagrams = [];
    const listings = fs.readdirSync(dirPath);
    for (const listing of listings) {
        const listingPath = path.join(dirPath, listing);
        const stat = fs.statSync(listingPath);
        if (stat.isDirectory()) {
            const dirDiagrams = findSvgDiagrams(listingPath);
            diagrams = diagrams.concat(...dirDiagrams);
        }
        else if (path.extname(listing).toLowerCase() === '.svg') {
            diagrams.push(listingPath);
        }
    }
    return diagrams;
}
describe('diagrams', () => {
    const diagrams = findSvgDiagrams();
    describe('have draw.io diagrams embedded', () => {
        diagrams.forEach(diagram => {
            const relativeDiagramPath = path.relative(DIAGRAM_ROOT, diagram);
            // We use string interpolation below, otherwise eslint incorrectly believes that the test name is not a valid
            // string
            test(`${relativeDiagramPath}`, async () => {
                // GIVEN
                const diagramContents = await fs_1.promises.readFile(diagram, { encoding: 'utf8' });
                // THEN
                expect(diagramContents).toMatch(DRAW_IO_EMBEDDED_XML_ATTR_REGEX);
            });
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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