import os

from . import aws, pypi, forms, cmd
from .workflow import Workflow


def main():
    if not os.path.exists(".git"):
        print("This script must be run from a Git repository.")
        return

    print("""# GitHub Actions Wizard\nhttps://github.com/cmdr2/github-actions-wizard\n""")

    interactive_workflow_wizard()


def interactive_workflow_wizard():
    workflow = Workflow()
    workflow.load()

    show_workflow_jobs(workflow)

    action = forms.ask_action_to_perform(workflow)

    if action == "build":
        add_build_job(workflow)
    elif action == "test":
        add_test_job(workflow)
    elif action == "deploy":
        add_deployment_job(workflow)

    update_job_dependencies(workflow)
    ensure_job_order(workflow)
    set_correct_checkout_step(workflow)

    # Write workflow file
    workflow_file = workflow.save()
    print(f"\n✅ Workflow update complete. Workflow written: {workflow_file}. Please customize it as necessary.")


def show_workflow_jobs(workflow):
    jobs = workflow.get_jobs_ids()
    if jobs:
        print("Current workflow jobs:")
        for job in jobs:
            print(f" - {job}")
        print("")


def add_build_job(workflow):
    workflow.add_job("build")
    workflow.add_checkout_step("build")
    workflow.add_job_shell_step("build", "echo Building...", name="Dummy Build Command")
    workflow.add_upload_artifact_step("build", path="build")

    print("Added build step. The deployment steps will now use the artifacts generated by the build step")


def add_test_job(workflow):
    workflow.add_job("test")

    if workflow.has_job("build"):
        workflow.add_download_artifact_step("test", path="build")
    else:
        workflow.add_checkout_step("test")

    workflow.add_job_shell_step("test", "echo Running tests...", name="Dummy Test Command")

    print("Added test step. The deployment steps will now run after the tests pass")


def add_deployment_job(workflow):
    target = forms.ask_deployment_target()
    job_id = f"deploy_to_{target}"

    workflow.add_job(job_id)

    # get repo and trigger info
    gh_owner, gh_repo = forms.ask_github_repo_name()
    gh_branch = None
    trigger = forms.ask_deployment_trigger()

    # set the job condition, based on the trigger
    if trigger == "push":
        gh_branch = forms.ask_github_branch_name(help_text="will react to pushes on this branch")
        workflow.add_trigger_push(gh_branch)
        workflow.set_job_field(job_id, "if", f"github.ref == 'refs/heads/{gh_branch}'")
    elif trigger == "release":
        workflow.add_trigger_release(types=["created"])
        workflow.set_job_field(job_id, "if", "github.event_name == 'release' && github.event.action == 'created'")

    # get the artifacts from build or checkout
    if workflow.has_job("build"):
        workflow.add_download_artifact_step(job_id, path="build")
    else:
        workflow.add_checkout_step(job_id)

    workflow.add_id_token_write_permission(job_id)

    # add the remaining target-specific deployment steps
    if target.startswith("aws_"):
        aws_account_id = aws.get_account_id()

        if target == "aws_s3":
            add_s3_deploy_job(workflow, job_id, aws_account_id, gh_owner, gh_repo, gh_branch)
        elif target == "aws_lambda":
            add_lambda_deploy_job(workflow, job_id, aws_account_id, gh_owner, gh_repo, gh_branch)
    elif target == "pypi":
        add_pypi_deploy_job(workflow, job_id)

    print(f"Added deployment step: {job_id}")


# --- Deploy job helpers ---
def add_s3_deploy_job(workflow, job_id, aws_account_id, gh_owner, gh_repo, gh_branch):
    ROLE_ENV_VAR = "S3_DEPLOY_ROLE"

    upload_format = forms.ask_upload_bundle_format()
    s3_path = forms.ask_aws_s3_path(is_file=upload_format == "zip")
    role_arn = aws.create_policy_and_role_for_github_to_s3_deploy(
        aws_account_id, s3_path, gh_owner, gh_repo, gh_branch, upload_format
    )

    aws.add_workflow_fetch_aws_credentials_step(workflow, job_id, role_env_var=ROLE_ENV_VAR)

    if upload_format == "zip":
        cmd.add_workflow_zip_step(workflow, job_id, zip_name="deploy.zip")
        aws.add_workflow_s3_cp_step(workflow, job_id, "deploy.zip", s3_path, acl="public-read")
    elif upload_format == "copy_all_files":
        aws.add_workflow_s3_sync_step(workflow, job_id, ".", s3_path)

    print("")
    print(
        f"**IMPORTANT:** Please ensure that you set the {ROLE_ENV_VAR} environment variable (in your GitHub repository) to {role_arn}"
    )


def add_lambda_deploy_job(workflow, job_id, aws_account_id, gh_owner, gh_repo, gh_branch):
    ROLE_ENV_VAR = "LAMBDA_DEPLOY_ROLE"

    function_name = forms.ask_aws_lambda_function_name()
    role_arn = aws.create_policy_and_role_for_github_to_lambda_deploy(
        aws_account_id, function_name, gh_owner, gh_repo, gh_branch
    )
    aws.add_workflow_fetch_aws_credentials_step(workflow, job_id, role_env_var=ROLE_ENV_VAR)
    cmd.add_workflow_zip_step(workflow, job_id, zip_name="function.zip")
    aws.add_workflow_lambda_deploy_step(workflow, job_id, function_name, "function.zip")

    print("")
    print(
        f"**IMPORTANT:** Please ensure that you set the {ROLE_ENV_VAR} environment variable (in your GitHub repository) to {role_arn}"
    )


def add_pypi_deploy_job(workflow, job_id):
    package_name = cmd.get_package_name_from_pyproject()
    print(f"Setting up PyPI publish for package: {package_name}")

    pypi.add_setup_python_step(workflow, job_id)
    pypi.add_install_dependencies_step(workflow, job_id)
    pypi.add_check_pypi_version_step(workflow, job_id, package_name)
    pypi.add_build_package_step(workflow, job_id)
    pypi.add_publish_to_pypi_step(workflow, job_id)

    print("")
    print(
        "**IMPORTANT:** Please ensure that you've added GitHub as a trusted publisher in your PyPI account: https://docs.pypi.org/trusted-publishers/"
    )
    print(f"Note: You can use the workflow file name ({workflow.file_name}) while configuring the trusted publisher.")


def ensure_job_order(workflow):
    """
    Reorders jobs in the workflow so that build comes first, then test, then all deploy jobs, then others.
    Does NOT modify dependencies.
    """
    jobs = workflow.get_jobs_ids()
    build_jobs = [j for j in jobs if j == "build"]
    test_jobs = [j for j in jobs if j == "test"]
    deploy_jobs = [j for j in jobs if j.startswith("deploy_to_")]
    other_jobs = [j for j in jobs if j not in build_jobs + test_jobs + deploy_jobs]

    ordered = build_jobs + test_jobs + deploy_jobs + other_jobs
    workflow.reorder_jobs(ordered)


def update_job_dependencies(workflow):
    # loop through all the jobs, and update their 'needs' based on the workflow
    has_build = workflow.has_job("build")
    has_test = workflow.has_job("test")

    for job_id in workflow.get_jobs_ids():
        if job_id.startswith("deploy_to_"):
            if has_test:
                workflow.set_job_field(job_id, "needs", "test")
            elif has_build:
                workflow.set_job_field(job_id, "needs", "build")
        elif job_id == "test":
            if has_build:
                workflow.set_job_field(job_id, "needs", "build")


def set_correct_checkout_step(workflow):
    if not workflow.has_job("build"):
        return

    for job_id in workflow.get_jobs_ids():
        if job_id != "build":
            workflow.replace_checkout_step_with_download_artifact(job_id)


if __name__ == "__main__":
    main()
