"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
const create_training_job_1 = require("../../lib/sagemaker/create-training-job");
/*
 * Creates a state machine with a task state to create a training job in AWS SageMaker
 * SageMaker jobs need training algorithms. These can be found in the AWS marketplace
 * or created.
 *
 * Subscribe to demo Algorithm vended by Amazon (free):
 * https://aws.amazon.com/marketplace/ai/procurement?productId=cc5186a0-b8d6-4750-a9bb-1dcdf10e787a
 * FIXME - create Input data pertinent for the training model and insert into S3 location specified in inputDataConfig.
 *
 * Stack verification steps:
 * The generated State Machine can be executed from the CLI (or Step Functions console)
 * and runs with an execution status of `Succeeded`.
 *
 * -- aws stepfunctions start-execution --state-machine-arn <state-machine-arn-from-output> provides execution arn
 * -- aws stepfunctions describe-execution --execution-arn <state-machine-arn-from-output> returns a status of `Succeeded`
 */
const app = new core_1.App();
const stack = new core_1.Stack(app, 'integ-stepfunctions-sagemaker');
const encryptionKey = new aws_kms_1.Key(stack, 'EncryptionKey', {
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const trainingData = new aws_s3_1.Bucket(stack, 'TrainingData', {
    encryption: aws_s3_1.BucketEncryption.KMS,
    encryptionKey,
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const sm = new aws_stepfunctions_1.StateMachine(stack, 'StateMachine', {
    definition: new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainTask', {
        algorithmSpecification: {
            algorithmName: 'arn:aws:sagemaker:us-east-1:865070037744:algorithm/scikit-decision-trees-15423055-57b73412d2e93e9239e4e16f83298b8f',
        },
        inputDataConfig: [{
                channelName: 'InputData',
                dataSource: {
                    s3DataSource: {
                        s3Location: lib_1.S3Location.fromBucket(trainingData, 'data/'),
                    },
                },
            }],
        outputDataConfig: { s3OutputLocation: lib_1.S3Location.fromBucket(trainingData, 'result/') },
        trainingJobName: 'mytrainingjob',
    }),
});
new core_1.CfnOutput(stack, 'stateMachineArn', {
    value: sm.stateMachineArn,
});
//# sourceMappingURL=data:application/json;base64,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