"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
const create_training_job_1 = require("../../lib/sagemaker/create-training-job");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('create basic training job', () => {
    // WHEN
    const task = new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainSagemaker', {
        trainingJobName: 'MyTrainJob',
        algorithmSpecification: {
            algorithmName: 'BlazingText',
        },
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            AlgorithmSpecification: {
                AlgorithmName: 'BlazingText',
                TrainingInputMode: 'File',
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']],
                            },
                        },
                    },
                },
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.m4.xlarge',
                VolumeSizeInGB: 10,
            },
            RoleArn: { 'Fn::GetAtt': ['TrainSagemakerSagemakerRole89E8C593', 'Arn'] },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600,
            },
            TrainingJobName: 'MyTrainJob',
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainSagemaker', {
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            trainingJobName: 'MyTrainJob',
            algorithmSpecification: {
                algorithmName: 'BlazingText',
            },
            inputDataConfig: [
                {
                    channelName: 'train',
                    dataSource: {
                        s3DataSource: {
                            s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucket', 'mybucket'), 'mytrainpath'),
                        },
                    },
                },
            ],
            outputDataConfig: {
                s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
            },
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/i);
});
test('create complex training job', () => {
    // WHEN
    const kmsKey = new kms.Key(stack, 'Key');
    const vpc = new ec2.Vpc(stack, 'VPC');
    const securityGroup = new ec2.SecurityGroup(stack, 'SecurityGroup', { vpc, description: 'My SG' });
    securityGroup.addIngressRule(ec2.Peer.anyIpv4(), ec2.Port.tcp(22), 'allow ssh access from the world');
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const trainTask = new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainSagemaker', {
        trainingJobName: 'MyTrainJob',
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        role,
        algorithmSpecification: {
            algorithmName: 'BlazingText',
            trainingInputMode: tasks.InputMode.FILE,
            metricDefinitions: [
                {
                    name: 'mymetric', regex: 'regex_pattern',
                },
            ],
        },
        hyperparameters: {
            lr: '0.1',
        },
        inputDataConfig: [
            {
                channelName: 'train',
                contentType: 'image/jpeg',
                compressionType: tasks.CompressionType.NONE,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        attributeNames: ['source-ref', 'class'],
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketA', 'mybucket'), 'mytrainpath'),
                    },
                },
            },
            {
                channelName: 'test',
                contentType: 'image/jpeg',
                compressionType: tasks.CompressionType.GZIP,
                recordWrapperType: tasks.RecordWrapperType.RECORD_IO,
                dataSource: {
                    s3DataSource: {
                        attributeNames: ['source-ref', 'class'],
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'InputBucketB', 'mybucket'), 'mytestpath'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'OutputBucket', 'mybucket'), 'myoutputpath'),
            encryptionKey: kmsKey,
        },
        resourceConfig: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            volumeSize: cdk.Size.gibibytes(50),
            volumeEncryptionKey: kmsKey,
        },
        stoppingCondition: {
            maxRuntime: cdk.Duration.hours(1),
        },
        tags: {
            Project: 'MyProject',
        },
        vpcConfig: {
            vpc,
        },
    });
    trainTask.addSecurityGroup(securityGroup);
    // THEN
    expect(stack.resolve(trainTask.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            TrainingJobName: 'MyTrainJob',
            RoleArn: { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            AlgorithmSpecification: {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
                MetricDefinitions: [
                    { Name: 'mymetric', Regex: 'regex_pattern' },
                ],
            },
            HyperParameters: {
                lr: '0.1',
            },
            InputDataConfig: [
                {
                    ChannelName: 'train',
                    CompressionType: 'None',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            AttributeNames: ['source-ref', 'class'],
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytrainpath']],
                            },
                        },
                    },
                },
                {
                    ChannelName: 'test',
                    CompressionType: 'Gzip',
                    RecordWrapperType: 'RecordIO',
                    ContentType: 'image/jpeg',
                    DataSource: {
                        S3DataSource: {
                            AttributeNames: ['source-ref', 'class'],
                            S3DataType: 'S3Prefix',
                            S3Uri: {
                                'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/mytestpath']],
                            },
                        },
                    },
                },
            ],
            OutputDataConfig: {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
                KmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            ResourceConfig: {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50,
                VolumeKmsKeyId: { 'Fn::GetAtt': ['Key961B73FD', 'Arn'] },
            },
            StoppingCondition: {
                MaxRuntimeInSeconds: 3600,
            },
            Tags: [
                { Key: 'Project', Value: 'MyProject' },
            ],
            VpcConfig: {
                SecurityGroupIds: [
                    { 'Fn::GetAtt': ['TrainSagemakerTrainJobSecurityGroup7C858EB9', 'GroupId'] },
                    { 'Fn::GetAtt': ['SecurityGroupDD263621', 'GroupId'] },
                ],
                Subnets: [
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ],
            },
        },
    });
});
test('pass param to training job', () => {
    // WHEN
    const role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
        managedPolicies: [
            iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess'),
        ],
    });
    const task = new create_training_job_1.SageMakerCreateTrainingJob(stack, 'TrainSagemaker', {
        trainingJobName: sfn.JsonPath.stringAt('$.JobName'),
        role,
        algorithmSpecification: {
            algorithmName: 'BlazingText',
            trainingInputMode: tasks.InputMode.FILE,
        },
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath'),
        },
        resourceConfig: {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.P3, ec2.InstanceSize.XLARGE2),
            volumeSize: cdk.Size.gibibytes(50),
        },
        stoppingCondition: {
            maxRuntime: cdk.Duration.hours(1),
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createTrainingJob',
                ],
            ],
        },
        End: true,
        Parameters: {
            'TrainingJobName.$': '$.JobName',
            'RoleArn': { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
            'AlgorithmSpecification': {
                TrainingInputMode: 'File',
                AlgorithmName: 'BlazingText',
            },
            'InputDataConfig': [
                {
                    ChannelName: 'train',
                    DataSource: {
                        S3DataSource: {
                            'S3DataType': 'S3Prefix',
                            'S3Uri.$': '$.S3Bucket',
                        },
                    },
                },
            ],
            'OutputDataConfig': {
                S3OutputPath: {
                    'Fn::Join': ['', ['https://s3.', { Ref: 'AWS::Region' }, '.', { Ref: 'AWS::URLSuffix' }, '/mybucket/myoutputpath']],
                },
            },
            'ResourceConfig': {
                InstanceCount: 1,
                InstanceType: 'ml.p3.2xlarge',
                VolumeSizeInGB: 50,
            },
            'StoppingCondition': {
                MaxRuntimeInSeconds: 3600,
            },
        },
    });
});
test('Cannot create a SageMaker train task with both algorithm name and image name missing', () => {
    expect(() => new create_training_job_1.SageMakerCreateTrainingJob(stack, 'SageMakerTrainingTask', {
        trainingJobName: 'myTrainJob',
        algorithmSpecification: {},
        inputDataConfig: [
            {
                channelName: 'train',
                dataSource: {
                    s3DataSource: {
                        s3DataType: tasks.S3DataType.S3_PREFIX,
                        s3Location: tasks.S3Location.fromJsonExpression('$.S3Bucket'),
                    },
                },
            },
        ],
        outputDataConfig: {
            s3OutputLocation: tasks.S3Location.fromBucket(s3.Bucket.fromBucketName(stack, 'Bucket', 'mybucket'), 'myoutputpath/'),
        },
    }))
        .toThrowError(/Must define either an algorithm name or training image URI in the algorithm specification/);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlLXRyYWluaW5nLWpvYi50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY3JlYXRlLXRyYWluaW5nLWpvYi50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsZ0NBQThCO0FBQzlCLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBQ3hDLHNDQUFzQztBQUN0QyxrREFBa0Q7QUFDbEQscUNBQXFDO0FBQ3JDLG1DQUFtQztBQUNuQyxpRkFBcUY7QUFFckYsSUFBSSxLQUFnQixDQUFDO0FBRXJCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7SUFDZCxRQUFRO0lBQ1IsS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0FBQzFCLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJCQUEyQixFQUFFLEdBQUcsRUFBRTtJQUNyQyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxnREFBMEIsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7UUFDbkUsZUFBZSxFQUFFLFlBQVk7UUFDN0Isc0JBQXNCLEVBQUU7WUFDdEIsYUFBYSxFQUFFLGFBQWE7U0FDN0I7UUFDRCxlQUFlLEVBQUU7WUFDZjtnQkFDRSxXQUFXLEVBQUUsT0FBTztnQkFDcEIsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRTt3QkFDWixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxVQUFVLENBQUMsRUFBRSxhQUFhLENBQUM7cUJBQ25IO2lCQUNGO2FBQ0Y7U0FDRjtRQUNELGdCQUFnQixFQUFFO1lBQ2hCLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsVUFBVSxDQUFDLEVBQUUsY0FBYyxDQUFDO1NBQzNIO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ2hELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCx1Q0FBdUM7aUJBQ3hDO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1Ysc0JBQXNCLEVBQUU7Z0JBQ3RCLGFBQWEsRUFBRSxhQUFhO2dCQUM1QixpQkFBaUIsRUFBRSxNQUFNO2FBQzFCO1lBQ0QsZUFBZSxFQUFFO2dCQUNmO29CQUNFLFdBQVcsRUFBRSxPQUFPO29CQUNwQixVQUFVLEVBQUU7d0JBQ1YsWUFBWSxFQUFFOzRCQUNaLFVBQVUsRUFBRSxVQUFVOzRCQUN0QixLQUFLLEVBQUU7Z0NBQ0wsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsYUFBYSxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLHVCQUF1QixDQUFDLENBQUM7NkJBQ25IO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsWUFBWSxFQUFFO29CQUNaLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO2lCQUNwSDthQUNGO1lBQ0QsY0FBYyxFQUFFO2dCQUNkLGFBQWEsRUFBRSxDQUFDO2dCQUNoQixZQUFZLEVBQUUsY0FBYztnQkFDNUIsY0FBYyxFQUFFLEVBQUU7YUFDbkI7WUFDRCxPQUFPLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxxQ0FBcUMsRUFBRSxLQUFLLENBQUMsRUFBRTtZQUN6RSxpQkFBaUIsRUFBRTtnQkFDakIsbUJBQW1CLEVBQUUsSUFBSTthQUMxQjtZQUNELGVBQWUsRUFBRSxZQUFZO1NBQzlCO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsK0VBQStFLEVBQUUsR0FBRyxFQUFFO0lBQ3pGLE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixJQUFJLGdEQUEwQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtZQUN0RCxrQkFBa0IsRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CO1lBQzlELGVBQWUsRUFBRSxZQUFZO1lBQzdCLHNCQUFzQixFQUFFO2dCQUN0QixhQUFhLEVBQUUsYUFBYTthQUM3QjtZQUNELGVBQWUsRUFBRTtnQkFDZjtvQkFDRSxXQUFXLEVBQUUsT0FBTztvQkFDcEIsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRTs0QkFDWixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxVQUFVLENBQUMsRUFBRSxhQUFhLENBQUM7eUJBQ25IO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxjQUFjLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxVQUFVLENBQUMsRUFBRSxjQUFjLENBQUM7YUFDM0g7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsdUhBQXVILENBQUMsQ0FBQztBQUN0SSxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2QkFBNkIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsT0FBTztJQUNQLE1BQU0sTUFBTSxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDekMsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsR0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN0QyxNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRSxFQUFFLEdBQUcsRUFBRSxXQUFXLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztJQUNuRyxhQUFhLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEVBQUUsaUNBQWlDLENBQUMsQ0FBQztJQUV0RyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMseUJBQXlCLENBQUM7UUFDOUQsZUFBZSxFQUFFO1lBQ2YsR0FBRyxDQUFDLGFBQWEsQ0FBQyx3QkFBd0IsQ0FBQywyQkFBMkIsQ0FBQztTQUN4RTtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sU0FBUyxHQUFHLElBQUksZ0RBQTBCLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO1FBQ3hFLGVBQWUsRUFBRSxZQUFZO1FBQzdCLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPO1FBQ2xELElBQUk7UUFDSixzQkFBc0IsRUFBRTtZQUN0QixhQUFhLEVBQUUsYUFBYTtZQUM1QixpQkFBaUIsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLElBQUk7WUFDdkMsaUJBQWlCLEVBQUU7Z0JBQ2pCO29CQUNFLElBQUksRUFBRSxVQUFVLEVBQUUsS0FBSyxFQUFFLGVBQWU7aUJBQ3pDO2FBQ0Y7U0FDRjtRQUNELGVBQWUsRUFBRTtZQUNmLEVBQUUsRUFBRSxLQUFLO1NBQ1Y7UUFDRCxlQUFlLEVBQUU7WUFDZjtnQkFDRSxXQUFXLEVBQUUsT0FBTztnQkFDcEIsV0FBVyxFQUFFLFlBQVk7Z0JBQ3pCLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZSxDQUFDLElBQUk7Z0JBQzNDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTO2dCQUNwRCxVQUFVLEVBQUU7b0JBQ1YsWUFBWSxFQUFFO3dCQUNaLGNBQWMsRUFBRSxDQUFDLFlBQVksRUFBRSxPQUFPLENBQUM7d0JBQ3ZDLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLFNBQVM7d0JBQ3RDLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLFVBQVUsQ0FBQyxFQUFFLGFBQWEsQ0FBQztxQkFDcEg7aUJBQ0Y7YUFDRjtZQUNEO2dCQUNFLFdBQVcsRUFBRSxNQUFNO2dCQUNuQixXQUFXLEVBQUUsWUFBWTtnQkFDekIsZUFBZSxFQUFFLEtBQUssQ0FBQyxlQUFlLENBQUMsSUFBSTtnQkFDM0MsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixDQUFDLFNBQVM7Z0JBQ3BELFVBQVUsRUFBRTtvQkFDVixZQUFZLEVBQUU7d0JBQ1osY0FBYyxFQUFFLENBQUMsWUFBWSxFQUFFLE9BQU8sQ0FBQzt3QkFDdkMsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUzt3QkFDdEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsVUFBVSxDQUFDLEVBQUUsWUFBWSxDQUFDO3FCQUNuSDtpQkFDRjthQUNGO1NBQ0Y7UUFDRCxnQkFBZ0IsRUFBRTtZQUNoQixnQkFBZ0IsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFLFVBQVUsQ0FBQyxFQUFFLGNBQWMsQ0FBQztZQUMxSCxhQUFhLEVBQUUsTUFBTTtTQUN0QjtRQUNELGNBQWMsRUFBRTtZQUNkLGFBQWEsRUFBRSxDQUFDO1lBQ2hCLFlBQVksRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQztZQUNqRixVQUFVLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDO1lBQ2xDLG1CQUFtQixFQUFFLE1BQU07U0FDNUI7UUFDRCxpQkFBaUIsRUFBRTtZQUNqQixVQUFVLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1NBQ2xDO1FBQ0QsSUFBSSxFQUFFO1lBQ0osT0FBTyxFQUFFLFdBQVc7U0FDckI7UUFDRCxTQUFTLEVBQUU7WUFDVCxHQUFHO1NBQ0o7S0FDRixDQUFDLENBQUM7SUFDSCxTQUFTLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDLENBQUM7SUFFMUMsT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ3JELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCw0Q0FBNEM7aUJBQzdDO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1YsZUFBZSxFQUFFLFlBQVk7WUFDN0IsT0FBTyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsY0FBYyxFQUFFLEtBQUssQ0FBQyxFQUFFO1lBQ2xELHNCQUFzQixFQUFFO2dCQUN0QixpQkFBaUIsRUFBRSxNQUFNO2dCQUN6QixhQUFhLEVBQUUsYUFBYTtnQkFDNUIsaUJBQWlCLEVBQUU7b0JBQ2pCLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUUsZUFBZSxFQUFFO2lCQUM3QzthQUNGO1lBQ0QsZUFBZSxFQUFFO2dCQUNmLEVBQUUsRUFBRSxLQUFLO2FBQ1Y7WUFDRCxlQUFlLEVBQUU7Z0JBQ2Y7b0JBQ0UsV0FBVyxFQUFFLE9BQU87b0JBQ3BCLGVBQWUsRUFBRSxNQUFNO29CQUN2QixpQkFBaUIsRUFBRSxVQUFVO29CQUM3QixXQUFXLEVBQUUsWUFBWTtvQkFDekIsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRTs0QkFDWixjQUFjLEVBQUUsQ0FBQyxZQUFZLEVBQUUsT0FBTyxDQUFDOzRCQUN2QyxVQUFVLEVBQUUsVUFBVTs0QkFDdEIsS0FBSyxFQUFFO2dDQUNMLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSx1QkFBdUIsQ0FBQyxDQUFDOzZCQUNuSDt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRDtvQkFDRSxXQUFXLEVBQUUsTUFBTTtvQkFDbkIsZUFBZSxFQUFFLE1BQU07b0JBQ3ZCLGlCQUFpQixFQUFFLFVBQVU7b0JBQzdCLFdBQVcsRUFBRSxZQUFZO29CQUN6QixVQUFVLEVBQUU7d0JBQ1YsWUFBWSxFQUFFOzRCQUNaLGNBQWMsRUFBRSxDQUFDLFlBQVksRUFBRSxPQUFPLENBQUM7NEJBQ3ZDLFVBQVUsRUFBRSxVQUFVOzRCQUN0QixLQUFLLEVBQUU7Z0NBQ0wsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsYUFBYSxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLHNCQUFzQixDQUFDLENBQUM7NkJBQ2xIO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsWUFBWSxFQUFFO29CQUNaLFVBQVUsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxhQUFhLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxHQUFHLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO2lCQUNwSDtnQkFDRCxRQUFRLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxDQUFDLEVBQUU7YUFDbkQ7WUFDRCxjQUFjLEVBQUU7Z0JBQ2QsYUFBYSxFQUFFLENBQUM7Z0JBQ2hCLFlBQVksRUFBRSxlQUFlO2dCQUM3QixjQUFjLEVBQUUsRUFBRTtnQkFDbEIsY0FBYyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssQ0FBQyxFQUFFO2FBQ3pEO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCLG1CQUFtQixFQUFFLElBQUk7YUFDMUI7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osRUFBRSxHQUFHLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxXQUFXLEVBQUU7YUFDdkM7WUFDRCxTQUFTLEVBQUU7Z0JBQ1QsZ0JBQWdCLEVBQUU7b0JBQ2hCLEVBQUUsWUFBWSxFQUFFLENBQUMsNkNBQTZDLEVBQUUsU0FBUyxDQUFDLEVBQUU7b0JBQzVFLEVBQUUsWUFBWSxFQUFFLENBQUMsdUJBQXVCLEVBQUUsU0FBUyxDQUFDLEVBQUU7aUJBQ3ZEO2dCQUNELE9BQU8sRUFBRTtvQkFDUCxFQUFFLEdBQUcsRUFBRSxpQ0FBaUMsRUFBRTtvQkFDMUMsRUFBRSxHQUFHLEVBQUUsaUNBQWlDLEVBQUU7aUJBQzNDO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDRCQUE0QixFQUFFLEdBQUcsRUFBRTtJQUN0QyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7UUFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO1FBQzlELGVBQWUsRUFBRTtZQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsMkJBQTJCLENBQUM7U0FDeEU7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLElBQUksR0FBRyxJQUFJLGdEQUEwQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUNuRSxlQUFlLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDO1FBQ25ELElBQUk7UUFDSixzQkFBc0IsRUFBRTtZQUN0QixhQUFhLEVBQUUsYUFBYTtZQUM1QixpQkFBaUIsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLElBQUk7U0FDeEM7UUFDRCxlQUFlLEVBQUU7WUFDZjtnQkFDRSxXQUFXLEVBQUUsT0FBTztnQkFDcEIsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRTt3QkFDWixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxTQUFTO3dCQUN0QyxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxZQUFZLENBQUM7cUJBQzlEO2lCQUNGO2FBQ0Y7U0FDRjtRQUNELGdCQUFnQixFQUFFO1lBQ2hCLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsVUFBVSxDQUFDLEVBQUUsY0FBYyxDQUFDO1NBQ3JIO1FBQ0QsY0FBYyxFQUFFO1lBQ2QsYUFBYSxFQUFFLENBQUM7WUFDaEIsWUFBWSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDO1lBQ2pGLFVBQVUsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUM7U0FDbkM7UUFDRCxpQkFBaUIsRUFBRTtZQUNqQixVQUFVLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1NBQ2xDO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsT0FBTztJQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ2hELElBQUksRUFBRSxNQUFNO1FBQ1osUUFBUSxFQUFFO1lBQ1IsVUFBVSxFQUFFO2dCQUNWLEVBQUU7Z0JBQ0Y7b0JBQ0UsTUFBTTtvQkFDTjt3QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3FCQUN0QjtvQkFDRCx1Q0FBdUM7aUJBQ3hDO2FBQ0Y7U0FDRjtRQUNELEdBQUcsRUFBRSxJQUFJO1FBQ1QsVUFBVSxFQUFFO1lBQ1YsbUJBQW1CLEVBQUUsV0FBVztZQUNoQyxTQUFTLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxjQUFjLEVBQUUsS0FBSyxDQUFDLEVBQUU7WUFDcEQsd0JBQXdCLEVBQUU7Z0JBQ3hCLGlCQUFpQixFQUFFLE1BQU07Z0JBQ3pCLGFBQWEsRUFBRSxhQUFhO2FBQzdCO1lBQ0QsaUJBQWlCLEVBQUU7Z0JBQ2pCO29CQUNFLFdBQVcsRUFBRSxPQUFPO29CQUNwQixVQUFVLEVBQUU7d0JBQ1YsWUFBWSxFQUFFOzRCQUNaLFlBQVksRUFBRSxVQUFVOzRCQUN4QixTQUFTLEVBQUUsWUFBWTt5QkFDeEI7cUJBQ0Y7aUJBQ0Y7YUFDRjtZQUNELGtCQUFrQixFQUFFO2dCQUNsQixZQUFZLEVBQUU7b0JBQ1osVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsYUFBYSxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLHdCQUF3QixDQUFDLENBQUM7aUJBQ3BIO2FBQ0Y7WUFDRCxnQkFBZ0IsRUFBRTtnQkFDaEIsYUFBYSxFQUFFLENBQUM7Z0JBQ2hCLFlBQVksRUFBRSxlQUFlO2dCQUM3QixjQUFjLEVBQUUsRUFBRTthQUNuQjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQixtQkFBbUIsRUFBRSxJQUFJO2FBQzFCO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzRkFBc0YsRUFBRSxHQUFHLEVBQUU7SUFFaEcsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksZ0RBQTBCLENBQUMsS0FBSyxFQUFFLHVCQUF1QixFQUFFO1FBQzFFLGVBQWUsRUFBRSxZQUFZO1FBQzdCLHNCQUFzQixFQUFFLEVBQUU7UUFDMUIsZUFBZSxFQUFFO1lBQ2Y7Z0JBQ0UsV0FBVyxFQUFFLE9BQU87Z0JBQ3BCLFVBQVUsRUFBRTtvQkFDVixZQUFZLEVBQUU7d0JBQ1osVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUzt3QkFDdEMsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsa0JBQWtCLENBQUMsWUFBWSxDQUFDO3FCQUM5RDtpQkFDRjthQUNGO1NBQ0Y7UUFDRCxnQkFBZ0IsRUFBRTtZQUNoQixnQkFBZ0IsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLFVBQVUsQ0FBQyxFQUFFLGVBQWUsQ0FBQztTQUN0SDtLQUNGLENBQUMsQ0FBQztTQUNBLFlBQVksQ0FBQywyRkFBMkYsQ0FBQyxDQUFDO0FBQy9HLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSAnQGF3cy1jZGsvYXdzLWttcyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIHRhc2tzIGZyb20gJy4uLy4uL2xpYic7XG5pbXBvcnQgeyBTYWdlTWFrZXJDcmVhdGVUcmFpbmluZ0pvYiB9IGZyb20gJy4uLy4uL2xpYi9zYWdlbWFrZXIvY3JlYXRlLXRyYWluaW5nLWpvYic7XG5cbmxldCBzdGFjazogY2RrLlN0YWNrO1xuXG5iZWZvcmVFYWNoKCgpID0+IHtcbiAgLy8gR0lWRU5cbiAgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG59KTtcblxudGVzdCgnY3JlYXRlIGJhc2ljIHRyYWluaW5nIGpvYicsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IFNhZ2VNYWtlckNyZWF0ZVRyYWluaW5nSm9iKHN0YWNrLCAnVHJhaW5TYWdlbWFrZXInLCB7XG4gICAgdHJhaW5pbmdKb2JOYW1lOiAnTXlUcmFpbkpvYicsXG4gICAgYWxnb3JpdGhtU3BlY2lmaWNhdGlvbjoge1xuICAgICAgYWxnb3JpdGhtTmFtZTogJ0JsYXppbmdUZXh0JyxcbiAgICB9LFxuICAgIGlucHV0RGF0YUNvbmZpZzogW1xuICAgICAge1xuICAgICAgICBjaGFubmVsTmFtZTogJ3RyYWluJyxcbiAgICAgICAgZGF0YVNvdXJjZToge1xuICAgICAgICAgIHMzRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgczNMb2NhdGlvbjogdGFza3MuUzNMb2NhdGlvbi5mcm9tQnVja2V0KHMzLkJ1Y2tldC5mcm9tQnVja2V0TmFtZShzdGFjaywgJ0lucHV0QnVja2V0JywgJ215YnVja2V0JyksICdteXRyYWlucGF0aCcpLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIF0sXG4gICAgb3V0cHV0RGF0YUNvbmZpZzoge1xuICAgICAgczNPdXRwdXRMb2NhdGlvbjogdGFza3MuUzNMb2NhdGlvbi5mcm9tQnVja2V0KHMzLkJ1Y2tldC5mcm9tQnVja2V0TmFtZShzdGFjaywgJ091dHB1dEJ1Y2tldCcsICdteWJ1Y2tldCcpLCAnbXlvdXRwdXRwYXRoJyksXG4gICAgfSxcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICBUeXBlOiAnVGFzaycsXG4gICAgUmVzb3VyY2U6IHtcbiAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgJycsXG4gICAgICAgIFtcbiAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAge1xuICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgJzpzdGF0ZXM6OjpzYWdlbWFrZXI6Y3JlYXRlVHJhaW5pbmdKb2InLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBBbGdvcml0aG1TcGVjaWZpY2F0aW9uOiB7XG4gICAgICAgIEFsZ29yaXRobU5hbWU6ICdCbGF6aW5nVGV4dCcsXG4gICAgICAgIFRyYWluaW5nSW5wdXRNb2RlOiAnRmlsZScsXG4gICAgICB9LFxuICAgICAgSW5wdXREYXRhQ29uZmlnOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBDaGFubmVsTmFtZTogJ3RyYWluJyxcbiAgICAgICAgICBEYXRhU291cmNlOiB7XG4gICAgICAgICAgICBTM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgUzNEYXRhVHlwZTogJ1MzUHJlZml4JyxcbiAgICAgICAgICAgICAgUzNVcmk6IHtcbiAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbJycsIFsnaHR0cHM6Ly9zMy4nLCB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LCAnLicsIHsgUmVmOiAnQVdTOjpVUkxTdWZmaXgnIH0sICcvbXlidWNrZXQvbXl0cmFpbnBhdGgnXV0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgT3V0cHV0RGF0YUNvbmZpZzoge1xuICAgICAgICBTM091dHB1dFBhdGg6IHtcbiAgICAgICAgICAnRm46OkpvaW4nOiBbJycsIFsnaHR0cHM6Ly9zMy4nLCB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LCAnLicsIHsgUmVmOiAnQVdTOjpVUkxTdWZmaXgnIH0sICcvbXlidWNrZXQvbXlvdXRwdXRwYXRoJ11dLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIFJlc291cmNlQ29uZmlnOiB7XG4gICAgICAgIEluc3RhbmNlQ291bnQ6IDEsXG4gICAgICAgIEluc3RhbmNlVHlwZTogJ21sLm00LnhsYXJnZScsXG4gICAgICAgIFZvbHVtZVNpemVJbkdCOiAxMCxcbiAgICAgIH0sXG4gICAgICBSb2xlQXJuOiB7ICdGbjo6R2V0QXR0JzogWydUcmFpblNhZ2VtYWtlclNhZ2VtYWtlclJvbGU4OUU4QzU5MycsICdBcm4nXSB9LFxuICAgICAgU3RvcHBpbmdDb25kaXRpb246IHtcbiAgICAgICAgTWF4UnVudGltZUluU2Vjb25kczogMzYwMCxcbiAgICAgIH0sXG4gICAgICBUcmFpbmluZ0pvYk5hbWU6ICdNeVRyYWluSm9iJyxcbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdUYXNrIHRocm93cyBpZiBXQUlUX0ZPUl9UQVNLX1RPS0VOIGlzIHN1cHBsaWVkIGFzIHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybicsICgpID0+IHtcbiAgZXhwZWN0KCgpID0+IHtcbiAgICBuZXcgU2FnZU1ha2VyQ3JlYXRlVHJhaW5pbmdKb2Ioc3RhY2ssICdUcmFpblNhZ2VtYWtlcicsIHtcbiAgICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLkludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICAgICAgdHJhaW5pbmdKb2JOYW1lOiAnTXlUcmFpbkpvYicsXG4gICAgICBhbGdvcml0aG1TcGVjaWZpY2F0aW9uOiB7XG4gICAgICAgIGFsZ29yaXRobU5hbWU6ICdCbGF6aW5nVGV4dCcsXG4gICAgICB9LFxuICAgICAgaW5wdXREYXRhQ29uZmlnOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBjaGFubmVsTmFtZTogJ3RyYWluJyxcbiAgICAgICAgICBkYXRhU291cmNlOiB7XG4gICAgICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgczNMb2NhdGlvbjogdGFza3MuUzNMb2NhdGlvbi5mcm9tQnVja2V0KHMzLkJ1Y2tldC5mcm9tQnVja2V0TmFtZShzdGFjaywgJ0lucHV0QnVja2V0JywgJ215YnVja2V0JyksICdteXRyYWlucGF0aCcpLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIG91dHB1dERhdGFDb25maWc6IHtcbiAgICAgICAgczNPdXRwdXRMb2NhdGlvbjogdGFza3MuUzNMb2NhdGlvbi5mcm9tQnVja2V0KHMzLkJ1Y2tldC5mcm9tQnVja2V0TmFtZShzdGFjaywgJ091dHB1dEJ1Y2tldCcsICdteWJ1Y2tldCcpLCAnbXlvdXRwdXRwYXRoJyksXG4gICAgICB9LFxuICAgIH0pO1xuICB9KS50b1Rocm93KC9VbnN1cHBvcnRlZCBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4uIFN1cHBvcnRlZCBQYXR0ZXJuczogUkVRVUVTVF9SRVNQT05TRSxSVU5fSk9CLiBSZWNlaXZlZDogV0FJVF9GT1JfVEFTS19UT0tFTi9pKTtcbn0pO1xuXG50ZXN0KCdjcmVhdGUgY29tcGxleCB0cmFpbmluZyBqb2InLCAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3Qga21zS2V5ID0gbmV3IGttcy5LZXkoc3RhY2ssICdLZXknKTtcbiAgY29uc3QgdnBjID0gbmV3IGVjMi5WcGMoc3RhY2ssICdWUEMnKTtcbiAgY29uc3Qgc2VjdXJpdHlHcm91cCA9IG5ldyBlYzIuU2VjdXJpdHlHcm91cChzdGFjaywgJ1NlY3VyaXR5R3JvdXAnLCB7IHZwYywgZGVzY3JpcHRpb246ICdNeSBTRycgfSk7XG4gIHNlY3VyaXR5R3JvdXAuYWRkSW5ncmVzc1J1bGUoZWMyLlBlZXIuYW55SXB2NCgpLCBlYzIuUG9ydC50Y3AoMjIpLCAnYWxsb3cgc3NoIGFjY2VzcyBmcm9tIHRoZSB3b3JsZCcpO1xuXG4gIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TYWdlTWFrZXJGdWxsQWNjZXNzJyksXG4gICAgXSxcbiAgfSk7XG5cbiAgY29uc3QgdHJhaW5UYXNrID0gbmV3IFNhZ2VNYWtlckNyZWF0ZVRyYWluaW5nSm9iKHN0YWNrLCAnVHJhaW5TYWdlbWFrZXInLCB7XG4gICAgdHJhaW5pbmdKb2JOYW1lOiAnTXlUcmFpbkpvYicsXG4gICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0IsXG4gICAgcm9sZSxcbiAgICBhbGdvcml0aG1TcGVjaWZpY2F0aW9uOiB7XG4gICAgICBhbGdvcml0aG1OYW1lOiAnQmxhemluZ1RleHQnLFxuICAgICAgdHJhaW5pbmdJbnB1dE1vZGU6IHRhc2tzLklucHV0TW9kZS5GSUxFLFxuICAgICAgbWV0cmljRGVmaW5pdGlvbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIG5hbWU6ICdteW1ldHJpYycsIHJlZ2V4OiAncmVnZXhfcGF0dGVybicsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgaHlwZXJwYXJhbWV0ZXJzOiB7XG4gICAgICBscjogJzAuMScsXG4gICAgfSxcbiAgICBpbnB1dERhdGFDb25maWc6IFtcbiAgICAgIHtcbiAgICAgICAgY2hhbm5lbE5hbWU6ICd0cmFpbicsXG4gICAgICAgIGNvbnRlbnRUeXBlOiAnaW1hZ2UvanBlZycsXG4gICAgICAgIGNvbXByZXNzaW9uVHlwZTogdGFza3MuQ29tcHJlc3Npb25UeXBlLk5PTkUsXG4gICAgICAgIHJlY29yZFdyYXBwZXJUeXBlOiB0YXNrcy5SZWNvcmRXcmFwcGVyVHlwZS5SRUNPUkRfSU8sXG4gICAgICAgIGRhdGFTb3VyY2U6IHtcbiAgICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgIGF0dHJpYnV0ZU5hbWVzOiBbJ3NvdXJjZS1yZWYnLCAnY2xhc3MnXSxcbiAgICAgICAgICAgIHMzRGF0YVR5cGU6IHRhc2tzLlMzRGF0YVR5cGUuUzNfUFJFRklYLFxuICAgICAgICAgICAgczNMb2NhdGlvbjogdGFza3MuUzNMb2NhdGlvbi5mcm9tQnVja2V0KHMzLkJ1Y2tldC5mcm9tQnVja2V0TmFtZShzdGFjaywgJ0lucHV0QnVja2V0QScsICdteWJ1Y2tldCcpLCAnbXl0cmFpbnBhdGgnKSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgY2hhbm5lbE5hbWU6ICd0ZXN0JyxcbiAgICAgICAgY29udGVudFR5cGU6ICdpbWFnZS9qcGVnJyxcbiAgICAgICAgY29tcHJlc3Npb25UeXBlOiB0YXNrcy5Db21wcmVzc2lvblR5cGUuR1pJUCxcbiAgICAgICAgcmVjb3JkV3JhcHBlclR5cGU6IHRhc2tzLlJlY29yZFdyYXBwZXJUeXBlLlJFQ09SRF9JTyxcbiAgICAgICAgZGF0YVNvdXJjZToge1xuICAgICAgICAgIHMzRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgYXR0cmlidXRlTmFtZXM6IFsnc291cmNlLXJlZicsICdjbGFzcyddLFxuICAgICAgICAgICAgczNEYXRhVHlwZTogdGFza3MuUzNEYXRhVHlwZS5TM19QUkVGSVgsXG4gICAgICAgICAgICBzM0xvY2F0aW9uOiB0YXNrcy5TM0xvY2F0aW9uLmZyb21CdWNrZXQoczMuQnVja2V0LmZyb21CdWNrZXROYW1lKHN0YWNrLCAnSW5wdXRCdWNrZXRCJywgJ215YnVja2V0JyksICdteXRlc3RwYXRoJyksXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgXSxcbiAgICBvdXRwdXREYXRhQ29uZmlnOiB7XG4gICAgICBzM091dHB1dExvY2F0aW9uOiB0YXNrcy5TM0xvY2F0aW9uLmZyb21CdWNrZXQoczMuQnVja2V0LmZyb21CdWNrZXROYW1lKHN0YWNrLCAnT3V0cHV0QnVja2V0JywgJ215YnVja2V0JyksICdteW91dHB1dHBhdGgnKSxcbiAgICAgIGVuY3J5cHRpb25LZXk6IGttc0tleSxcbiAgICB9LFxuICAgIHJlc291cmNlQ29uZmlnOiB7XG4gICAgICBpbnN0YW5jZUNvdW50OiAxLFxuICAgICAgaW5zdGFuY2VUeXBlOiBlYzIuSW5zdGFuY2VUeXBlLm9mKGVjMi5JbnN0YW5jZUNsYXNzLlAzLCBlYzIuSW5zdGFuY2VTaXplLlhMQVJHRTIpLFxuICAgICAgdm9sdW1lU2l6ZTogY2RrLlNpemUuZ2liaWJ5dGVzKDUwKSxcbiAgICAgIHZvbHVtZUVuY3J5cHRpb25LZXk6IGttc0tleSxcbiAgICB9LFxuICAgIHN0b3BwaW5nQ29uZGl0aW9uOiB7XG4gICAgICBtYXhSdW50aW1lOiBjZGsuRHVyYXRpb24uaG91cnMoMSksXG4gICAgfSxcbiAgICB0YWdzOiB7XG4gICAgICBQcm9qZWN0OiAnTXlQcm9qZWN0JyxcbiAgICB9LFxuICAgIHZwY0NvbmZpZzoge1xuICAgICAgdnBjLFxuICAgIH0sXG4gIH0pO1xuICB0cmFpblRhc2suYWRkU2VjdXJpdHlHcm91cChzZWN1cml0eUdyb3VwKTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRyYWluVGFzay50b1N0YXRlSnNvbigpKSkudG9FcXVhbCh7XG4gICAgVHlwZTogJ1Rhc2snLFxuICAgIFJlc291cmNlOiB7XG4gICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICcnLFxuICAgICAgICBbXG4gICAgICAgICAgJ2FybjonLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICB9LFxuICAgICAgICAgICc6c3RhdGVzOjo6c2FnZW1ha2VyOmNyZWF0ZVRyYWluaW5nSm9iLnN5bmMnLFxuICAgICAgICBdLFxuICAgICAgXSxcbiAgICB9LFxuICAgIEVuZDogdHJ1ZSxcbiAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICBUcmFpbmluZ0pvYk5hbWU6ICdNeVRyYWluSm9iJyxcbiAgICAgIFJvbGVBcm46IHsgJ0ZuOjpHZXRBdHQnOiBbJ1JvbGUxQUJDQzVGMCcsICdBcm4nXSB9LFxuICAgICAgQWxnb3JpdGhtU3BlY2lmaWNhdGlvbjoge1xuICAgICAgICBUcmFpbmluZ0lucHV0TW9kZTogJ0ZpbGUnLFxuICAgICAgICBBbGdvcml0aG1OYW1lOiAnQmxhemluZ1RleHQnLFxuICAgICAgICBNZXRyaWNEZWZpbml0aW9uczogW1xuICAgICAgICAgIHsgTmFtZTogJ215bWV0cmljJywgUmVnZXg6ICdyZWdleF9wYXR0ZXJuJyB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICAgIEh5cGVyUGFyYW1ldGVyczoge1xuICAgICAgICBscjogJzAuMScsXG4gICAgICB9LFxuICAgICAgSW5wdXREYXRhQ29uZmlnOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBDaGFubmVsTmFtZTogJ3RyYWluJyxcbiAgICAgICAgICBDb21wcmVzc2lvblR5cGU6ICdOb25lJyxcbiAgICAgICAgICBSZWNvcmRXcmFwcGVyVHlwZTogJ1JlY29yZElPJyxcbiAgICAgICAgICBDb250ZW50VHlwZTogJ2ltYWdlL2pwZWcnLFxuICAgICAgICAgIERhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgIFMzRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgICBBdHRyaWJ1dGVOYW1lczogWydzb3VyY2UtcmVmJywgJ2NsYXNzJ10sXG4gICAgICAgICAgICAgIFMzRGF0YVR5cGU6ICdTM1ByZWZpeCcsXG4gICAgICAgICAgICAgIFMzVXJpOiB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogWycnLCBbJ2h0dHBzOi8vczMuJywgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSwgJy4nLCB7IFJlZjogJ0FXUzo6VVJMU3VmZml4JyB9LCAnL215YnVja2V0L215dHJhaW5wYXRoJ11dLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgQ2hhbm5lbE5hbWU6ICd0ZXN0JyxcbiAgICAgICAgICBDb21wcmVzc2lvblR5cGU6ICdHemlwJyxcbiAgICAgICAgICBSZWNvcmRXcmFwcGVyVHlwZTogJ1JlY29yZElPJyxcbiAgICAgICAgICBDb250ZW50VHlwZTogJ2ltYWdlL2pwZWcnLFxuICAgICAgICAgIERhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgIFMzRGF0YVNvdXJjZToge1xuICAgICAgICAgICAgICBBdHRyaWJ1dGVOYW1lczogWydzb3VyY2UtcmVmJywgJ2NsYXNzJ10sXG4gICAgICAgICAgICAgIFMzRGF0YVR5cGU6ICdTM1ByZWZpeCcsXG4gICAgICAgICAgICAgIFMzVXJpOiB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogWycnLCBbJ2h0dHBzOi8vczMuJywgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSwgJy4nLCB7IFJlZjogJ0FXUzo6VVJMU3VmZml4JyB9LCAnL215YnVja2V0L215dGVzdHBhdGgnXV0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgICAgT3V0cHV0RGF0YUNvbmZpZzoge1xuICAgICAgICBTM091dHB1dFBhdGg6IHtcbiAgICAgICAgICAnRm46OkpvaW4nOiBbJycsIFsnaHR0cHM6Ly9zMy4nLCB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LCAnLicsIHsgUmVmOiAnQVdTOjpVUkxTdWZmaXgnIH0sICcvbXlidWNrZXQvbXlvdXRwdXRwYXRoJ11dLFxuICAgICAgICB9LFxuICAgICAgICBLbXNLZXlJZDogeyAnRm46OkdldEF0dCc6IFsnS2V5OTYxQjczRkQnLCAnQXJuJ10gfSxcbiAgICAgIH0sXG4gICAgICBSZXNvdXJjZUNvbmZpZzoge1xuICAgICAgICBJbnN0YW5jZUNvdW50OiAxLFxuICAgICAgICBJbnN0YW5jZVR5cGU6ICdtbC5wMy4yeGxhcmdlJyxcbiAgICAgICAgVm9sdW1lU2l6ZUluR0I6IDUwLFxuICAgICAgICBWb2x1bWVLbXNLZXlJZDogeyAnRm46OkdldEF0dCc6IFsnS2V5OTYxQjczRkQnLCAnQXJuJ10gfSxcbiAgICAgIH0sXG4gICAgICBTdG9wcGluZ0NvbmRpdGlvbjoge1xuICAgICAgICBNYXhSdW50aW1lSW5TZWNvbmRzOiAzNjAwLFxuICAgICAgfSxcbiAgICAgIFRhZ3M6IFtcbiAgICAgICAgeyBLZXk6ICdQcm9qZWN0JywgVmFsdWU6ICdNeVByb2plY3QnIH0sXG4gICAgICBdLFxuICAgICAgVnBjQ29uZmlnOiB7XG4gICAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IFtcbiAgICAgICAgICB7ICdGbjo6R2V0QXR0JzogWydUcmFpblNhZ2VtYWtlclRyYWluSm9iU2VjdXJpdHlHcm91cDdDODU4RUI5JywgJ0dyb3VwSWQnXSB9LFxuICAgICAgICAgIHsgJ0ZuOjpHZXRBdHQnOiBbJ1NlY3VyaXR5R3JvdXBERDI2MzYyMScsICdHcm91cElkJ10gfSxcbiAgICAgICAgXSxcbiAgICAgICAgU3VibmV0czogW1xuICAgICAgICAgIHsgUmVmOiAnVlBDUHJpdmF0ZVN1Ym5ldDFTdWJuZXQ4QkNBMTBFMCcgfSxcbiAgICAgICAgICB7IFJlZjogJ1ZQQ1ByaXZhdGVTdWJuZXQyU3VibmV0Q0ZDREFBN0EnIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ3Bhc3MgcGFyYW0gdG8gdHJhaW5pbmcgam9iJywgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzYWdlbWFrZXIuYW1hem9uYXdzLmNvbScpLFxuICAgIG1hbmFnZWRQb2xpY2llczogW1xuICAgICAgaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdBbWF6b25TYWdlTWFrZXJGdWxsQWNjZXNzJyksXG4gICAgXSxcbiAgfSk7XG5cbiAgY29uc3QgdGFzayA9IG5ldyBTYWdlTWFrZXJDcmVhdGVUcmFpbmluZ0pvYihzdGFjaywgJ1RyYWluU2FnZW1ha2VyJywge1xuICAgIHRyYWluaW5nSm9iTmFtZTogc2ZuLkpzb25QYXRoLnN0cmluZ0F0KCckLkpvYk5hbWUnKSxcbiAgICByb2xlLFxuICAgIGFsZ29yaXRobVNwZWNpZmljYXRpb246IHtcbiAgICAgIGFsZ29yaXRobU5hbWU6ICdCbGF6aW5nVGV4dCcsXG4gICAgICB0cmFpbmluZ0lucHV0TW9kZTogdGFza3MuSW5wdXRNb2RlLkZJTEUsXG4gICAgfSxcbiAgICBpbnB1dERhdGFDb25maWc6IFtcbiAgICAgIHtcbiAgICAgICAgY2hhbm5lbE5hbWU6ICd0cmFpbicsXG4gICAgICAgIGRhdGFTb3VyY2U6IHtcbiAgICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgIHMzRGF0YVR5cGU6IHRhc2tzLlMzRGF0YVR5cGUuUzNfUFJFRklYLFxuICAgICAgICAgICAgczNMb2NhdGlvbjogdGFza3MuUzNMb2NhdGlvbi5mcm9tSnNvbkV4cHJlc3Npb24oJyQuUzNCdWNrZXQnKSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICBdLFxuICAgIG91dHB1dERhdGFDb25maWc6IHtcbiAgICAgIHMzT3V0cHV0TG9jYXRpb246IHRhc2tzLlMzTG9jYXRpb24uZnJvbUJ1Y2tldChzMy5CdWNrZXQuZnJvbUJ1Y2tldE5hbWUoc3RhY2ssICdCdWNrZXQnLCAnbXlidWNrZXQnKSwgJ215b3V0cHV0cGF0aCcpLFxuICAgIH0sXG4gICAgcmVzb3VyY2VDb25maWc6IHtcbiAgICAgIGluc3RhbmNlQ291bnQ6IDEsXG4gICAgICBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuUDMsIGVjMi5JbnN0YW5jZVNpemUuWExBUkdFMiksXG4gICAgICB2b2x1bWVTaXplOiBjZGsuU2l6ZS5naWJpYnl0ZXMoNTApLFxuICAgIH0sXG4gICAgc3RvcHBpbmdDb25kaXRpb246IHtcbiAgICAgIG1heFJ1bnRpbWU6IGNkay5EdXJhdGlvbi5ob3VycygxKSxcbiAgICB9LFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OnNhZ2VtYWtlcjpjcmVhdGVUcmFpbmluZ0pvYicsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICdUcmFpbmluZ0pvYk5hbWUuJCc6ICckLkpvYk5hbWUnLFxuICAgICAgJ1JvbGVBcm4nOiB7ICdGbjo6R2V0QXR0JzogWydSb2xlMUFCQ0M1RjAnLCAnQXJuJ10gfSxcbiAgICAgICdBbGdvcml0aG1TcGVjaWZpY2F0aW9uJzoge1xuICAgICAgICBUcmFpbmluZ0lucHV0TW9kZTogJ0ZpbGUnLFxuICAgICAgICBBbGdvcml0aG1OYW1lOiAnQmxhemluZ1RleHQnLFxuICAgICAgfSxcbiAgICAgICdJbnB1dERhdGFDb25maWcnOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBDaGFubmVsTmFtZTogJ3RyYWluJyxcbiAgICAgICAgICBEYXRhU291cmNlOiB7XG4gICAgICAgICAgICBTM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgICAgJ1MzRGF0YVR5cGUnOiAnUzNQcmVmaXgnLFxuICAgICAgICAgICAgICAnUzNVcmkuJCc6ICckLlMzQnVja2V0JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICAnT3V0cHV0RGF0YUNvbmZpZyc6IHtcbiAgICAgICAgUzNPdXRwdXRQYXRoOiB7XG4gICAgICAgICAgJ0ZuOjpKb2luJzogWycnLCBbJ2h0dHBzOi8vczMuJywgeyBSZWY6ICdBV1M6OlJlZ2lvbicgfSwgJy4nLCB7IFJlZjogJ0FXUzo6VVJMU3VmZml4JyB9LCAnL215YnVja2V0L215b3V0cHV0cGF0aCddXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICAnUmVzb3VyY2VDb25maWcnOiB7XG4gICAgICAgIEluc3RhbmNlQ291bnQ6IDEsXG4gICAgICAgIEluc3RhbmNlVHlwZTogJ21sLnAzLjJ4bGFyZ2UnLFxuICAgICAgICBWb2x1bWVTaXplSW5HQjogNTAsXG4gICAgICB9LFxuICAgICAgJ1N0b3BwaW5nQ29uZGl0aW9uJzoge1xuICAgICAgICBNYXhSdW50aW1lSW5TZWNvbmRzOiAzNjAwLFxuICAgICAgfSxcbiAgICB9LFxuICB9KTtcbn0pO1xuXG50ZXN0KCdDYW5ub3QgY3JlYXRlIGEgU2FnZU1ha2VyIHRyYWluIHRhc2sgd2l0aCBib3RoIGFsZ29yaXRobSBuYW1lIGFuZCBpbWFnZSBuYW1lIG1pc3NpbmcnLCAoKSA9PiB7XG5cbiAgZXhwZWN0KCgpID0+IG5ldyBTYWdlTWFrZXJDcmVhdGVUcmFpbmluZ0pvYihzdGFjaywgJ1NhZ2VNYWtlclRyYWluaW5nVGFzaycsIHtcbiAgICB0cmFpbmluZ0pvYk5hbWU6ICdteVRyYWluSm9iJyxcbiAgICBhbGdvcml0aG1TcGVjaWZpY2F0aW9uOiB7fSxcbiAgICBpbnB1dERhdGFDb25maWc6IFtcbiAgICAgIHtcbiAgICAgICAgY2hhbm5lbE5hbWU6ICd0cmFpbicsXG4gICAgICAgIGRhdGFTb3VyY2U6IHtcbiAgICAgICAgICBzM0RhdGFTb3VyY2U6IHtcbiAgICAgICAgICAgIHMzRGF0YVR5cGU6IHRhc2tzLlMzRGF0YVR5cGUuUzNfUFJFRklYLFxuICAgICAgICAgICAgczNMb2NhdGlvbjogdGFza3MuUzNMb2NhdGlvbi5mcm9tSnNvbkV4cHJlc3Npb24oJyQuUzNCdWNrZXQnKSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICBdLFxuICAgIG91dHB1dERhdGFDb25maWc6IHtcbiAgICAgIHMzT3V0cHV0TG9jYXRpb246IHRhc2tzLlMzTG9jYXRpb24uZnJvbUJ1Y2tldChzMy5CdWNrZXQuZnJvbUJ1Y2tldE5hbWUoc3RhY2ssICdCdWNrZXQnLCAnbXlidWNrZXQnKSwgJ215b3V0cHV0cGF0aC8nKSxcbiAgICB9LFxuICB9KSlcbiAgICAudG9UaHJvd0Vycm9yKC9NdXN0IGRlZmluZSBlaXRoZXIgYW4gYWxnb3JpdGhtIG5hbWUgb3IgdHJhaW5pbmcgaW1hZ2UgVVJJIGluIHRoZSBhbGdvcml0aG0gc3BlY2lmaWNhdGlvbi8pO1xufSk7XG4iXX0=