"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lambda = require("@aws-cdk/aws-lambda");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../lib");
/* eslint-disable quote-props */
describe('LambdaInvoke', () => {
    let stack;
    let lambdaFunction;
    beforeEach(() => {
        // GIVEN
        stack = new core_1.Stack();
        lambdaFunction = new lambda.Function(stack, 'Fn', {
            code: lambda.Code.fromInline('foo'),
            handler: 'handler',
            runtime: lambda.Runtime.NODEJS_12_X,
        });
    });
    test('default settings', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                'Payload.$': '$',
            },
            Retry: [
                {
                    ErrorEquals: [
                        'Lambda.ServiceException',
                        'Lambda.AWSLambdaException',
                        'Lambda.SdkClientException',
                    ],
                    IntervalSeconds: 2,
                    MaxAttempts: 6,
                    BackoffRate: 2,
                },
            ],
        });
    });
    test('optional settings', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payload: sfn.TaskInput.fromObject({
                foo: 'bar',
            }),
            invocationType: lib_1.LambdaInvocationType.REQUEST_RESPONSE,
            clientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            qualifier: '1',
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual(expect.objectContaining({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            End: true,
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                Payload: {
                    foo: 'bar',
                },
                InvocationType: 'RequestResponse',
                ClientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
                Qualifier: '1',
            },
        }));
    });
    test('invoke Lambda function and wait for task token', () => {
        // GIVEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: sfn.TaskInput.fromObject({
                token: sfn.JsonPath.taskToken,
            }),
            qualifier: 'my-alias',
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual(expect.objectContaining({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke.waitForTaskToken',
                    ],
                ],
            },
            End: true,
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                Payload: {
                    'token.$': '$$.Task.Token',
                },
                Qualifier: 'my-alias',
            },
        }));
    });
    test('pass part of state input as input to Lambda function ', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payload: sfn.TaskInput.fromDataAt('$.foo'),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual(expect.objectContaining({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            End: true,
            Parameters: {
                'FunctionName': {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                'Payload.$': '$.foo',
            },
        }));
    });
    test('Invoke lambda with payloadResponseOnly', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payloadResponseOnly: true,
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual(expect.objectContaining({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::GetAtt': [
                    'Fn9270CBC0',
                    'Arn',
                ],
            },
        }));
    });
    test('Invoke lambda with payloadResponseOnly with payload', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            payloadResponseOnly: true,
            payload: sfn.TaskInput.fromObject({
                foo: 'bar',
            }),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual(expect.objectContaining({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::GetAtt': [
                    'Fn9270CBC0',
                    'Arn',
                ],
            },
            Parameters: {
                foo: 'bar',
            },
        }));
    });
    test('with retryOnServiceExceptions set to false', () => {
        // WHEN
        const task = new lib_1.LambdaInvoke(stack, 'Task', {
            lambdaFunction,
            retryOnServiceExceptions: false,
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::lambda:invoke',
                    ],
                ],
            },
            Parameters: {
                FunctionName: {
                    'Fn::GetAtt': [
                        'Fn9270CBC0',
                        'Arn',
                    ],
                },
                'Payload.$': '$',
            },
        });
    });
    test('fails when integrationPattern used with payloadResponseOnly', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                payloadResponseOnly: true,
                integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                payload: sfn.TaskInput.fromObject({
                    token: sfn.JsonPath.taskToken,
                }),
            });
        }).toThrow(/The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified./);
    });
    test('fails when invocationType used with payloadResponseOnly', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                payloadResponseOnly: true,
                payload: sfn.TaskInput.fromObject({
                    foo: 'bar',
                }),
                invocationType: lib_1.LambdaInvocationType.REQUEST_RESPONSE,
            });
        }).toThrow(/The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified./);
    });
    test('fails when clientContext used with payloadResponseOnly', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                payloadResponseOnly: true,
                payload: sfn.TaskInput.fromObject({
                    foo: 'bar',
                }),
                clientContext: 'eyJoZWxsbyI6IndvcmxkIn0=',
            });
        }).toThrow(/The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified./);
    });
    test('fails when qualifier used with payloadResponseOnly', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                payloadResponseOnly: true,
                payload: sfn.TaskInput.fromObject({
                    foo: 'bar',
                }),
                qualifier: '1',
            });
        }).toThrow(/The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified./);
    });
    test('fails when WAIT_FOR_TASK_TOKEN integration pattern is used without supplying a task token in payload', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            });
        }).toThrow(/Task Token is required in `payload` for callback. Use JsonPath.taskToken to set the token./);
    });
    test('fails when RUN_JOB integration pattern is used', () => {
        expect(() => {
            new lib_1.LambdaInvoke(stack, 'Task', {
                lambdaFunction,
                integrationPattern: sfn.IntegrationPattern.RUN_JOB,
            });
        }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,WAIT_FOR_TASK_TOKEN. Received: RUN_JOB/);
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW52b2tlLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnZva2UudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLGdDQUE4QjtBQUM5Qiw4Q0FBOEM7QUFDOUMsa0RBQWtEO0FBQ2xELHdDQUFzQztBQUN0QyxtQ0FBK0Q7QUFFL0QsZ0NBQWdDO0FBRWhDLFFBQVEsQ0FBQyxjQUFjLEVBQUUsR0FBRyxFQUFFO0lBRTVCLElBQUksS0FBWSxDQUFDO0lBQ2pCLElBQUksY0FBK0IsQ0FBQztJQUVwQyxVQUFVLENBQUMsR0FBRyxFQUFFO1FBQ2QsUUFBUTtRQUNSLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQ3BCLGNBQWMsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtZQUNoRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDO1lBQ25DLE9BQU8sRUFBRSxTQUFTO1lBQ2xCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7U0FDcEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxFQUFFO1FBQzVCLE9BQU87UUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUMzQyxjQUFjO1NBQ2YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1lBQ2hELEdBQUcsRUFBRSxJQUFJO1lBQ1QsSUFBSSxFQUFFLE1BQU07WUFDWixRQUFRLEVBQUU7Z0JBQ1IsVUFBVSxFQUFFO29CQUNWLEVBQUU7b0JBQ0Y7d0JBQ0UsTUFBTTt3QkFDTjs0QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3lCQUN0Qjt3QkFDRCx5QkFBeUI7cUJBQzFCO2lCQUNGO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFO29CQUNaLFlBQVksRUFBRTt3QkFDWixZQUFZO3dCQUNaLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QsV0FBVyxFQUFFLEdBQUc7YUFDakI7WUFDRCxLQUFLLEVBQUU7Z0JBQ0w7b0JBQ0UsV0FBVyxFQUFFO3dCQUNYLHlCQUF5Qjt3QkFDekIsMkJBQTJCO3dCQUMzQiwyQkFBMkI7cUJBQzVCO29CQUNELGVBQWUsRUFBRSxDQUFDO29CQUNsQixXQUFXLEVBQUUsQ0FBQztvQkFDZCxXQUFXLEVBQUUsQ0FBQztpQkFDZjthQUNGO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsbUJBQW1CLEVBQUUsR0FBRyxFQUFFO1FBQzdCLE9BQU87UUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUMzQyxjQUFjO1lBQ2QsT0FBTyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDO2dCQUNoQyxHQUFHLEVBQUUsS0FBSzthQUNYLENBQUM7WUFDRixjQUFjLEVBQUUsMEJBQW9CLENBQUMsZ0JBQWdCO1lBQ3JELGFBQWEsRUFBRSwwQkFBMEI7WUFDekMsU0FBUyxFQUFFLEdBQUc7U0FDZixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ3hFLElBQUksRUFBRSxNQUFNO1lBQ1osUUFBUSxFQUFFO2dCQUNSLFVBQVUsRUFBRTtvQkFDVixFQUFFO29CQUNGO3dCQUNFLE1BQU07d0JBQ047NEJBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5QkFDdEI7d0JBQ0QseUJBQXlCO3FCQUMxQjtpQkFDRjthQUNGO1lBQ0QsR0FBRyxFQUFFLElBQUk7WUFDVCxVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFO29CQUNaLFlBQVksRUFBRTt3QkFDWixZQUFZO3dCQUNaLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QsT0FBTyxFQUFFO29CQUNQLEdBQUcsRUFBRSxLQUFLO2lCQUNYO2dCQUNELGNBQWMsRUFBRSxpQkFBaUI7Z0JBQ2pDLGFBQWEsRUFBRSwwQkFBMEI7Z0JBQ3pDLFNBQVMsRUFBRSxHQUFHO2FBQ2Y7U0FDRixDQUFDLENBQUMsQ0FBQztJQUNOLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGdEQUFnRCxFQUFFLEdBQUcsRUFBRTtRQUMxRCxRQUFRO1FBQ1IsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7WUFDM0MsY0FBYztZQUNkLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUI7WUFDOUQsT0FBTyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDO2dCQUNoQyxLQUFLLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxTQUFTO2FBQzlCLENBQUM7WUFDRixTQUFTLEVBQUUsVUFBVTtTQUN0QixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ3hFLElBQUksRUFBRSxNQUFNO1lBQ1osUUFBUSxFQUFFO2dCQUNSLFVBQVUsRUFBRTtvQkFDVixFQUFFO29CQUNGO3dCQUNFLE1BQU07d0JBQ047NEJBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5QkFDdEI7d0JBQ0QsMENBQTBDO3FCQUMzQztpQkFDRjthQUNGO1lBQ0QsR0FBRyxFQUFFLElBQUk7WUFDVCxVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFO29CQUNaLFlBQVksRUFBRTt3QkFDWixZQUFZO3dCQUNaLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QsT0FBTyxFQUFFO29CQUNQLFNBQVMsRUFBRSxlQUFlO2lCQUMzQjtnQkFDRCxTQUFTLEVBQUUsVUFBVTthQUN0QjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsdURBQXVELEVBQUUsR0FBRyxFQUFFO1FBQ2pFLE9BQU87UUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUMzQyxjQUFjO1lBQ2QsT0FBTyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQztTQUMzQyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDO1lBQ3hFLElBQUksRUFBRSxNQUFNO1lBQ1osUUFBUSxFQUFFO2dCQUNSLFVBQVUsRUFBRTtvQkFDVixFQUFFO29CQUNGO3dCQUNFLE1BQU07d0JBQ047NEJBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5QkFDdEI7d0JBQ0QseUJBQXlCO3FCQUMxQjtpQkFDRjthQUNGO1lBQ0QsR0FBRyxFQUFFLElBQUk7WUFDVCxVQUFVLEVBQUU7Z0JBQ1YsY0FBYyxFQUFFO29CQUNkLFlBQVksRUFBRTt3QkFDWixZQUFZO3dCQUNaLEtBQUs7cUJBQ047aUJBQ0Y7Z0JBQ0QsV0FBVyxFQUFFLE9BQU87YUFDckI7U0FDRixDQUFDLENBQUMsQ0FBQztJQUNOLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHdDQUF3QyxFQUFFLEdBQUcsRUFBRTtRQUNsRCxPQUFPO1FBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7WUFDM0MsY0FBYztZQUNkLG1CQUFtQixFQUFFLElBQUk7U0FDMUIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQztZQUN4RSxHQUFHLEVBQUUsSUFBSTtZQUNULElBQUksRUFBRSxNQUFNO1lBQ1osUUFBUSxFQUFFO2dCQUNSLFlBQVksRUFBRTtvQkFDWixZQUFZO29CQUNaLEtBQUs7aUJBQ047YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMscURBQXFELEVBQUUsR0FBRyxFQUFFO1FBQy9ELE9BQU87UUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUMzQyxjQUFjO1lBQ2QsbUJBQW1CLEVBQUUsSUFBSTtZQUN6QixPQUFPLEVBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUM7Z0JBQ2hDLEdBQUcsRUFBRSxLQUFLO2FBQ1gsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7WUFDeEUsR0FBRyxFQUFFLElBQUk7WUFDVCxJQUFJLEVBQUUsTUFBTTtZQUNaLFFBQVEsRUFBRTtnQkFDUixZQUFZLEVBQUU7b0JBQ1osWUFBWTtvQkFDWixLQUFLO2lCQUNOO2FBQ0Y7WUFDRCxVQUFVLEVBQUU7Z0JBQ1YsR0FBRyxFQUFFLEtBQUs7YUFDWDtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNENBQTRDLEVBQUUsR0FBRyxFQUFFO1FBQ3RELE9BQU87UUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUMzQyxjQUFjO1lBQ2Qsd0JBQXdCLEVBQUUsS0FBSztTQUNoQyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7WUFDaEQsR0FBRyxFQUFFLElBQUk7WUFDVCxJQUFJLEVBQUUsTUFBTTtZQUNaLFFBQVEsRUFBRTtnQkFDUixVQUFVLEVBQUU7b0JBQ1YsRUFBRTtvQkFDRjt3QkFDRSxNQUFNO3dCQUNOOzRCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUJBQ3RCO3dCQUNELHlCQUF5QjtxQkFDMUI7aUJBQ0Y7YUFDRjtZQUNELFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUU7b0JBQ1osWUFBWSxFQUFFO3dCQUNaLFlBQVk7d0JBQ1osS0FBSztxQkFDTjtpQkFDRjtnQkFDRCxXQUFXLEVBQUUsR0FBRzthQUNqQjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLDZEQUE2RCxFQUFFLEdBQUcsRUFBRTtRQUN2RSxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQzlCLGNBQWM7Z0JBQ2QsbUJBQW1CLEVBQUUsSUFBSTtnQkFDekIsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLGtCQUFrQixDQUFDLG1CQUFtQjtnQkFDOUQsT0FBTyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDO29CQUNoQyxLQUFLLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxTQUFTO2lCQUM5QixDQUFDO2FBQ0gsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLDZJQUE2SSxDQUFDLENBQUM7SUFDNUosQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMseURBQXlELEVBQUUsR0FBRyxFQUFFO1FBQ25FLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDOUIsY0FBYztnQkFDZCxtQkFBbUIsRUFBRSxJQUFJO2dCQUN6QixPQUFPLEVBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUM7b0JBQ2hDLEdBQUcsRUFBRSxLQUFLO2lCQUNYLENBQUM7Z0JBQ0YsY0FBYyxFQUFFLDBCQUFvQixDQUFDLGdCQUFnQjthQUN0RCxDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsNklBQTZJLENBQUMsQ0FBQztJQUM1SixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx3REFBd0QsRUFBRSxHQUFHLEVBQUU7UUFDbEUsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLElBQUksa0JBQVksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUM5QixjQUFjO2dCQUNkLG1CQUFtQixFQUFFLElBQUk7Z0JBQ3pCLE9BQU8sRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQztvQkFDaEMsR0FBRyxFQUFFLEtBQUs7aUJBQ1gsQ0FBQztnQkFDRixhQUFhLEVBQUUsMEJBQTBCO2FBQzFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyw2SUFBNkksQ0FBQyxDQUFDO0lBQzVKLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLG9EQUFvRCxFQUFFLEdBQUcsRUFBRTtRQUM5RCxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQzlCLGNBQWM7Z0JBQ2QsbUJBQW1CLEVBQUUsSUFBSTtnQkFDekIsT0FBTyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDO29CQUNoQyxHQUFHLEVBQUUsS0FBSztpQkFDWCxDQUFDO2dCQUNGLFNBQVMsRUFBRSxHQUFHO2FBQ2YsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLDZJQUE2SSxDQUFDLENBQUM7SUFDNUosQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsc0dBQXNHLEVBQUUsR0FBRyxFQUFFO1FBQ2hILE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDOUIsY0FBYztnQkFDZCxrQkFBa0IsRUFBRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CO2FBQy9ELENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyw0RkFBNEYsQ0FBQyxDQUFDO0lBQzNHLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGdEQUFnRCxFQUFFLEdBQUcsRUFBRTtRQUMxRCxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQzlCLGNBQWM7Z0JBQ2Qsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU87YUFDbkQsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLHNIQUFzSCxDQUFDLENBQUM7SUFDckksQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCB7IFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBMYW1iZGFJbnZvY2F0aW9uVHlwZSwgTGFtYmRhSW52b2tlIH0gZnJvbSAnLi4vLi4vbGliJztcblxuLyogZXNsaW50LWRpc2FibGUgcXVvdGUtcHJvcHMgKi9cblxuZGVzY3JpYmUoJ0xhbWJkYUludm9rZScsICgpID0+IHtcblxuICBsZXQgc3RhY2s6IFN0YWNrO1xuICBsZXQgbGFtYmRhRnVuY3Rpb246IGxhbWJkYS5GdW5jdGlvbjtcblxuICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgbGFtYmRhRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnRm4nLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tSW5saW5lKCdmb28nKSxcbiAgICAgIGhhbmRsZXI6ICdoYW5kbGVyJyxcbiAgICAgIHJ1bnRpbWU6IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMl9YLFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdkZWZhdWx0IHNldHRpbmdzJywgKCkgPT4ge1xuICAgIC8vIFdIRU5cbiAgICBjb25zdCB0YXNrID0gbmV3IExhbWJkYUludm9rZShzdGFjaywgJ1Rhc2snLCB7XG4gICAgICBsYW1iZGFGdW5jdGlvbixcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICAgIEVuZDogdHJ1ZSxcbiAgICAgIFR5cGU6ICdUYXNrJyxcbiAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAnJyxcbiAgICAgICAgICBbXG4gICAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAnOnN0YXRlczo6OmxhbWJkYTppbnZva2UnLFxuICAgICAgICAgIF0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICBGdW5jdGlvbk5hbWU6IHtcbiAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICdGbjkyNzBDQkMwJyxcbiAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgICdQYXlsb2FkLiQnOiAnJCcsXG4gICAgICB9LFxuICAgICAgUmV0cnk6IFtcbiAgICAgICAge1xuICAgICAgICAgIEVycm9yRXF1YWxzOiBbXG4gICAgICAgICAgICAnTGFtYmRhLlNlcnZpY2VFeGNlcHRpb24nLFxuICAgICAgICAgICAgJ0xhbWJkYS5BV1NMYW1iZGFFeGNlcHRpb24nLFxuICAgICAgICAgICAgJ0xhbWJkYS5TZGtDbGllbnRFeGNlcHRpb24nLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgSW50ZXJ2YWxTZWNvbmRzOiAyLFxuICAgICAgICAgIE1heEF0dGVtcHRzOiA2LFxuICAgICAgICAgIEJhY2tvZmZSYXRlOiAyLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnb3B0aW9uYWwgc2V0dGluZ3MnLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHRhc2sgPSBuZXcgTGFtYmRhSW52b2tlKHN0YWNrLCAnVGFzaycsIHtcbiAgICAgIGxhbWJkYUZ1bmN0aW9uLFxuICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgZm9vOiAnYmFyJyxcbiAgICAgIH0pLFxuICAgICAgaW52b2NhdGlvblR5cGU6IExhbWJkYUludm9jYXRpb25UeXBlLlJFUVVFU1RfUkVTUE9OU0UsXG4gICAgICBjbGllbnRDb250ZXh0OiAnZXlKb1pXeHNieUk2SW5kdmNteGtJbjA9JyxcbiAgICAgIHF1YWxpZmllcjogJzEnLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgVHlwZTogJ1Rhc2snLFxuICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICcnLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICc6c3RhdGVzOjo6bGFtYmRhOmludm9rZScsXG4gICAgICAgICAgXSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICBFbmQ6IHRydWUsXG4gICAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAgIEZ1bmN0aW9uTmFtZToge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgJ0ZuOTI3MENCQzAnLFxuICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICAgUGF5bG9hZDoge1xuICAgICAgICAgIGZvbzogJ2JhcicsXG4gICAgICAgIH0sXG4gICAgICAgIEludm9jYXRpb25UeXBlOiAnUmVxdWVzdFJlc3BvbnNlJyxcbiAgICAgICAgQ2xpZW50Q29udGV4dDogJ2V5Sm9aV3hzYnlJNkluZHZjbXhrSW4wPScsXG4gICAgICAgIFF1YWxpZmllcjogJzEnLFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2ludm9rZSBMYW1iZGEgZnVuY3Rpb24gYW5kIHdhaXQgZm9yIHRhc2sgdG9rZW4nLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCB0YXNrID0gbmV3IExhbWJkYUludm9rZShzdGFjaywgJ1Rhc2snLCB7XG4gICAgICBsYW1iZGFGdW5jdGlvbixcbiAgICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLkludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgdG9rZW46IHNmbi5Kc29uUGF0aC50YXNrVG9rZW4sXG4gICAgICB9KSxcbiAgICAgIHF1YWxpZmllcjogJ215LWFsaWFzJyxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKGV4cGVjdC5vYmplY3RDb250YWluaW5nKHtcbiAgICAgIFR5cGU6ICdUYXNrJyxcbiAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAnJyxcbiAgICAgICAgICBbXG4gICAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAnOnN0YXRlczo6OmxhbWJkYTppbnZva2Uud2FpdEZvclRhc2tUb2tlbicsXG4gICAgICAgICAgXSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICBFbmQ6IHRydWUsXG4gICAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAgIEZ1bmN0aW9uTmFtZToge1xuICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgJ0ZuOTI3MENCQzAnLFxuICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgXSxcbiAgICAgICAgfSxcbiAgICAgICAgUGF5bG9hZDoge1xuICAgICAgICAgICd0b2tlbi4kJzogJyQkLlRhc2suVG9rZW4nLFxuICAgICAgICB9LFxuICAgICAgICBRdWFsaWZpZXI6ICdteS1hbGlhcycsXG4gICAgICB9LFxuICAgIH0pKTtcbiAgfSk7XG5cbiAgdGVzdCgncGFzcyBwYXJ0IG9mIHN0YXRlIGlucHV0IGFzIGlucHV0IHRvIExhbWJkYSBmdW5jdGlvbiAnLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHRhc2sgPSBuZXcgTGFtYmRhSW52b2tlKHN0YWNrLCAnVGFzaycsIHtcbiAgICAgIGxhbWJkYUZ1bmN0aW9uLFxuICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tRGF0YUF0KCckLmZvbycpLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgVHlwZTogJ1Rhc2snLFxuICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICcnLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICc6c3RhdGVzOjo6bGFtYmRhOmludm9rZScsXG4gICAgICAgICAgXSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICBFbmQ6IHRydWUsXG4gICAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAgICdGdW5jdGlvbk5hbWUnOiB7XG4gICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAnRm45MjcwQ0JDMCcsXG4gICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgICAnUGF5bG9hZC4kJzogJyQuZm9vJyxcbiAgICAgIH0sXG4gICAgfSkpO1xuICB9KTtcblxuICB0ZXN0KCdJbnZva2UgbGFtYmRhIHdpdGggcGF5bG9hZFJlc3BvbnNlT25seScsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFzayA9IG5ldyBMYW1iZGFJbnZva2Uoc3RhY2ssICdUYXNrJywge1xuICAgICAgbGFtYmRhRnVuY3Rpb24sXG4gICAgICBwYXlsb2FkUmVzcG9uc2VPbmx5OiB0cnVlLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgRW5kOiB0cnVlLFxuICAgICAgVHlwZTogJ1Rhc2snLFxuICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0ZuOTI3MENCQzAnLFxuICAgICAgICAgICdBcm4nLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ0ludm9rZSBsYW1iZGEgd2l0aCBwYXlsb2FkUmVzcG9uc2VPbmx5IHdpdGggcGF5bG9hZCcsICgpID0+IHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFzayA9IG5ldyBMYW1iZGFJbnZva2Uoc3RhY2ssICdUYXNrJywge1xuICAgICAgbGFtYmRhRnVuY3Rpb24sXG4gICAgICBwYXlsb2FkUmVzcG9uc2VPbmx5OiB0cnVlLFxuICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgZm9vOiAnYmFyJyxcbiAgICAgIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoZXhwZWN0Lm9iamVjdENvbnRhaW5pbmcoe1xuICAgICAgRW5kOiB0cnVlLFxuICAgICAgVHlwZTogJ1Rhc2snLFxuICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0ZuOTI3MENCQzAnLFxuICAgICAgICAgICdBcm4nLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICAgZm9vOiAnYmFyJyxcbiAgICAgIH0sXG4gICAgfSkpO1xuICB9KTtcblxuICB0ZXN0KCd3aXRoIHJldHJ5T25TZXJ2aWNlRXhjZXB0aW9ucyBzZXQgdG8gZmFsc2UnLCAoKSA9PiB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHRhc2sgPSBuZXcgTGFtYmRhSW52b2tlKHN0YWNrLCAnVGFzaycsIHtcbiAgICAgIGxhbWJkYUZ1bmN0aW9uLFxuICAgICAgcmV0cnlPblNlcnZpY2VFeGNlcHRpb25zOiBmYWxzZSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2sucmVzb2x2ZSh0YXNrLnRvU3RhdGVKc29uKCkpKS50b0VxdWFsKHtcbiAgICAgIEVuZDogdHJ1ZSxcbiAgICAgIFR5cGU6ICdUYXNrJyxcbiAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAnJyxcbiAgICAgICAgICBbXG4gICAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAnOnN0YXRlczo6OmxhbWJkYTppbnZva2UnLFxuICAgICAgICAgIF0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICBGdW5jdGlvbk5hbWU6IHtcbiAgICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAgICdGbjkyNzBDQkMwJyxcbiAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0sXG4gICAgICAgICdQYXlsb2FkLiQnOiAnJCcsXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdmYWlscyB3aGVuIGludGVncmF0aW9uUGF0dGVybiB1c2VkIHdpdGggcGF5bG9hZFJlc3BvbnNlT25seScsICgpID0+IHtcbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IExhbWJkYUludm9rZShzdGFjaywgJ1Rhc2snLCB7XG4gICAgICAgIGxhbWJkYUZ1bmN0aW9uLFxuICAgICAgICBwYXlsb2FkUmVzcG9uc2VPbmx5OiB0cnVlLFxuICAgICAgICBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uV0FJVF9GT1JfVEFTS19UT0tFTixcbiAgICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgICB0b2tlbjogc2ZuLkpzb25QYXRoLnRhc2tUb2tlbixcbiAgICAgICAgfSksXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93KC9UaGUgJ3BheWxvYWRSZXNwb25zZU9ubHknIHByb3BlcnR5IGNhbm5vdCBiZSB1c2VkIGlmICdpbnRlZ3JhdGlvblBhdHRlcm4nLCAnaW52b2NhdGlvblR5cGUnLCAnY2xpZW50Q29udGV4dCcsIG9yICdxdWFsaWZpZXInIGFyZSBzcGVjaWZpZWQuLyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2ZhaWxzIHdoZW4gaW52b2NhdGlvblR5cGUgdXNlZCB3aXRoIHBheWxvYWRSZXNwb25zZU9ubHknLCAoKSA9PiB7XG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIG5ldyBMYW1iZGFJbnZva2Uoc3RhY2ssICdUYXNrJywge1xuICAgICAgICBsYW1iZGFGdW5jdGlvbixcbiAgICAgICAgcGF5bG9hZFJlc3BvbnNlT25seTogdHJ1ZSxcbiAgICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgICBmb286ICdiYXInLFxuICAgICAgICB9KSxcbiAgICAgICAgaW52b2NhdGlvblR5cGU6IExhbWJkYUludm9jYXRpb25UeXBlLlJFUVVFU1RfUkVTUE9OU0UsXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93KC9UaGUgJ3BheWxvYWRSZXNwb25zZU9ubHknIHByb3BlcnR5IGNhbm5vdCBiZSB1c2VkIGlmICdpbnRlZ3JhdGlvblBhdHRlcm4nLCAnaW52b2NhdGlvblR5cGUnLCAnY2xpZW50Q29udGV4dCcsIG9yICdxdWFsaWZpZXInIGFyZSBzcGVjaWZpZWQuLyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2ZhaWxzIHdoZW4gY2xpZW50Q29udGV4dCB1c2VkIHdpdGggcGF5bG9hZFJlc3BvbnNlT25seScsICgpID0+IHtcbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IExhbWJkYUludm9rZShzdGFjaywgJ1Rhc2snLCB7XG4gICAgICAgIGxhbWJkYUZ1bmN0aW9uLFxuICAgICAgICBwYXlsb2FkUmVzcG9uc2VPbmx5OiB0cnVlLFxuICAgICAgICBwYXlsb2FkOiBzZm4uVGFza0lucHV0LmZyb21PYmplY3Qoe1xuICAgICAgICAgIGZvbzogJ2JhcicsXG4gICAgICAgIH0pLFxuICAgICAgICBjbGllbnRDb250ZXh0OiAnZXlKb1pXeHNieUk2SW5kdmNteGtJbjA9JyxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3coL1RoZSAncGF5bG9hZFJlc3BvbnNlT25seScgcHJvcGVydHkgY2Fubm90IGJlIHVzZWQgaWYgJ2ludGVncmF0aW9uUGF0dGVybicsICdpbnZvY2F0aW9uVHlwZScsICdjbGllbnRDb250ZXh0Jywgb3IgJ3F1YWxpZmllcicgYXJlIHNwZWNpZmllZC4vKTtcbiAgfSk7XG5cbiAgdGVzdCgnZmFpbHMgd2hlbiBxdWFsaWZpZXIgdXNlZCB3aXRoIHBheWxvYWRSZXNwb25zZU9ubHknLCAoKSA9PiB7XG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIG5ldyBMYW1iZGFJbnZva2Uoc3RhY2ssICdUYXNrJywge1xuICAgICAgICBsYW1iZGFGdW5jdGlvbixcbiAgICAgICAgcGF5bG9hZFJlc3BvbnNlT25seTogdHJ1ZSxcbiAgICAgICAgcGF5bG9hZDogc2ZuLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgICBmb286ICdiYXInLFxuICAgICAgICB9KSxcbiAgICAgICAgcXVhbGlmaWVyOiAnMScsXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93KC9UaGUgJ3BheWxvYWRSZXNwb25zZU9ubHknIHByb3BlcnR5IGNhbm5vdCBiZSB1c2VkIGlmICdpbnRlZ3JhdGlvblBhdHRlcm4nLCAnaW52b2NhdGlvblR5cGUnLCAnY2xpZW50Q29udGV4dCcsIG9yICdxdWFsaWZpZXInIGFyZSBzcGVjaWZpZWQuLyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2ZhaWxzIHdoZW4gV0FJVF9GT1JfVEFTS19UT0tFTiBpbnRlZ3JhdGlvbiBwYXR0ZXJuIGlzIHVzZWQgd2l0aG91dCBzdXBwbHlpbmcgYSB0YXNrIHRva2VuIGluIHBheWxvYWQnLCAoKSA9PiB7XG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIG5ldyBMYW1iZGFJbnZva2Uoc3RhY2ssICdUYXNrJywge1xuICAgICAgICBsYW1iZGFGdW5jdGlvbixcbiAgICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLldBSVRfRk9SX1RBU0tfVE9LRU4sXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93KC9UYXNrIFRva2VuIGlzIHJlcXVpcmVkIGluIGBwYXlsb2FkYCBmb3IgY2FsbGJhY2suIFVzZSBKc29uUGF0aC50YXNrVG9rZW4gdG8gc2V0IHRoZSB0b2tlbi4vKTtcbiAgfSk7XG5cbiAgdGVzdCgnZmFpbHMgd2hlbiBSVU5fSk9CIGludGVncmF0aW9uIHBhdHRlcm4gaXMgdXNlZCcsICgpID0+IHtcbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IExhbWJkYUludm9rZShzdGFjaywgJ1Rhc2snLCB7XG4gICAgICAgIGxhbWJkYUZ1bmN0aW9uLFxuICAgICAgICBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQixcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3coL1Vuc3VwcG9ydGVkIHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybi4gU3VwcG9ydGVkIFBhdHRlcm5zOiBSRVFVRVNUX1JFU1BPTlNFLFdBSVRfRk9SX1RBU0tfVE9LRU4uIFJlY2VpdmVkOiBSVU5fSk9CLyk7XG4gIH0pO1xufSk7XG4iXX0=