"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const iam = require("@aws-cdk/aws-iam");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test minimal deployment snapshot
// --------------------------------------------------------------
test('Test minimal deployment snapshot', () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template'
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test construct properties
// --------------------------------------------------------------
test('Test construct properties', () => {
    const stack = new core_1.Stack();
    const pattern = new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template'
    });
    expect(pattern.apiGateway !== null);
    expect(pattern.apiGatewayRole !== null);
    expect(pattern.apiGatewayCloudWatchRole !== null);
    expect(pattern.apiGatewayLogGroup !== null);
});
// --------------------------------------------------------------
// Test deployment w/ overwritten properties
// --------------------------------------------------------------
test('Test deployment w/ overwritten properties', () => {
    const stack = new core_1.Stack();
    const existingRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        description: 'existing role for SageMaker integration',
        inlinePolicies: {
            InvokePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:endpoint/my-endpoint`],
                        actions: ['sagemaker:InvokeEndpoint']
                    })]
            })
        }
    });
    new lib_1.ApiGatewayToSageMakerEndpoint(stack, 'api-gateway-sagemakerendpoint', {
        endpointName: 'my-endpoint',
        resourcePath: '{my_param}',
        requestMappingTemplate: 'my-request-vtl-template',
        apiGatewayProps: {
            restApiName: 'my-api',
            deployOptions: {
                methodOptions: {
                    '/*/*': {
                        throttlingRateLimit: 100,
                        throttlingBurstLimit: 25
                    }
                }
            }
        },
        apiGatewayExecutionRole: existingRole,
        resourceName: 'my-resource',
        responseMappingTemplate: 'my-response-vtl-template'
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Stage', {
        MethodSettings: [
            {
                DataTraceEnabled: true,
                HttpMethod: '*',
                LoggingLevel: 'INFO',
                ResourcePath: '/*'
            },
            {
                HttpMethod: '*',
                ResourcePath: '/*',
                ThrottlingRateLimit: 100,
                ThrottlingBurstLimit: 25
            }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Resource', {
        PathPart: 'my-resource'
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        Integration: {
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        'application/json': 'my-response-vtl-template',
                    },
                    StatusCode: '200'
                },
                {
                    StatusCode: '500',
                    SelectionPattern: '5\\d{2}'
                },
                {
                    StatusCode: '400',
                    SelectionPattern: '4\\d{2}'
                }
            ]
        },
        MethodResponses: [
            { StatusCode: '200' },
            { StatusCode: '500' },
            { StatusCode: '400' }
        ]
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Description: 'existing role for SageMaker integration'
    });
});
//# sourceMappingURL=data:application/json;base64,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