"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSageMakerEndpoint = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSageMakerEndpoint class.
 */
class ApiGatewayToSageMakerEndpoint extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToSageMakerEndpoint class.
     * @since 1.68.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        if (props.apiGatewayExecutionRole !== undefined) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
            });
            // Setup the IAM policy for SageMaker endpoint
            const invokePolicy = new iam.Policy(this, 'InvokeEndpointPolicy', {
                statements: [
                    new iam.PolicyStatement({
                        actions: ['sagemaker:InvokeEndpoint'],
                        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:endpoint/${props.endpointName}`]
                    })
                ]
            });
            invokePolicy.attachToRole(this.apiGatewayRole);
        }
        // Setup request validation
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-param-validator',
            // Setting this property to true makes sure the following are validated:
            // - Required request parameters in the URI
            // - Query string
            // - Headers
            validateRequestParameters: true
        });
        // Setup method and integration responses
        const methodResponses = [
            { statusCode: '200' },
            { statusCode: '500' },
            { statusCode: '400' }
        ];
        const integResponses = [];
        if (props.responseMappingTemplate !== undefined) {
            integResponses.push({
                statusCode: '200',
                responseTemplates: { 'application/json': props.responseMappingTemplate }
            });
        }
        else {
            integResponses.push({ statusCode: '200' });
        }
        integResponses.push({ statusCode: '500', selectionPattern: '5\\d{2}' }, { statusCode: '400', selectionPattern: '4\\d{2}' });
        // The SageMaker integration can be added either at the root of the API (GET https://execute-api.amazonaws.com/{some-param}),
        // or as a sub-resource (GET https://execute-api.amazonaws.com/inference/{some-param}).
        // The following lines will make sure only the necessary resources are created.
        let apiResource = this.apiGateway.root;
        if (props.resourceName !== undefined) {
            apiResource = apiResource.addResource(props.resourceName);
        }
        apiResource = apiResource.addResource(props.resourcePath);
        // Setup API Gateway method
        defaults.addProxyMethodToApiResource({
            service: 'runtime.sagemaker',
            path: `endpoints/${props.endpointName}/invocations`,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'GET',
            apiResource,
            requestValidator,
            requestTemplate: props.requestMappingTemplate,
            awsIntegrationProps: {
                options: { integrationResponses: integResponses }
            },
            methodOptions: { methodResponses }
        });
    }
}
exports.ApiGatewayToSageMakerEndpoint = ApiGatewayToSageMakerEndpoint;
//# sourceMappingURL=data:application/json;base64,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