"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentStrategy = exports.PercentOrAbsolute = exports.Deployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
const pod_1 = require("./pod");
const service_1 = require("./service");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _d, _e, _f;
        super(scope, id);
        this.apiObject = new k8s.KubeDeployment(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.replicas = (_d = props.replicas) !== null && _d !== void 0 ? _d : 1;
        this.strategy = (_e = props.strategy) !== null && _e !== void 0 ? _e : DeploymentStrategy.rollingUpdate();
        this._podTemplate = new pod_1.PodTemplate(props);
        this._labelSelector = {};
        if ((_f = props.defaultSelector) !== null && _f !== void 0 ? _f : true) {
            const selector = 'cdk8s.deployment';
            const matcher = cdk8s_1.Names.toLabelValue(this);
            this.podMetadata.addLabel(selector, matcher);
            this.selectByLabel(selector, matcher);
        }
    }
    get podMetadata() {
        return this._podTemplate.podMetadata;
    }
    /**
     * The labels this deployment will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    get containers() {
        return this._podTemplate.containers;
    }
    get initContainers() {
        return this._podTemplate.initContainers;
    }
    get hostAliases() {
        return this._podTemplate.hostAliases;
    }
    get volumes() {
        return this._podTemplate.volumes;
    }
    get restartPolicy() {
        return this._podTemplate.restartPolicy;
    }
    get serviceAccount() {
        return this._podTemplate.serviceAccount;
    }
    get securityContext() {
        return this._podTemplate.securityContext;
    }
    /**
     * Configure a label selector to this deployment.
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options Options to determine details of the service and port exposed.
     */
    exposeViaService(options = {}) {
        var _d;
        const service = new service_1.Service(this, 'Service', {
            metadata: options.name ? { name: options.name } : undefined,
            type: (_d = options.serviceType) !== null && _d !== void 0 ? _d : service_1.ServiceType.CLUSTER_IP,
        });
        service.addDeployment(this, { protocol: options.protocol, targetPort: options.targetPort, port: options.port });
        return service;
    }
    /**
     * Expose a deployment via an ingress.
     *
     * This will first expose the deployment with a service, and then expose the service via an ingress.
     *
     * @param path The ingress path to register under.
     * @param options Additional options.
     */
    exposeViaIngress(path, options = {}) {
        const service = this.exposeViaService(options);
        return service.exposeViaIngress(path, options);
    }
    addContainer(container) {
        return this._podTemplate.addContainer(container);
    }
    addInitContainer(container) {
        return this._podTemplate.addInitContainer(container);
    }
    addHostAlias(hostAlias) {
        return this._podTemplate.addHostAlias(hostAlias);
    }
    addVolume(volume) {
        return this._podTemplate.addVolume(volume);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            replicas: this.replicas,
            template: this._podTemplate._toPodTemplateSpec(),
            selector: {
                matchLabels: this._labelSelector,
            },
            strategy: this.strategy._toKube(),
        };
    }
}
exports.Deployment = Deployment;
_a = JSII_RTTI_SYMBOL_1;
Deployment[_a] = { fqn: "cdk8s-plus-22.Deployment", version: "1.0.0-beta.193" };
/**
 * Union like class repsenting either a ration in
 * percents or an absolute number.
 */
class PercentOrAbsolute {
    constructor(value) {
        this.value = value;
    }
    /**
     * Percent ratio.
     */
    static percent(percent) {
        return new PercentOrAbsolute(`${percent}%`);
    }
    /**
     * Absolute number.
     */
    static absolute(num) {
        return new PercentOrAbsolute(num);
    }
    isZero() {
        return this.value === PercentOrAbsolute.absolute(0).value || this.value === PercentOrAbsolute.percent(0).value;
    }
}
exports.PercentOrAbsolute = PercentOrAbsolute;
_b = JSII_RTTI_SYMBOL_1;
PercentOrAbsolute[_b] = { fqn: "cdk8s-plus-22.PercentOrAbsolute", version: "1.0.0-beta.193" };
/**
 * Deployment strategies.
 */
class DeploymentStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * All existing Pods are killed before new ones are created.
     *
     * @see https://kubernetes.io/docs/concepts/workloads/controllers/deployment/#recreate-deployment
     */
    static recreate() {
        return new DeploymentStrategy({
            type: 'Recreate',
        });
    }
    static rollingUpdate(options = {}) {
        var _d, _e;
        const maxSurge = (_d = options.maxSurge) !== null && _d !== void 0 ? _d : PercentOrAbsolute.percent(25);
        const maxUnavailable = (_e = options.maxSurge) !== null && _e !== void 0 ? _e : PercentOrAbsolute.percent(25);
        if (maxSurge.isZero() && maxUnavailable.isZero()) {
            throw new Error('\'maxSurge\' and \'maxUnavailable\' cannot be both zero');
        }
        return new DeploymentStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { maxSurge, maxUnavailable },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.DeploymentStrategy = DeploymentStrategy;
_c = JSII_RTTI_SYMBOL_1;
DeploymentStrategy[_c] = { fqn: "cdk8s-plus-22.DeploymentStrategy", version: "1.0.0-beta.193" };
//# sourceMappingURL=data:application/json;base64,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