"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        const createKmsKey = (_a = props.createKmsKey) !== null && _a !== void 0 ? _a : true;
        let encryptionAlias;
        if (createKmsKey) {
            const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
                targetKey: encryptionKey,
                aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
        }
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryption: encryptionAlias ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default', {
            createKmsKey: props.createKmsKey,
        });
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3Jvc3MtcmVnaW9uLXN1cHBvcnQtc3RhY2suanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjcm9zcy1yZWdpb24tc3VwcG9ydC1zdGFjay50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx3Q0FBd0M7QUFDeEMsc0NBQXNDO0FBQ3RDLHFDQUFxQztBQUVyQyxpR0FBaUc7QUFDakcsOERBQThEO0FBQzlELHdDQUEwQztBQUUxQyxNQUFNLHFCQUFxQixHQUFHLFFBQVEsQ0FBQztBQUV2Qzs7OztHQUlHO0FBQ0gsTUFBTSw2QkFBOEIsU0FBUSxHQUFHLENBQUMsS0FBSztJQUN6QyxvQkFBb0I7UUFDNUIsSUFBSSxRQUFRLEdBQUcsS0FBSyxDQUFDLG9CQUFvQixFQUFFLENBQUM7UUFDNUMsSUFBSSxRQUFRLENBQUMsVUFBVSxDQUFDLHFCQUFxQixDQUFDLEVBQUU7WUFDOUMsZ0ZBQWdGO1lBQ2hGLFFBQVEsR0FBRyxRQUFRLENBQUMsU0FBUyxDQUFDLHFCQUFxQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1NBQzdEO1FBQ0QsTUFBTSxTQUFTLEdBQUcsRUFBRSxHQUFHLHFCQUFxQixDQUFDLE1BQU0sQ0FBQztRQUNwRCxzREFBc0Q7UUFDdEQsK0NBQStDO1FBQy9DLE9BQU8scUJBQXFCLEdBQUcsZUFBZSxDQUFDLFFBQVEsRUFBRSxTQUFTLENBQUMsQ0FBQztJQUN0RSxDQUFDO0NBQ0Y7QUFFRCxTQUFTLGVBQWUsQ0FBQyxHQUFXLEVBQUUsQ0FBUztJQUM3QyxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQy9DLE9BQU8sR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsQ0FBQztBQUNuQyxDQUFDO0FBZ0JELE1BQWEsMkJBQTRCLFNBQVEsZ0JBQVM7SUFHeEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUEwQyxFQUFFOztRQUNwRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sWUFBWSxTQUFHLEtBQUssQ0FBQyxZQUFZLG1DQUFJLElBQUksQ0FBQztRQUVoRCxJQUFJLGVBQWUsQ0FBQztRQUNwQixJQUFJLFlBQVksRUFBRTtZQUNoQixNQUFNLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxFQUFFLHVEQUF1RCxFQUFFO2dCQUMvRixhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxPQUFPO2FBQ3pDLENBQUMsQ0FBQztZQUNILGVBQWUsR0FBRyxJQUFJLDZCQUE2QixDQUFDLElBQUksRUFBRSx5REFBeUQsRUFBRTtnQkFDbkgsU0FBUyxFQUFFLGFBQWE7Z0JBQ3hCLFNBQVMsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLGtCQUFrQjtnQkFDOUMsYUFBYSxFQUFFLEdBQUcsQ0FBQyxhQUFhLENBQUMsT0FBTzthQUN6QyxDQUFDLENBQUM7U0FDSjtRQUNELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLDBDQUEwQyxFQUFFO1lBQ3ZGLFVBQVUsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLGtCQUFrQjtZQUMvQyxVQUFVLEVBQUUsZUFBZSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsV0FBVztZQUN2RixhQUFhLEVBQUUsZUFBZTtTQUMvQixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUF6QkQsa0VBeUJDO0FBcUNEOzs7R0FHRztBQUNILE1BQWEsdUJBQXdCLFNBQVEsR0FBRyxDQUFDLEtBQUs7SUFNcEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFtQztRQUMzRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFNBQVMsRUFBRSxpQkFBaUIsQ0FBQyxLQUFLLENBQUM7WUFDbkMsR0FBRyxFQUFFO2dCQUNILE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtnQkFDcEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO2FBQ3ZCO1lBQ0QsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1NBQy9CLENBQUMsQ0FBQztRQUVILE1BQU0sMkJBQTJCLEdBQUcsSUFBSSwyQkFBMkIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ25GLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtTQUNqQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsMkJBQTJCLENBQUMsaUJBQWlCLENBQUM7SUFDekUsQ0FBQztDQUNGO0FBckJELDBEQXFCQztBQUVELFNBQVMsaUJBQWlCLENBQUMsS0FBbUM7SUFDNUQsT0FBTyxHQUFHLEtBQUssQ0FBQyxpQkFBaUIsWUFBWSxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7QUFDOUQsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGttcyBmcm9tICdAYXdzLWNkay9hd3Mta21zJztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbi8vIGtlZXAgdGhpcyBpbXBvcnQgc2VwYXJhdGUgZnJvbSBvdGhlciBpbXBvcnRzIHRvIHJlZHVjZSBjaGFuY2UgZm9yIG1lcmdlIGNvbmZsaWN0cyB3aXRoIHYyLW1haW5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1kdXBsaWNhdGUtaW1wb3J0cywgaW1wb3J0L29yZGVyXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuY29uc3QgUkVRVUlSRURfQUxJQVNfUFJFRklYID0gJ2FsaWFzLyc7XG5cbi8qKlxuICogQSBjbGFzcyBuZWVkZWQgdG8gd29yayBhcm91bmQgQ29kZVBpcGVsaW5lJ3MgZXh0cmVtZWx5IHNtYWxsICgxMDAgY2hhcmFjdGVycylcbiAqIGxpbWl0IGZvciB0aGUgbmFtZS9BUk4gb2YgdGhlIGtleSBpbiB0aGUgQXJ0aWZhY3RTdG9yZS5cbiAqIExpbWl0cyB0aGUgbGVuZ3RoIG9mIHRoZSBhbGlhcycgYXV0by1nZW5lcmF0ZWQgbmFtZSB0byA1MCBjaGFyYWN0ZXJzLlxuICovXG5jbGFzcyBBbGlhc1dpdGhTaG9ydGVyR2VuZXJhdGVkTmFtZSBleHRlbmRzIGttcy5BbGlhcyB7XG4gIHByb3RlY3RlZCBnZW5lcmF0ZVBoeXNpY2FsTmFtZSgpOiBzdHJpbmcge1xuICAgIGxldCBiYXNlTmFtZSA9IHN1cGVyLmdlbmVyYXRlUGh5c2ljYWxOYW1lKCk7XG4gICAgaWYgKGJhc2VOYW1lLnN0YXJ0c1dpdGgoUkVRVUlSRURfQUxJQVNfUFJFRklYKSkge1xuICAgICAgLy8gcmVtb3ZlIHRoZSBwcmVmaXgsIGJlY2F1c2Ugd2UncmUgdGFraW5nIHRoZSBsYXN0IGNoYXJhY3RlcnMgb2YgdGhlIG5hbWUgYmVsb3dcbiAgICAgIGJhc2VOYW1lID0gYmFzZU5hbWUuc3Vic3RyaW5nKFJFUVVJUkVEX0FMSUFTX1BSRUZJWC5sZW5ndGgpO1xuICAgIH1cbiAgICBjb25zdCBtYXhMZW5ndGggPSA1MCAtIFJFUVVJUkVEX0FMSUFTX1BSRUZJWC5sZW5ndGg7XG4gICAgLy8gdGFrZSB0aGUgbGFzdCBjaGFyYWN0ZXJzLCBhcyB0aGV5IGluY2x1ZGUgdGhlIGhhc2gsXG4gICAgLy8gYW5kIHNvIGhhdmUgYSBoaWdoZXIgY2hhbmNlIG9mIG5vdCBjb2xsaWRpbmdcbiAgICByZXR1cm4gUkVRVUlSRURfQUxJQVNfUFJFRklYICsgbGFzdE5DaGFyYWN0ZXJzKGJhc2VOYW1lLCBtYXhMZW5ndGgpO1xuICB9XG59XG5cbmZ1bmN0aW9uIGxhc3ROQ2hhcmFjdGVycyhzdHI6IHN0cmluZywgbjogbnVtYmVyKSB7XG4gIGNvbnN0IHN0YXJ0SW5kZXggPSBNYXRoLm1heChzdHIubGVuZ3RoIC0gbiwgMCk7XG4gIHJldHVybiBzdHIuc3Vic3RyaW5nKHN0YXJ0SW5kZXgpO1xufVxuXG4vKipcbiAqIFByb3BzIGZvciB0aGUgc3VwcG9ydCBzdGFja1xuICovXG5leHBvcnQgaW50ZXJmYWNlIENyb3NzUmVnaW9uU3VwcG9ydENvbnN0cnVjdFByb3BzIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gY3JlYXRlIHRoZSBLTVMgQ01LXG4gICAqXG4gICAqIChSZXF1aXJlZCBmb3IgY3Jvc3MtYWNjb3VudCBkZXBsb3ltZW50cylcbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY3JlYXRlS21zS2V5PzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGNsYXNzIENyb3NzUmVnaW9uU3VwcG9ydENvbnN0cnVjdCBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSByZXBsaWNhdGlvbkJ1Y2tldDogczMuSUJ1Y2tldDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ3Jvc3NSZWdpb25TdXBwb3J0Q29uc3RydWN0UHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBjcmVhdGVLbXNLZXkgPSBwcm9wcy5jcmVhdGVLbXNLZXkgPz8gdHJ1ZTtcblxuICAgIGxldCBlbmNyeXB0aW9uQWxpYXM7XG4gICAgaWYgKGNyZWF0ZUttc0tleSkge1xuICAgICAgY29uc3QgZW5jcnlwdGlvbktleSA9IG5ldyBrbXMuS2V5KHRoaXMsICdDcm9zc1JlZ2lvbkNvZGVQaXBlbGluZVJlcGxpY2F0aW9uQnVja2V0RW5jcnlwdGlvbktleScsIHtcbiAgICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIH0pO1xuICAgICAgZW5jcnlwdGlvbkFsaWFzID0gbmV3IEFsaWFzV2l0aFNob3J0ZXJHZW5lcmF0ZWROYW1lKHRoaXMsICdDcm9zc1JlZ2lvbkNvZGVQaXBlbGluZVJlcGxpY2F0aW9uQnVja2V0RW5jcnlwdGlvbkFsaWFzJywge1xuICAgICAgICB0YXJnZXRLZXk6IGVuY3J5cHRpb25LZXksXG4gICAgICAgIGFsaWFzTmFtZTogY2RrLlBoeXNpY2FsTmFtZS5HRU5FUkFURV9JRl9ORUVERUQsXG4gICAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICB9KTtcbiAgICB9XG4gICAgdGhpcy5yZXBsaWNhdGlvbkJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQodGhpcywgJ0Nyb3NzUmVnaW9uQ29kZVBpcGVsaW5lUmVwbGljYXRpb25CdWNrZXQnLCB7XG4gICAgICBidWNrZXROYW1lOiBjZGsuUGh5c2ljYWxOYW1lLkdFTkVSQVRFX0lGX05FRURFRCxcbiAgICAgIGVuY3J5cHRpb246IGVuY3J5cHRpb25BbGlhcyA/IHMzLkJ1Y2tldEVuY3J5cHRpb24uS01TIDogczMuQnVja2V0RW5jcnlwdGlvbi5LTVNfTUFOQUdFRCxcbiAgICAgIGVuY3J5cHRpb25LZXk6IGVuY3J5cHRpb25BbGlhcyxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIGZvciB7QGxpbmsgQ3Jvc3NSZWdpb25TdXBwb3J0U3RhY2t9LlxuICogVGhpcyBpbnRlcmZhY2UgaXMgcHJpdmF0ZSB0byB0aGUgYXdzLWNvZGVwaXBlbGluZSBwYWNrYWdlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENyb3NzUmVnaW9uU3VwcG9ydFN0YWNrUHJvcHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIFN0YWNrIHRoZSBQaXBlbGluZSBpdHNlbGYgYmVsb25ncyB0by5cbiAgICogVXNlZCB0byBnZW5lcmF0ZSBhIG1vcmUgZnJpZW5kbHkgbmFtZSBmb3IgdGhlIHN1cHBvcnQgU3RhY2suXG4gICAqL1xuICByZWFkb25seSBwaXBlbGluZVN0YWNrTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVdTIHJlZ2lvbiB0aGlzIFN0YWNrIHJlc2lkZXMgaW4uXG4gICAqL1xuICByZWFkb25seSByZWdpb246IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFXUyBhY2NvdW50IElEIHRoaXMgU3RhY2sgYmVsb25ncyB0by5cbiAgICpcbiAgICogQGV4YW1wbGUgJzAxMjM0NTY3ODkwMSdcbiAgICovXG4gIHJlYWRvbmx5IGFjY291bnQ6IHN0cmluZztcblxuICByZWFkb25seSBzeW50aGVzaXplcjogY2RrLklTdGFja1N5bnRoZXNpemVyIHwgdW5kZWZpbmVkO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBjcmVhdGUgYSBLTVMga2V5IGluIHRoZSBzdXBwb3J0IHN0YWNrXG4gICAqXG4gICAqIChSZXF1aXJlZCBmb3IgY3Jvc3MtYWNjb3VudCBkZXBsb3ltZW50cylcbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY3JlYXRlS21zS2V5PzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBIFN0YWNrIGNvbnRhaW5pbmcgcmVzb3VyY2VzIHJlcXVpcmVkIGZvciB0aGUgY3Jvc3MtcmVnaW9uIENvZGVQaXBlbGluZSBmdW5jdGlvbmFsaXR5IHRvIHdvcmsuXG4gKiBUaGlzIGNsYXNzIGlzIHByaXZhdGUgdG8gdGhlIGF3cy1jb2RlcGlwZWxpbmUgcGFja2FnZS5cbiAqL1xuZXhwb3J0IGNsYXNzIENyb3NzUmVnaW9uU3VwcG9ydFN0YWNrIGV4dGVuZHMgY2RrLlN0YWNrIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBTMyBCdWNrZXQgdXNlZCBmb3IgcmVwbGljYXRpbmcgdGhlIFBpcGVsaW5lJ3MgYXJ0aWZhY3RzIGludG8gdGhlIHJlZ2lvbi5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZXBsaWNhdGlvbkJ1Y2tldDogczMuSUJ1Y2tldDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ3Jvc3NSZWdpb25TdXBwb3J0U3RhY2tQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgc3RhY2tOYW1lOiBnZW5lcmF0ZVN0YWNrTmFtZShwcm9wcyksXG4gICAgICBlbnY6IHtcbiAgICAgICAgcmVnaW9uOiBwcm9wcy5yZWdpb24sXG4gICAgICAgIGFjY291bnQ6IHByb3BzLmFjY291bnQsXG4gICAgICB9LFxuICAgICAgc3ludGhlc2l6ZXI6IHByb3BzLnN5bnRoZXNpemVyLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY3Jvc3NSZWdpb25TdXBwb3J0Q29uc3RydWN0ID0gbmV3IENyb3NzUmVnaW9uU3VwcG9ydENvbnN0cnVjdCh0aGlzLCAnRGVmYXVsdCcsIHtcbiAgICAgIGNyZWF0ZUttc0tleTogcHJvcHMuY3JlYXRlS21zS2V5LFxuICAgIH0pO1xuICAgIHRoaXMucmVwbGljYXRpb25CdWNrZXQgPSBjcm9zc1JlZ2lvblN1cHBvcnRDb25zdHJ1Y3QucmVwbGljYXRpb25CdWNrZXQ7XG4gIH1cbn1cblxuZnVuY3Rpb24gZ2VuZXJhdGVTdGFja05hbWUocHJvcHM6IENyb3NzUmVnaW9uU3VwcG9ydFN0YWNrUHJvcHMpOiBzdHJpbmcge1xuICByZXR1cm4gYCR7cHJvcHMucGlwZWxpbmVTdGFja05hbWV9LXN1cHBvcnQtJHtwcm9wcy5yZWdpb259YDtcbn1cbiJdfQ==