"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodePipelineDockerfileLinterAction = void 0;
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_codepipeline_1 = require("@aws-cdk/aws-codepipeline");
const aws_codepipeline_actions_1 = require("@aws-cdk/aws-codepipeline-actions");
const aws_iam_1 = require("@aws-cdk/aws-iam");
class CodePipelineDockerfileLinterAction extends aws_codepipeline_actions_1.Action {
    constructor(props) {
        super({
            ...props,
            category: aws_codepipeline_1.ActionCategory.TEST,
            provider: 'CodeBuild',
            artifactBounds: {
                minInputs: 1,
                maxInputs: 1,
                minOutputs: 0,
                maxOutputs: 0,
            },
            inputs: [props.input],
        });
        this.props = props;
    }
    bound(scope, _stage, options) {
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_4_0;
        const version = this.props.version || 'v1.18.0';
        const project = new aws_codebuild_1.PipelineProject(scope, 'LinterProject', {
            cache: aws_codebuild_1.Cache.local(aws_codebuild_1.LocalCacheMode.DOCKER_LAYER),
            environment: {
                buildImage,
                computeType: this.props.computeType || buildImage.defaultComputeType,
                privileged: true,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    pre_build: {
                        commands: [
                            'echo Pulling the hadolint docker image',
                            `docker pull hadolint/hadolint:${version}`,
                        ],
                    },
                    build: {
                        commands: [],
                        finally: [
                            'echo Scan started on `date`',
                            `result=$(docker run --rm -i hadolint/hadolint:${version} hadolint -f json - < Dockerfile)`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'if [ "$result" != "[]" ]; then echo $result | jq .; else echo "Awesome! No findings!"; fi',
                            'echo Scan completed on `date`',
                        ],
                    },
                },
            }),
        });
        // grant the Pipeline role the required permissions to this Project
        options.role.addToPolicy(new aws_iam_1.PolicyStatement({
            resources: [project.projectArn],
            actions: [
                'codebuild:BatchGetBuilds',
                'codebuild:StartBuild',
                'codebuild:StopBuild',
            ],
        }));
        // allow the Project access to the Pipeline's artifact Bucket
        // but only if the project is not imported
        // (ie., has a role) - otherwise, the IAM library throws an error
        if (project.role) {
            options.bucket.grantRead(project);
        }
        return {
            configuration: {
                ProjectName: project.projectName,
            },
        };
    }
}
exports.CodePipelineDockerfileLinterAction = CodePipelineDockerfileLinterAction;
//# sourceMappingURL=data:application/json;base64,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