"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatefulSetUpdateStrategy = exports.StatefulSet = exports.PodManagementPolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const workload = require("./workload");
/**
 * Controls how pods are created during initial scale up, when replacing pods on nodes,
 * or when scaling down.
 *
 * The default policy is `OrderedReady`, where pods are created in increasing order
 * (pod-0, then pod-1, etc) and the controller will wait until each pod is ready before
 * continuing. When scaling down, the pods are removed in the opposite order.
 *
 * The alternative policy is `Parallel` which will create pods in parallel to match the
 * desired scale without waiting, and on scale down will delete all pods at once.
 */
var PodManagementPolicy;
(function (PodManagementPolicy) {
    PodManagementPolicy["ORDERED_READY"] = "OrderedReady";
    PodManagementPolicy["PARALLEL"] = "Parallel";
})(PodManagementPolicy = exports.PodManagementPolicy || (exports.PodManagementPolicy = {}));
/**
 * StatefulSet is the workload API object used to manage stateful applications.
 *
 * Manages the deployment and scaling of a set of Pods, and provides guarantees
 * about the ordering and uniqueness of these Pods.
 *
 * Like a Deployment, a StatefulSet manages Pods that are based on an identical
 * container spec. Unlike a Deployment, a StatefulSet maintains a sticky identity
 * for each of their Pods. These pods are created from the same spec, but are not
 * interchangeable: each has a persistent identifier that it maintains across any
 * rescheduling.
 *
 * If you want to use storage volumes to provide persistence for your workload, you
 * can use a StatefulSet as part of the solution. Although individual Pods in a StatefulSet
 * are susceptible to failure, the persistent Pod identifiers make it easier to match existing
 * volumes to the new Pods that replace any that have failed.
 *
 * Using StatefulSets
 * ------------------
 * StatefulSets are valuable for applications that require one or more of the following.
 *
 * - Stable, unique network identifiers.
 * - Stable, persistent storage.
 * - Ordered, graceful deployment and scaling.
 * - Ordered, automated rolling updates.
 */
class StatefulSet extends workload.Workload {
    constructor(scope, id, props) {
        var _c, _d, _e, _f;
        super(scope, id, props);
        this.resourceType = 'statefulsets';
        this.apiObject = new k8s.KubeStatefulSet(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this._service = props.service;
        this.apiObject.addDependency(this._service);
        this.replicas = (_c = props.replicas) !== null && _c !== void 0 ? _c : 1;
        this.strategy = (_d = props.strategy) !== null && _d !== void 0 ? _d : StatefulSetUpdateStrategy.rollingUpdate(),
            this.podManagementPolicy = (_e = props.podManagementPolicy) !== null && _e !== void 0 ? _e : PodManagementPolicy.ORDERED_READY;
        this.minReady = (_f = props.minReady) !== null && _f !== void 0 ? _f : cdk8s_1.Duration.seconds(0);
        const selectors = Object.entries(this.matchLabels);
        for (const [k, v] of selectors) {
            this._service.addSelector(k, v);
        }
    }
    /**
      * @internal
      */
    _toKube() {
        return {
            serviceName: this._service.name,
            replicas: this.replicas,
            minReadySeconds: this.minReady.toSeconds(),
            template: {
                metadata: this.podMetadata.toJson(),
                spec: this._toPodSpec(),
            },
            selector: {
                matchExpressions: this.matchExpressions,
                matchLabels: this.matchLabels,
            },
            podManagementPolicy: this.podManagementPolicy,
            updateStrategy: this.strategy._toKube(),
        };
    }
}
exports.StatefulSet = StatefulSet;
_a = JSII_RTTI_SYMBOL_1;
StatefulSet[_a] = { fqn: "cdk8s-plus-22.StatefulSet", version: "2.0.0-beta.4" };
/**
 * StatefulSet update strategies.
 */
class StatefulSetUpdateStrategy {
    constructor(strategy) {
        this.strategy = strategy;
    }
    /**
     * The controller will not automatically update the Pods in a StatefulSet.
     * Users must manually delete Pods to cause the controller to create new Pods
     * that reflect modifications.
     */
    static onDelete() {
        return new StatefulSetUpdateStrategy({
            type: 'OnDelete',
        });
    }
    /**
     * The controller will delete and recreate each Pod in the StatefulSet.
     * It will proceed in the same order as Pod termination (from the largest ordinal to the smallest),
     * updating each Pod one at a time. The Kubernetes control plane waits until an updated
     * Pod is Running and Ready prior to updating its predecessor.
     */
    static rollingUpdate(options = {}) {
        var _c;
        return new StatefulSetUpdateStrategy({
            type: 'RollingUpdate',
            rollingUpdate: { partition: (_c = options.partition) !== null && _c !== void 0 ? _c : 0 },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this.strategy;
    }
}
exports.StatefulSetUpdateStrategy = StatefulSetUpdateStrategy;
_b = JSII_RTTI_SYMBOL_1;
StatefulSetUpdateStrategy[_b] = { fqn: "cdk8s-plus-22.StatefulSetUpdateStrategy", version: "2.0.0-beta.4" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGVmdWwtc2V0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3N0YXRlZnVsLXNldC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLGlDQUFrRDtBQUVsRCxxQ0FBcUM7QUFFckMsdUNBQXVDO0FBRXZDOzs7Ozs7Ozs7O0dBVUc7QUFDSCxJQUFZLG1CQUdYO0FBSEQsV0FBWSxtQkFBbUI7SUFDN0IscURBQThCLENBQUE7SUFDOUIsNENBQXFCLENBQUE7QUFDdkIsQ0FBQyxFQUhXLG1CQUFtQixHQUFuQiwyQkFBbUIsS0FBbkIsMkJBQW1CLFFBRzlCO0FBNkNEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBeUJHO0FBQ0gsTUFBYSxXQUFZLFNBQVEsUUFBUSxDQUFDLFFBQVE7SUErQmhELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7O1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBTFYsaUJBQVksR0FBRyxjQUFjLENBQUM7UUFPNUMsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN6RCxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsSUFBSSxFQUFFLFlBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUM7U0FDbEQsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBRTlCLElBQUksQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUU1QyxJQUFJLENBQUMsUUFBUSxTQUFHLEtBQUssQ0FBQyxRQUFRLG1DQUFJLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsUUFBUSxTQUFHLEtBQUssQ0FBQyxRQUFRLG1DQUFJLHlCQUF5QixDQUFDLGFBQWEsRUFBRTtZQUMzRSxJQUFJLENBQUMsbUJBQW1CLFNBQUcsS0FBSyxDQUFDLG1CQUFtQixtQ0FBSSxtQkFBbUIsQ0FBQyxhQUFhLENBQUM7UUFDMUYsSUFBSSxDQUFDLFFBQVEsU0FBRyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxnQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUV0RCxNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNuRCxLQUFLLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLElBQUksU0FBUyxFQUFFO1lBQzlCLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztTQUNqQztJQUNILENBQUM7SUFFRDs7UUFFSTtJQUNHLE9BQU87UUFDWixPQUFPO1lBQ0wsV0FBVyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSTtZQUMvQixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDdkIsZUFBZSxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFO1lBQzFDLFFBQVEsRUFBRTtnQkFDUixRQUFRLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUU7Z0JBQ25DLElBQUksRUFBRSxJQUFJLENBQUMsVUFBVSxFQUFFO2FBQ3hCO1lBQ0QsUUFBUSxFQUFFO2dCQUNSLGdCQUFnQixFQUFFLElBQUksQ0FBQyxnQkFBZ0I7Z0JBQ3ZDLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVzthQUM5QjtZQUNELG1CQUFtQixFQUFFLElBQUksQ0FBQyxtQkFBbUI7WUFDN0MsY0FBYyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUFFO1NBQ3hDLENBQUM7SUFDSixDQUFDOztBQXhFSCxrQ0F5RUM7OztBQXdCRDs7R0FFRztBQUNILE1BQWEseUJBQXlCO0lBMkJwQyxZQUFxQyxRQUF1QztRQUF2QyxhQUFRLEdBQVIsUUFBUSxDQUErQjtJQUFHLENBQUM7SUF6QmhGOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsUUFBUTtRQUNwQixPQUFPLElBQUkseUJBQXlCLENBQUM7WUFDbkMsSUFBSSxFQUFFLFVBQVU7U0FDakIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLGFBQWEsQ0FBQyxVQUF5RCxFQUFFOztRQUVyRixPQUFPLElBQUkseUJBQXlCLENBQUM7WUFDbkMsSUFBSSxFQUFFLGVBQWU7WUFDckIsYUFBYSxFQUFFLEVBQUUsU0FBUyxRQUFFLE9BQU8sQ0FBQyxTQUFTLG1DQUFJLENBQUMsRUFBRTtTQUNyRCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBSUQ7O09BRUc7SUFDSSxPQUFPO1FBQ1osT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDO0lBQ3ZCLENBQUM7O0FBbENILDhEQW9DQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFwaU9iamVjdCwgTGF6eSwgRHVyYXRpb24gfSBmcm9tICdjZGs4cyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCAqIGFzIGs4cyBmcm9tICcuL2ltcG9ydHMvazhzJztcbmltcG9ydCAqIGFzIHNlcnZpY2UgZnJvbSAnLi9zZXJ2aWNlJztcbmltcG9ydCAqIGFzIHdvcmtsb2FkIGZyb20gJy4vd29ya2xvYWQnO1xuXG4vKipcbiAqIENvbnRyb2xzIGhvdyBwb2RzIGFyZSBjcmVhdGVkIGR1cmluZyBpbml0aWFsIHNjYWxlIHVwLCB3aGVuIHJlcGxhY2luZyBwb2RzIG9uIG5vZGVzLFxuICogb3Igd2hlbiBzY2FsaW5nIGRvd24uXG4gKlxuICogVGhlIGRlZmF1bHQgcG9saWN5IGlzIGBPcmRlcmVkUmVhZHlgLCB3aGVyZSBwb2RzIGFyZSBjcmVhdGVkIGluIGluY3JlYXNpbmcgb3JkZXJcbiAqIChwb2QtMCwgdGhlbiBwb2QtMSwgZXRjKSBhbmQgdGhlIGNvbnRyb2xsZXIgd2lsbCB3YWl0IHVudGlsIGVhY2ggcG9kIGlzIHJlYWR5IGJlZm9yZVxuICogY29udGludWluZy4gV2hlbiBzY2FsaW5nIGRvd24sIHRoZSBwb2RzIGFyZSByZW1vdmVkIGluIHRoZSBvcHBvc2l0ZSBvcmRlci5cbiAqXG4gKiBUaGUgYWx0ZXJuYXRpdmUgcG9saWN5IGlzIGBQYXJhbGxlbGAgd2hpY2ggd2lsbCBjcmVhdGUgcG9kcyBpbiBwYXJhbGxlbCB0byBtYXRjaCB0aGVcbiAqIGRlc2lyZWQgc2NhbGUgd2l0aG91dCB3YWl0aW5nLCBhbmQgb24gc2NhbGUgZG93biB3aWxsIGRlbGV0ZSBhbGwgcG9kcyBhdCBvbmNlLlxuICovXG5leHBvcnQgZW51bSBQb2RNYW5hZ2VtZW50UG9saWN5IHtcbiAgT1JERVJFRF9SRUFEWSA9ICdPcmRlcmVkUmVhZHknLFxuICBQQVJBTExFTCA9ICdQYXJhbGxlbCcsXG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgaW5pdGlhbGl6YXRpb24gb2YgYFN0YXRlZnVsU2V0YC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdGF0ZWZ1bFNldFByb3BzIGV4dGVuZHMgd29ya2xvYWQuV29ya2xvYWRQcm9wcyB7XG4gIC8qKlxuICAgKiBTZXJ2aWNlIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBzdGF0ZWZ1bHNldC5cbiAgICovXG4gIHJlYWRvbmx5IHNlcnZpY2U6IHNlcnZpY2UuU2VydmljZTtcblxuICAvKipcbiAgICAqIE51bWJlciBvZiBkZXNpcmVkIHBvZHMuXG4gICAgKlxuICAgICogQGRlZmF1bHQgMVxuICAgICovXG4gIHJlYWRvbmx5IHJlcGxpY2FzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgICogUG9kIG1hbmFnZW1lbnQgcG9saWN5IHRvIHVzZSBmb3IgdGhpcyBzdGF0ZWZ1bHNldC5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCBQb2RNYW5hZ2VtZW50UG9saWN5Lk9SREVSRURfUkVBRFlcbiAgICAqL1xuICByZWFkb25seSBwb2RNYW5hZ2VtZW50UG9saWN5PzogUG9kTWFuYWdlbWVudFBvbGljeTtcblxuICAvKipcbiAgICogSW5kaWNhdGVzIHRoZSBTdGF0ZWZ1bFNldFVwZGF0ZVN0cmF0ZWd5IHRoYXQgd2lsbCBiZSBlbXBsb3llZCB0byB1cGRhdGUgUG9kcyBpbiB0aGUgU3RhdGVmdWxTZXQgd2hlbiBhIHJldmlzaW9uIGlzIG1hZGUgdG8gVGVtcGxhdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gUm9sbGluZ1VwZGF0ZSB3aXRoIHBhcnRpdGlvbiBzZXQgdG8gMFxuICAgKi9cbiAgcmVhZG9ubHkgc3RyYXRlZ3k/OiBTdGF0ZWZ1bFNldFVwZGF0ZVN0cmF0ZWd5O1xuXG4gIC8qKlxuICAgKiBNaW5pbXVtIGR1cmF0aW9uIGZvciB3aGljaCBhIG5ld2x5IGNyZWF0ZWQgcG9kIHNob3VsZCBiZSByZWFkeSB3aXRob3V0IGFueSBvZiBpdHMgY29udGFpbmVyIGNyYXNoaW5nLFxuICAgKiBmb3IgaXQgdG8gYmUgY29uc2lkZXJlZCBhdmFpbGFibGUuIFplcm8gbWVhbnMgdGhlIHBvZCB3aWxsIGJlIGNvbnNpZGVyZWQgYXZhaWxhYmxlIGFzIHNvb24gYXMgaXQgaXMgcmVhZHkuXG4gICAqXG4gICAqIFRoaXMgaXMgYW4gYWxwaGEgZmllbGQgYW5kIHJlcXVpcmVzIGVuYWJsaW5nIFN0YXRlZnVsU2V0TWluUmVhZHlTZWNvbmRzIGZlYXR1cmUgZ2F0ZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy93b3JrbG9hZHMvY29udHJvbGxlcnMvZGVwbG95bWVudC8jbWluLXJlYWR5LXNlY29uZHNcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uc2Vjb25kcygwKVxuICAgKi9cbiAgcmVhZG9ubHkgbWluUmVhZHk/OiBEdXJhdGlvbjtcblxufVxuXG4vKipcbiAqIFN0YXRlZnVsU2V0IGlzIHRoZSB3b3JrbG9hZCBBUEkgb2JqZWN0IHVzZWQgdG8gbWFuYWdlIHN0YXRlZnVsIGFwcGxpY2F0aW9ucy5cbiAqXG4gKiBNYW5hZ2VzIHRoZSBkZXBsb3ltZW50IGFuZCBzY2FsaW5nIG9mIGEgc2V0IG9mIFBvZHMsIGFuZCBwcm92aWRlcyBndWFyYW50ZWVzXG4gKiBhYm91dCB0aGUgb3JkZXJpbmcgYW5kIHVuaXF1ZW5lc3Mgb2YgdGhlc2UgUG9kcy5cbiAqXG4gKiBMaWtlIGEgRGVwbG95bWVudCwgYSBTdGF0ZWZ1bFNldCBtYW5hZ2VzIFBvZHMgdGhhdCBhcmUgYmFzZWQgb24gYW4gaWRlbnRpY2FsXG4gKiBjb250YWluZXIgc3BlYy4gVW5saWtlIGEgRGVwbG95bWVudCwgYSBTdGF0ZWZ1bFNldCBtYWludGFpbnMgYSBzdGlja3kgaWRlbnRpdHlcbiAqIGZvciBlYWNoIG9mIHRoZWlyIFBvZHMuIFRoZXNlIHBvZHMgYXJlIGNyZWF0ZWQgZnJvbSB0aGUgc2FtZSBzcGVjLCBidXQgYXJlIG5vdFxuICogaW50ZXJjaGFuZ2VhYmxlOiBlYWNoIGhhcyBhIHBlcnNpc3RlbnQgaWRlbnRpZmllciB0aGF0IGl0IG1haW50YWlucyBhY3Jvc3MgYW55XG4gKiByZXNjaGVkdWxpbmcuXG4gKlxuICogSWYgeW91IHdhbnQgdG8gdXNlIHN0b3JhZ2Ugdm9sdW1lcyB0byBwcm92aWRlIHBlcnNpc3RlbmNlIGZvciB5b3VyIHdvcmtsb2FkLCB5b3VcbiAqIGNhbiB1c2UgYSBTdGF0ZWZ1bFNldCBhcyBwYXJ0IG9mIHRoZSBzb2x1dGlvbi4gQWx0aG91Z2ggaW5kaXZpZHVhbCBQb2RzIGluIGEgU3RhdGVmdWxTZXRcbiAqIGFyZSBzdXNjZXB0aWJsZSB0byBmYWlsdXJlLCB0aGUgcGVyc2lzdGVudCBQb2QgaWRlbnRpZmllcnMgbWFrZSBpdCBlYXNpZXIgdG8gbWF0Y2ggZXhpc3RpbmdcbiAqIHZvbHVtZXMgdG8gdGhlIG5ldyBQb2RzIHRoYXQgcmVwbGFjZSBhbnkgdGhhdCBoYXZlIGZhaWxlZC5cbiAqXG4gKiBVc2luZyBTdGF0ZWZ1bFNldHNcbiAqIC0tLS0tLS0tLS0tLS0tLS0tLVxuICogU3RhdGVmdWxTZXRzIGFyZSB2YWx1YWJsZSBmb3IgYXBwbGljYXRpb25zIHRoYXQgcmVxdWlyZSBvbmUgb3IgbW9yZSBvZiB0aGUgZm9sbG93aW5nLlxuICpcbiAqIC0gU3RhYmxlLCB1bmlxdWUgbmV0d29yayBpZGVudGlmaWVycy5cbiAqIC0gU3RhYmxlLCBwZXJzaXN0ZW50IHN0b3JhZ2UuXG4gKiAtIE9yZGVyZWQsIGdyYWNlZnVsIGRlcGxveW1lbnQgYW5kIHNjYWxpbmcuXG4gKiAtIE9yZGVyZWQsIGF1dG9tYXRlZCByb2xsaW5nIHVwZGF0ZXMuXG4gKi9cbmV4cG9ydCBjbGFzcyBTdGF0ZWZ1bFNldCBleHRlbmRzIHdvcmtsb2FkLldvcmtsb2FkIHtcbiAgLyoqXG4gICAgKiBOdW1iZXIgb2YgZGVzaXJlZCBwb2RzLlxuICAgICovXG4gIHB1YmxpYyByZWFkb25seSByZXBsaWNhczogbnVtYmVyO1xuXG4gIC8qKlxuICAgICogTWFuYWdlbWVudCBwb2xpY3kgdG8gdXNlIGZvciB0aGUgc2V0LlxuICAgICovXG4gIHB1YmxpYyByZWFkb25seSBwb2RNYW5hZ2VtZW50UG9saWN5OiBQb2RNYW5hZ2VtZW50UG9saWN5O1xuXG4gIC8qKlxuICAgKiBUaGUgdXBkYXRlIHN0YXJ0ZWd5IG9mIHRoaXMgc3RhdGVmdWwgc2V0LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN0cmF0ZWd5OiBTdGF0ZWZ1bFNldFVwZGF0ZVN0cmF0ZWd5O1xuXG4gIC8qKlxuICAgKiBNaW5pbXVtIGR1cmF0aW9uIGZvciB3aGljaCBhIG5ld2x5IGNyZWF0ZWQgcG9kIHNob3VsZCBiZSByZWFkeSB3aXRob3V0XG4gICAqIGFueSBvZiBpdHMgY29udGFpbmVyIGNyYXNoaW5nLCBmb3IgaXQgdG8gYmUgY29uc2lkZXJlZCBhdmFpbGFibGUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbWluUmVhZHk6IER1cmF0aW9uO1xuXG4gIC8qKlxuICAgICogQHNlZSBiYXNlLlJlc291cmNlLmFwaU9iamVjdFxuICAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBhcGlPYmplY3Q6IEFwaU9iamVjdDtcblxuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VUeXBlID0gJ3N0YXRlZnVsc2V0cyc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBfc2VydmljZTogc2VydmljZS5TZXJ2aWNlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTdGF0ZWZ1bFNldFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLmFwaU9iamVjdCA9IG5ldyBrOHMuS3ViZVN0YXRlZnVsU2V0KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG1ldGFkYXRhOiBwcm9wcy5tZXRhZGF0YSxcbiAgICAgIHNwZWM6IExhenkuYW55KHsgcHJvZHVjZTogKCkgPT4gdGhpcy5fdG9LdWJlKCkgfSksXG4gICAgfSk7XG4gICAgdGhpcy5fc2VydmljZSA9IHByb3BzLnNlcnZpY2U7XG5cbiAgICB0aGlzLmFwaU9iamVjdC5hZGREZXBlbmRlbmN5KHRoaXMuX3NlcnZpY2UpO1xuXG4gICAgdGhpcy5yZXBsaWNhcyA9IHByb3BzLnJlcGxpY2FzID8/IDE7XG4gICAgdGhpcy5zdHJhdGVneSA9IHByb3BzLnN0cmF0ZWd5ID8/IFN0YXRlZnVsU2V0VXBkYXRlU3RyYXRlZ3kucm9sbGluZ1VwZGF0ZSgpLFxuICAgIHRoaXMucG9kTWFuYWdlbWVudFBvbGljeSA9IHByb3BzLnBvZE1hbmFnZW1lbnRQb2xpY3kgPz8gUG9kTWFuYWdlbWVudFBvbGljeS5PUkRFUkVEX1JFQURZO1xuICAgIHRoaXMubWluUmVhZHkgPSBwcm9wcy5taW5SZWFkeSA/PyBEdXJhdGlvbi5zZWNvbmRzKDApO1xuXG4gICAgY29uc3Qgc2VsZWN0b3JzID0gT2JqZWN0LmVudHJpZXModGhpcy5tYXRjaExhYmVscyk7XG4gICAgZm9yIChjb25zdCBbaywgdl0gb2Ygc2VsZWN0b3JzKSB7XG4gICAgICB0aGlzLl9zZXJ2aWNlLmFkZFNlbGVjdG9yKGssIHYpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgICogQGludGVybmFsXG4gICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLlN0YXRlZnVsU2V0U3BlYyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHNlcnZpY2VOYW1lOiB0aGlzLl9zZXJ2aWNlLm5hbWUsXG4gICAgICByZXBsaWNhczogdGhpcy5yZXBsaWNhcyxcbiAgICAgIG1pblJlYWR5U2Vjb25kczogdGhpcy5taW5SZWFkeS50b1NlY29uZHMoKSxcbiAgICAgIHRlbXBsYXRlOiB7XG4gICAgICAgIG1ldGFkYXRhOiB0aGlzLnBvZE1ldGFkYXRhLnRvSnNvbigpLFxuICAgICAgICBzcGVjOiB0aGlzLl90b1BvZFNwZWMoKSxcbiAgICAgIH0sXG4gICAgICBzZWxlY3Rvcjoge1xuICAgICAgICBtYXRjaEV4cHJlc3Npb25zOiB0aGlzLm1hdGNoRXhwcmVzc2lvbnMsXG4gICAgICAgIG1hdGNoTGFiZWxzOiB0aGlzLm1hdGNoTGFiZWxzLFxuICAgICAgfSxcbiAgICAgIHBvZE1hbmFnZW1lbnRQb2xpY3k6IHRoaXMucG9kTWFuYWdlbWVudFBvbGljeSxcbiAgICAgIHVwZGF0ZVN0cmF0ZWd5OiB0aGlzLnN0cmF0ZWd5Ll90b0t1YmUoKSxcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYFN0YXRlZnVsU2V0VXBkYXRlU3RyYXRlZ3kucm9sbGluZ1VwZGF0ZWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3RhdGVmdWxTZXRVcGRhdGVTdHJhdGVneVJvbGxpbmdVcGRhdGVPcHRpb25zIHtcblxuICAvKipcbiAgICogSWYgc3BlY2lmaWVkLCBhbGwgUG9kcyB3aXRoIGFuIG9yZGluYWwgdGhhdCBpcyBncmVhdGVyIHRoYW4gb3IgZXF1YWwgdG8gdGhlIHBhcnRpdGlvbiB3aWxsXG4gICAqIGJlIHVwZGF0ZWQgd2hlbiB0aGUgU3RhdGVmdWxTZXQncyAuc3BlYy50ZW1wbGF0ZSBpcyB1cGRhdGVkLiBBbGwgUG9kcyB3aXRoIGFuIG9yZGluYWwgdGhhdFxuICAgKiBpcyBsZXNzIHRoYW4gdGhlIHBhcnRpdGlvbiB3aWxsIG5vdCBiZSB1cGRhdGVkLCBhbmQsIGV2ZW4gaWYgdGhleSBhcmUgZGVsZXRlZCwgdGhleSB3aWxsIGJlXG4gICAqIHJlY3JlYXRlZCBhdCB0aGUgcHJldmlvdXMgdmVyc2lvbi5cbiAgICpcbiAgICogSWYgdGhlIHBhcnRpdGlvbiBpcyBncmVhdGVyIHRoYW4gcmVwbGljYXMsIHVwZGF0ZXMgdG8gdGhlIHBvZCB0ZW1wbGF0ZSB3aWxsIG5vdCBiZSBwcm9wYWdhdGVkIHRvIFBvZHMuXG4gICAqIEluIG1vc3QgY2FzZXMgeW91IHdpbGwgbm90IG5lZWQgdG8gdXNlIGEgcGFydGl0aW9uLCBidXQgdGhleSBhcmUgdXNlZnVsIGlmIHlvdSB3YW50IHRvIHN0YWdlIGFuXG4gICAqIHVwZGF0ZSwgcm9sbCBvdXQgYSBjYW5hcnksIG9yIHBlcmZvcm0gYSBwaGFzZWQgcm9sbCBvdXQuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvd29ya2xvYWRzL2NvbnRyb2xsZXJzL3N0YXRlZnVsc2V0LyNwYXJ0aXRpb25zXG4gICAqIEBkZWZhdWx0IDBcbiAgICovXG4gIHJlYWRvbmx5IHBhcnRpdGlvbj86IG51bWJlcjtcblxufVxuXG4vKipcbiAqIFN0YXRlZnVsU2V0IHVwZGF0ZSBzdHJhdGVnaWVzLlxuICovXG5leHBvcnQgY2xhc3MgU3RhdGVmdWxTZXRVcGRhdGVTdHJhdGVneSB7XG5cbiAgLyoqXG4gICAqIFRoZSBjb250cm9sbGVyIHdpbGwgbm90IGF1dG9tYXRpY2FsbHkgdXBkYXRlIHRoZSBQb2RzIGluIGEgU3RhdGVmdWxTZXQuXG4gICAqIFVzZXJzIG11c3QgbWFudWFsbHkgZGVsZXRlIFBvZHMgdG8gY2F1c2UgdGhlIGNvbnRyb2xsZXIgdG8gY3JlYXRlIG5ldyBQb2RzXG4gICAqIHRoYXQgcmVmbGVjdCBtb2RpZmljYXRpb25zLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBvbkRlbGV0ZSgpOiBTdGF0ZWZ1bFNldFVwZGF0ZVN0cmF0ZWd5IHtcbiAgICByZXR1cm4gbmV3IFN0YXRlZnVsU2V0VXBkYXRlU3RyYXRlZ3koe1xuICAgICAgdHlwZTogJ09uRGVsZXRlJyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgY29udHJvbGxlciB3aWxsIGRlbGV0ZSBhbmQgcmVjcmVhdGUgZWFjaCBQb2QgaW4gdGhlIFN0YXRlZnVsU2V0LlxuICAgKiBJdCB3aWxsIHByb2NlZWQgaW4gdGhlIHNhbWUgb3JkZXIgYXMgUG9kIHRlcm1pbmF0aW9uIChmcm9tIHRoZSBsYXJnZXN0IG9yZGluYWwgdG8gdGhlIHNtYWxsZXN0KSxcbiAgICogdXBkYXRpbmcgZWFjaCBQb2Qgb25lIGF0IGEgdGltZS4gVGhlIEt1YmVybmV0ZXMgY29udHJvbCBwbGFuZSB3YWl0cyB1bnRpbCBhbiB1cGRhdGVkXG4gICAqIFBvZCBpcyBSdW5uaW5nIGFuZCBSZWFkeSBwcmlvciB0byB1cGRhdGluZyBpdHMgcHJlZGVjZXNzb3IuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJvbGxpbmdVcGRhdGUob3B0aW9uczogU3RhdGVmdWxTZXRVcGRhdGVTdHJhdGVneVJvbGxpbmdVcGRhdGVPcHRpb25zID0ge30pOiBTdGF0ZWZ1bFNldFVwZGF0ZVN0cmF0ZWd5IHtcblxuICAgIHJldHVybiBuZXcgU3RhdGVmdWxTZXRVcGRhdGVTdHJhdGVneSh7XG4gICAgICB0eXBlOiAnUm9sbGluZ1VwZGF0ZScsXG4gICAgICByb2xsaW5nVXBkYXRlOiB7IHBhcnRpdGlvbjogb3B0aW9ucy5wYXJ0aXRpb24gPz8gMCB9LFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHN0cmF0ZWd5OiBrOHMuU3RhdGVmdWxTZXRVcGRhdGVTdHJhdGVneSkge31cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiBrOHMuU3RhdGVmdWxTZXRVcGRhdGVTdHJhdGVneSB7XG4gICAgcmV0dXJuIHRoaXMuc3RyYXRlZ3k7XG4gIH1cblxufSJdfQ==