"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
const ingress = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base.Resource {
    constructor(scope, id, props = {}) {
        var _b, _c, _d;
        super(scope, id);
        this.resourceType = 'services';
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        }
        this._externalIPs = (_c = props.externalIPs) !== null && _c !== void 0 ? _c : [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        for (const portAndOptions of (_d = props.ports) !== null && _d !== void 0 ? _d : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     *
     * @returns The `Ingress` resource that was used.
     */
    exposeViaIngress(path, options = {}) {
        var _b;
        const ingr = (_b = options.ingress) !== null && _b !== void 0 ? _b : new ingress.Ingress(this, 'Ingress');
        ingr.addRule(path, ingress.IngressBackend.fromService(this), options.pathType);
        return ingr;
    }
    /**
     * Returns the labels which are used to select pods for this service.
     */
    get selector() {
        return this._selector;
    }
    /**
     * Ports for this service.
     *
     * Use `serve()` to expose additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Associate a deployment to this service.
     *
     * If not targetPort is specific in the portOptions, then requests will be routed
     * to the port exposed by the first container in the deployment's pods.
     * The deployment's `labelSelector` will be used to select pods.
     *
     * @param depl The deployment to expose
     * @param options Optional settings for the port.
     */
    addDeployment(depl, options = {}) {
        var _b, _c;
        const containers = depl.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        // just a PoC, we assume the first container is the main one.
        // TODO: figure out what the correct thing to do here.
        const container = containers[0];
        const port = (_b = options.port) !== null && _b !== void 0 ? _b : container.port;
        const targetPort = (_c = options.targetPort) !== null && _c !== void 0 ? _c : containers[0].port;
        if (!port) {
            throw new Error('Cannot determine port. Either pass `port` in options or configure a port on the first container of the deployment');
        }
        const selector = Object.entries(depl.matchLabels);
        if (selector.length === 0) {
            throw new Error('deployment does not have a label selector');
        }
        if (Object.keys(this.selector).length > 0) {
            throw new Error('a selector is already defined for this service. cannot add a deployment');
        }
        for (const [k, v] of selector) {
            this.addSelector(k, v);
        }
        this.serve(port, { ...options, targetPort });
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    serve(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol,
            });
        }
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: this.type,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: this.type,
            externalName: this.externalName,
        };
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-22.Service", version: "2.0.0-beta.4" };
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
//# sourceMappingURL=data:application/json;base64,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