"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GCEPersistentDiskPersistentVolume = exports.AzureDiskPersistentVolume = exports.AwsElasticBlockStorePersistentVolume = exports.PersistentVolumeReclaimPolicy = exports.PersistentVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const k8s = require("./imports/k8s");
const pvc = require("./pvc");
const volume = require("./volume");
/**
 * A PersistentVolume (PV) is a piece of storage in the cluster that has been
 * provisioned by an administrator or dynamically provisioned using Storage Classes.
 * It is a resource in the cluster just like a node is a cluster resource.
 * PVs are volume plugins like Volumes, but have a lifecycle independent of any
 * individual Pod that uses the PV. This API object captures the details of the
 * implementation of the storage, be that NFS, iSCSI, or a
 * cloud-provider-specific storage system.
 */
class PersistentVolume extends base.Resource {
    constructor(scope, id, props = {}) {
        var _e, _f;
        super(scope, id);
        this.resourceType = 'persistentvolumes';
        this.storage = props.storage;
        this.mode = (_e = props.volumeMode) !== null && _e !== void 0 ? _e : pvc.PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        this.mountOptions = props.mountOptions;
        this.reclaimPolicy = (_f = props.reclaimPolicy) !== null && _f !== void 0 ? _f : PersistentVolumeReclaimPolicy.RETAIN;
        if (props.claim) {
            this.bind(props.claim);
        }
        this.apiObject = new k8s.KubePersistentVolume(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pv from the cluster as a reference.
     * @param volumeName The name of the pv to reference.
     */
    static fromPersistentVolumeName(volumeName) {
        return {
            apiGroup: '',
            name: volumeName,
            ...k8s.KubePersistentVolume.GVK,
        };
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PVC this volume is bound to. Undefined means this volume is not yet
     * claimed by any PVC.
     */
    get claim() {
        return this._claim;
    }
    /**
     * Reserve a `PersistentVolume` by creating a `PersistentVolumeClaim`
     * that is wired to claim this volume.
     *
     * Note that this method will throw in case the volume is already claimed.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#reserving-a-persistentvolume
     */
    reserve() {
        const claim = new pvc.PersistentVolumeClaim(this, `${this.name}PVC`, {
            metadata: { name: `pvc-${this.name}`, namespace: this.metadata.namespace },
            // the storage classes must match, otherwise the claim
            // will use the default class (or no class at all), which may be different than the class
            // of this volume. note that other requirements are not needed since
            // when they are not defined, any volume satisifies them.
            storageClassName: this.storageClassName,
        });
        this.bind(claim);
        claim.bind(this);
        return claim;
    }
    /**
     * Bind a volume to a specific claim.
     * Note that you must also bind the claim to the volume.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param claim The PVC to bind to.
     */
    bind(claim) {
        if (this._claim && this._claim.name !== claim.name) {
            throw new Error(`Cannot bind volume '${this.name}' to claim '${claim.name}' since it is already bound to claim '${this._claim.name}'`);
        }
        this._claim = claim;
    }
    asVolume() {
        const claim = this.reserve();
        return volume.Volume.fromPersistentVolumeClaim(claim);
    }
    /**
     * @internal
     */
    _toKube() {
        var _e, _f, _g;
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            claimRef: this._claim ? { name: (_e = this._claim) === null || _e === void 0 ? void 0 : _e.name } : undefined,
            accessModes: (_f = this.accessModes) === null || _f === void 0 ? void 0 : _f.map(a => a.toString()),
            capacity: storage ? { storage } : undefined,
            mountOptions: (_g = this.mountOptions) === null || _g === void 0 ? void 0 : _g.map(o => o),
            storageClassName: this.storageClassName,
            persistentVolumeReclaimPolicy: this.reclaimPolicy,
            volumeMode: this.mode,
        };
    }
}
exports.PersistentVolume = PersistentVolume;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolume[_a] = { fqn: "cdk8s-plus-22.PersistentVolume", version: "2.0.0-beta.4" };
/**
 * Reclaim Policies.
 */
var PersistentVolumeReclaimPolicy;
(function (PersistentVolumeReclaimPolicy) {
    /**
     * The Retain reclaim policy allows for manual reclamation of the resource.
     * When the PersistentVolumeClaim is deleted, the PersistentVolume still exists and the
     * volume is considered "released". But it is not yet available for another claim
     * because the previous claimant's data remains on the volume.
     * An administrator can manually reclaim the volume with the following steps:
     *
     * 1. Delete the PersistentVolume. The associated storage asset in external
     *    infrastructure (such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume) still exists after the PV is deleted.
     * 2. Manually clean up the data on the associated storage asset accordingly.
     * 3. Manually delete the associated storage asset.
     *
     * If you want to reuse the same storage asset, create a new PersistentVolume
     * with the same storage asset definition.
     */
    PersistentVolumeReclaimPolicy["RETAIN"] = "Retain";
    /**
     * For volume plugins that support the Delete reclaim policy, deletion removes both the
     * PersistentVolume object from Kubernetes, as well as the associated storage asset in
     * the external infrastructure, such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume.
     * Volumes that were dynamically provisioned inherit the reclaim policy of their StorageClass, which defaults to Delete.
     * The administrator should configure the StorageClass according to users' expectations; otherwise,
     * the PV must be edited or patched after it is created
     */
    PersistentVolumeReclaimPolicy["DELETE"] = "Delete";
})(PersistentVolumeReclaimPolicy = exports.PersistentVolumeReclaimPolicy || (exports.PersistentVolumeReclaimPolicy = {}));
/**
 * Represents an AWS Disk resource that is attached to a kubelet's host machine and
 * then exposed to the pod.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 */
class AwsElasticBlockStorePersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        this.volumeId = props.volumeId;
        this.readOnly = (_e = props.readOnly) !== null && _e !== void 0 ? _e : false;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            awsElasticBlockStore: {
                volumeId: this.volumeId,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AwsElasticBlockStorePersistentVolume = AwsElasticBlockStorePersistentVolume;
_b = JSII_RTTI_SYMBOL_1;
AwsElasticBlockStorePersistentVolume[_b] = { fqn: "cdk8s-plus-22.AwsElasticBlockStorePersistentVolume", version: "2.0.0-beta.4" };
/**
 * AzureDisk represents an Azure Data Disk mount on the host and bind mount to the pod.
 */
class AzureDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f, _g, _h;
        super(scope, id, props);
        this.diskName = props.diskName;
        this.diskUri = props.diskUri;
        this.cachingMode = (_e = props.cachingMode) !== null && _e !== void 0 ? _e : volume.AzureDiskPersistentVolumeCachingMode.NONE;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.azureKind = (_g = props.kind) !== null && _g !== void 0 ? _g : volume.AzureDiskPersistentVolumeKind.SHARED;
        this.readOnly = (_h = props.readOnly) !== null && _h !== void 0 ? _h : false;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            azureDisk: {
                diskName: this.diskName,
                diskUri: this.diskUri,
                cachingMode: this.cachingMode,
                fsType: this.fsType,
                kind: this.azureKind,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AzureDiskPersistentVolume = AzureDiskPersistentVolume;
_c = JSII_RTTI_SYMBOL_1;
AzureDiskPersistentVolume[_c] = { fqn: "cdk8s-plus-22.AzureDiskPersistentVolume", version: "2.0.0-beta.4" };
/**
 * GCEPersistentDisk represents a GCE Disk resource that is attached to a kubelet's host machine
 * and then exposed to the pod. Provisioned by an admin.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#gcepersistentdisk
 */
class GCEPersistentDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        var _e, _f;
        super(scope, id, props);
        this.pdName = props.pdName;
        this.readOnly = (_e = props.readOnly) !== null && _e !== void 0 ? _e : false;
        this.fsType = (_f = props.fsType) !== null && _f !== void 0 ? _f : 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            gcePersistentDisk: {
                pdName: this.pdName,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.GCEPersistentDiskPersistentVolume = GCEPersistentDiskPersistentVolume;
_d = JSII_RTTI_SYMBOL_1;
GCEPersistentDiskPersistentVolume[_d] = { fqn: "cdk8s-plus-22.GCEPersistentDiskPersistentVolume", version: "2.0.0-beta.4" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHYuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvcHYudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBOEM7QUFFOUMsK0JBQStCO0FBQy9CLHFDQUFxQztBQUNyQyw2QkFBNkI7QUFDN0IsbUNBQW1DO0FBd0VuQzs7Ozs7Ozs7R0FRRztBQUNILE1BQWEsZ0JBQWlCLFNBQVEsSUFBSSxDQUFDLFFBQVE7SUFxRGpELFlBQXNCLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQStCLEVBQUc7O1FBQ3BGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFuQ0gsaUJBQVksR0FBRyxtQkFBbUIsQ0FBQztRQXFDakQsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxJQUFJLFNBQUcsS0FBSyxDQUFDLFVBQVUsbUNBQUksR0FBRyxDQUFDLG9CQUFvQixDQUFDLFdBQVcsQ0FBQztRQUNyRSxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1FBQy9DLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUN0QyxJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFDdkMsSUFBSSxDQUFDLGFBQWEsU0FBRyxLQUFLLENBQUMsYUFBYSxtQ0FBSSw2QkFBNkIsQ0FBQyxNQUFNLENBQUM7UUFFakYsSUFBSSxLQUFLLENBQUMsS0FBSyxFQUFFO1lBQ2YsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDeEI7UUFFRCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLG9CQUFvQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDOUQsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLElBQUksRUFBRSxZQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDO1NBQ2xELENBQUMsQ0FBQztJQUNMLENBQUM7SUFyRUQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLHdCQUF3QixDQUFDLFVBQWtCO1FBQ3ZELE9BQU87WUFDTCxRQUFRLEVBQUUsRUFBRTtZQUNaLElBQUksRUFBRSxVQUFVO1lBQ2hCLEdBQUcsR0FBRyxDQUFDLG9CQUFvQixDQUFDLEdBQUc7U0FDaEMsQ0FBQztJQUNKLENBQUM7SUE2REQ7O09BRUc7SUFDSCxJQUFXLFdBQVc7UUFDcEIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7SUFDaEUsQ0FBQztJQUVEOzs7T0FHRztJQUNILElBQVcsS0FBSztRQUNkLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQztJQUNyQixDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLE9BQU87UUFDWixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxLQUFLLEVBQUU7WUFDbkUsUUFBUSxFQUFFLEVBQUUsSUFBSSxFQUFFLE9BQU8sSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsRUFBRTtZQUUxRSxzREFBc0Q7WUFDdEQseUZBQXlGO1lBQ3pGLG9FQUFvRTtZQUNwRSx5REFBeUQ7WUFDekQsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQjtTQUN4QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ2pCLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakIsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNJLElBQUksQ0FBQyxLQUFpQztRQUMzQyxJQUFJLElBQUksQ0FBQyxNQUFNLElBQUksSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLEtBQUssS0FBSyxDQUFDLElBQUksRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixJQUFJLENBQUMsSUFBSSxlQUFlLEtBQUssQ0FBQyxJQUFJLHlDQUF5QyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUM7U0FDeEk7UUFDRCxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQztJQUN0QixDQUFDO0lBRU0sUUFBUTtRQUNiLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUM3QixPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMseUJBQXlCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDeEQsQ0FBQztJQUVEOztPQUVHO0lBQ0ksT0FBTzs7UUFDWixNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxHQUFHLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDdEcsT0FBTztZQUNMLFFBQVEsRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLElBQUksUUFBRSxJQUFJLENBQUMsTUFBTSwwQ0FBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMvRCxXQUFXLFFBQUUsSUFBSSxDQUFDLFdBQVcsMENBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ3JELFFBQVEsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDM0MsWUFBWSxRQUFFLElBQUksQ0FBQyxZQUFZLDBDQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztZQUM1QyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsZ0JBQWdCO1lBQ3ZDLDZCQUE2QixFQUFFLElBQUksQ0FBQyxhQUFhO1lBQ2pELFVBQVUsRUFBRSxJQUFJLENBQUMsSUFBSTtTQUN0QixDQUFDO0lBQ0osQ0FBQzs7QUFuSkgsNENBb0pDOzs7QUFFRDs7R0FFRztBQUNILElBQVksNkJBNkJYO0FBN0JELFdBQVksNkJBQTZCO0lBRXZDOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0gsa0RBQWlCLENBQUE7SUFFakI7Ozs7Ozs7T0FPRztJQUNILGtEQUFpQixDQUFBO0FBRW5CLENBQUMsRUE3QlcsNkJBQTZCLEdBQTdCLHFDQUE2QixLQUE3QixxQ0FBNkIsUUE2QnhDO0FBMENEOzs7OztHQUtHO0FBQ0gsTUFBYSxvQ0FBcUMsU0FBUSxnQkFBZ0I7SUFzQnhFLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBZ0Q7O1FBQ3hGLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXhCLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQztRQUMvQixJQUFJLENBQUMsUUFBUSxTQUFHLEtBQUssQ0FBQyxRQUFRLG1DQUFJLEtBQUssQ0FBQztRQUN4QyxJQUFJLENBQUMsTUFBTSxTQUFHLEtBQUssQ0FBQyxNQUFNLG1DQUFJLE1BQU0sQ0FBQztRQUNyQyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7SUFDbkMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksT0FBTztRQUNaLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUM3QixPQUFPO1lBQ0wsR0FBRyxJQUFJO1lBQ1Asb0JBQW9CLEVBQUU7Z0JBQ3BCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtnQkFDdkIsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO2dCQUNuQixTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3pCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTthQUN4QjtTQUNGLENBQUM7SUFDSixDQUFDOztBQTdDSCxvRkE4Q0M7OztBQStDRDs7R0FFRztBQUNILE1BQWEseUJBQTBCLFNBQVEsZ0JBQWdCO0lBZ0M3RCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXFDOztRQUM3RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxXQUFXLFNBQUcsS0FBSyxDQUFDLFdBQVcsbUNBQUksTUFBTSxDQUFDLG9DQUFvQyxDQUFDLElBQUksQ0FBQztRQUN6RixJQUFJLENBQUMsTUFBTSxTQUFHLEtBQUssQ0FBQyxNQUFNLG1DQUFJLE1BQU0sQ0FBQztRQUNyQyxJQUFJLENBQUMsU0FBUyxTQUFHLEtBQUssQ0FBQyxJQUFJLG1DQUFJLE1BQU0sQ0FBQyw2QkFBNkIsQ0FBQyxNQUFNLENBQUM7UUFDM0UsSUFBSSxDQUFDLFFBQVEsU0FBRyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxLQUFLLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxPQUFPO1FBQ1osTUFBTSxJQUFJLEdBQUcsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQzdCLE9BQU87WUFDTCxHQUFHLElBQUk7WUFDUCxTQUFTLEVBQUU7Z0JBQ1QsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO2dCQUN2QixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87Z0JBQ3JCLFdBQVcsRUFBRSxJQUFJLENBQUMsV0FBVztnQkFDN0IsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO2dCQUNuQixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3BCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTthQUN4QjtTQUNGLENBQUM7SUFDSixDQUFDOztBQTdESCw4REE4REM7OztBQTBDRDs7Ozs7R0FLRztBQUNILE1BQWEsaUNBQWtDLFNBQVEsZ0JBQWdCO0lBc0JyRSxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQTZDOztRQUNyRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7UUFDM0IsSUFBSSxDQUFDLFFBQVEsU0FBRyxLQUFLLENBQUMsUUFBUSxtQ0FBSSxLQUFLLENBQUM7UUFDeEMsSUFBSSxDQUFDLE1BQU0sU0FBRyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxNQUFNLENBQUM7UUFDckMsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO0lBRW5DLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksT0FBTztRQUNaLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUM3QixPQUFPO1lBQ0wsR0FBRyxJQUFJO1lBQ1AsaUJBQWlCLEVBQUU7Z0JBQ2pCLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtnQkFDbkIsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO2dCQUNuQixTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7Z0JBQ3pCLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTthQUN4QjtTQUNGLENBQUM7SUFDSixDQUFDOztBQWhESCw4RUFpREMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBcGlPYmplY3QsIExhenksIFNpemUgfSBmcm9tICdjZGs4cyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCAqIGFzIGJhc2UgZnJvbSAnLi9iYXNlJztcbmltcG9ydCAqIGFzIGs4cyBmcm9tICcuL2ltcG9ydHMvazhzJztcbmltcG9ydCAqIGFzIHB2YyBmcm9tICcuL3B2Yyc7XG5pbXBvcnQgKiBhcyB2b2x1bWUgZnJvbSAnLi92b2x1bWUnO1xuXG4vKipcbiAqIENvbnRyYWN0IG9mIGEgYFBlcnNpc3RlbnRWb2x1bWVDbGFpbWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVBlcnNpc3RlbnRWb2x1bWUgZXh0ZW5kcyBiYXNlLklSZXNvdXJjZSB7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYFBlcnNpc3RlbnRWb2x1bWVgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFBlcnNpc3RlbnRWb2x1bWVQcm9wcyBleHRlbmRzIGJhc2UuUmVzb3VyY2VQcm9wcyB7XG5cbiAgLyoqXG4gICAqIENvbnRhaW5zIGFsbCB3YXlzIHRoZSB2b2x1bWUgY2FuIGJlIG1vdW50ZWQuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS9wZXJzaXN0ZW50LXZvbHVtZXMjYWNjZXNzLW1vZGVzXG4gICAqIEBkZWZhdWx0IC0gTm8gYWNjZXNzIG1vZGVzLlxuICAgKi9cbiAgcmVhZG9ubHkgYWNjZXNzTW9kZXM/OiBwdmMuUGVyc2lzdGVudFZvbHVtZUFjY2Vzc01vZGVbXTtcblxuICAvKipcbiAgICogV2hhdCBpcyB0aGUgc3RvcmFnZSBjYXBhY2l0eSBvZiB0aGlzIHZvbHVtZS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3BlcnNpc3RlbnQtdm9sdW1lcyNyZXNvdXJjZXNcbiAgICogQGRlZmF1bHQgLSBObyBzcGVjaWZpZWQuXG4gICAqL1xuICByZWFkb25seSBzdG9yYWdlPzogU2l6ZTtcblxuICAvKipcbiAgICogUGFydCBvZiBhIGJpLWRpcmVjdGlvbmFsIGJpbmRpbmcgYmV0d2VlbiBQZXJzaXN0ZW50Vm9sdW1lIGFuZCBQZXJzaXN0ZW50Vm9sdW1lQ2xhaW0uXG4gICAqIEV4cGVjdGVkIHRvIGJlIG5vbi1uaWwgd2hlbiBib3VuZC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3BlcnNpc3RlbnQtdm9sdW1lcyNiaW5kaW5nXG4gICAqIEBkZWZhdWx0IC0gTm90IGJvdW5kIHRvIGEgc3BlY2lmaWMgY2xhaW0uXG4gICAqL1xuICByZWFkb25seSBjbGFpbT86IHB2Yy5JUGVyc2lzdGVudFZvbHVtZUNsYWltO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgbW91bnQgb3B0aW9ucywgZS5nLiBbXCJyb1wiLCBcInNvZnRcIl0uIE5vdCB2YWxpZGF0ZWQgLSBtb3VudCB3aWxsIHNpbXBseSBmYWlsIGlmIG9uZSBpcyBpbnZhbGlkLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2UvcGVyc2lzdGVudC12b2x1bWVzLyNtb3VudC1vcHRpb25zXG4gICAqIEBkZWZhdWx0IC0gTm8gb3B0aW9ucy5cbiAgICovXG4gIHJlYWRvbmx5IG1vdW50T3B0aW9ucz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBXaGVuIGEgdXNlciBpcyBkb25lIHdpdGggdGhlaXIgdm9sdW1lLCB0aGV5IGNhbiBkZWxldGUgdGhlIFBWQyBvYmplY3RzIGZyb20gdGhlIEFQSSB0aGF0XG4gICAqIGFsbG93cyByZWNsYW1hdGlvbiBvZiB0aGUgcmVzb3VyY2UuIFRoZSByZWNsYWltIHBvbGljeSB0ZWxscyB0aGUgY2x1c3RlciB3aGF0IHRvIGRvIHdpdGhcbiAgICogdGhlIHZvbHVtZSBhZnRlciBpdCBoYXMgYmVlbiByZWxlYXNlZCBvZiBpdHMgY2xhaW0uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS9wZXJzaXN0ZW50LXZvbHVtZXMjcmVjbGFpbWluZ1xuICAgKiBAZGVmYXVsdCBQZXJzaXN0ZW50Vm9sdW1lUmVjbGFpbVBvbGljeS5SRVRBSU5cbiAgICovXG4gIHJlYWRvbmx5IHJlY2xhaW1Qb2xpY3k/OiBQZXJzaXN0ZW50Vm9sdW1lUmVjbGFpbVBvbGljeTtcblxuICAvKipcbiAgICogTmFtZSBvZiBTdG9yYWdlQ2xhc3MgdG8gd2hpY2ggdGhpcyBwZXJzaXN0ZW50IHZvbHVtZSBiZWxvbmdzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFZvbHVtZSBkb2VzIG5vdCBiZWxvbmcgdG8gYW55IHN0b3JhZ2UgY2xhc3MuXG4gICAqL1xuICByZWFkb25seSBzdG9yYWdlQ2xhc3NOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBEZWZpbmVzIHdoYXQgdHlwZSBvZiB2b2x1bWUgaXMgcmVxdWlyZWQgYnkgdGhlIGNsYWltLlxuICAgKlxuICAgKiBAZGVmYXVsdCBWb2x1bWVNb2RlLkZJTEVfU1lTVEVNXG4gICAqL1xuICByZWFkb25seSB2b2x1bWVNb2RlPzogcHZjLlBlcnNpc3RlbnRWb2x1bWVNb2RlO1xuXG59XG5cbi8qKlxuICogQSBQZXJzaXN0ZW50Vm9sdW1lIChQVikgaXMgYSBwaWVjZSBvZiBzdG9yYWdlIGluIHRoZSBjbHVzdGVyIHRoYXQgaGFzIGJlZW5cbiAqIHByb3Zpc2lvbmVkIGJ5IGFuIGFkbWluaXN0cmF0b3Igb3IgZHluYW1pY2FsbHkgcHJvdmlzaW9uZWQgdXNpbmcgU3RvcmFnZSBDbGFzc2VzLlxuICogSXQgaXMgYSByZXNvdXJjZSBpbiB0aGUgY2x1c3RlciBqdXN0IGxpa2UgYSBub2RlIGlzIGEgY2x1c3RlciByZXNvdXJjZS5cbiAqIFBWcyBhcmUgdm9sdW1lIHBsdWdpbnMgbGlrZSBWb2x1bWVzLCBidXQgaGF2ZSBhIGxpZmVjeWNsZSBpbmRlcGVuZGVudCBvZiBhbnlcbiAqIGluZGl2aWR1YWwgUG9kIHRoYXQgdXNlcyB0aGUgUFYuIFRoaXMgQVBJIG9iamVjdCBjYXB0dXJlcyB0aGUgZGV0YWlscyBvZiB0aGVcbiAqIGltcGxlbWVudGF0aW9uIG9mIHRoZSBzdG9yYWdlLCBiZSB0aGF0IE5GUywgaVNDU0ksIG9yIGFcbiAqIGNsb3VkLXByb3ZpZGVyLXNwZWNpZmljIHN0b3JhZ2Ugc3lzdGVtLlxuICovXG5leHBvcnQgY2xhc3MgUGVyc2lzdGVudFZvbHVtZSBleHRlbmRzIGJhc2UuUmVzb3VyY2UgaW1wbGVtZW50cyBJUGVyc2lzdGVudFZvbHVtZSwgdm9sdW1lLklTdG9yYWdlIHtcblxuICAvKipcbiAgICogSW1wb3J0cyBhIHB2IGZyb20gdGhlIGNsdXN0ZXIgYXMgYSByZWZlcmVuY2UuXG4gICAqIEBwYXJhbSB2b2x1bWVOYW1lIFRoZSBuYW1lIG9mIHRoZSBwdiB0byByZWZlcmVuY2UuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21QZXJzaXN0ZW50Vm9sdW1lTmFtZSh2b2x1bWVOYW1lOiBzdHJpbmcpOiBJUGVyc2lzdGVudFZvbHVtZSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGFwaUdyb3VwOiAnJyxcbiAgICAgIG5hbWU6IHZvbHVtZU5hbWUsXG4gICAgICAuLi5rOHMuS3ViZVBlcnNpc3RlbnRWb2x1bWUuR1ZLLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQHNlZSBiYXNlLlJlc291cmNlLmFwaU9iamVjdFxuICAgKi9cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGFwaU9iamVjdDogQXBpT2JqZWN0O1xuXG4gIHB1YmxpYyByZWFkb25seSByZXNvdXJjZVR5cGUgPSAncGVyc2lzdGVudHZvbHVtZXMnO1xuXG4gIHByaXZhdGUgX2NsYWltPzogcHZjLklQZXJzaXN0ZW50Vm9sdW1lQ2xhaW07XG5cbiAgLyoqXG4gICAqIFN0b3JhZ2Ugc2l6ZSBvZiB0aGlzIHZvbHVtZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzdG9yYWdlPzogU2l6ZTtcblxuICAvKipcbiAgICogVm9sdW1lIG1vZGUgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbW9kZTogcHZjLlBlcnNpc3RlbnRWb2x1bWVNb2RlO1xuXG4gIC8qKlxuICAgICogU3RvcmFnZSBjbGFzcyB0aGlzIHZvbHVtZSBiZWxvbmdzIHRvLlxuICAgICovXG4gIHB1YmxpYyByZWFkb25seSBzdG9yYWdlQ2xhc3NOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBY2Nlc3MgbW9kZXMgcmVxdWlyZW1lbnQgb2YgdGhpcyBjbGFpbS5cbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgX2FjY2Vzc01vZGVzPzogcHZjLlBlcnNpc3RlbnRWb2x1bWVBY2Nlc3NNb2RlW107XG5cbiAgLyoqXG4gICAqIE1vdW50IG9wdGlvbnMgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbW91bnRPcHRpb25zPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFJlY2xhaW0gcG9saWN5IG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlY2xhaW1Qb2xpY3k6IFBlcnNpc3RlbnRWb2x1bWVSZWNsYWltUG9saWN5O1xuXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUGVyc2lzdGVudFZvbHVtZVByb3BzID0geyB9KSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuc3RvcmFnZSA9IHByb3BzLnN0b3JhZ2U7XG4gICAgdGhpcy5tb2RlID0gcHJvcHMudm9sdW1lTW9kZSA/PyBwdmMuUGVyc2lzdGVudFZvbHVtZU1vZGUuRklMRV9TWVNURU07XG4gICAgdGhpcy5zdG9yYWdlQ2xhc3NOYW1lID0gcHJvcHMuc3RvcmFnZUNsYXNzTmFtZTtcbiAgICB0aGlzLl9hY2Nlc3NNb2RlcyA9IHByb3BzLmFjY2Vzc01vZGVzO1xuICAgIHRoaXMubW91bnRPcHRpb25zID0gcHJvcHMubW91bnRPcHRpb25zO1xuICAgIHRoaXMucmVjbGFpbVBvbGljeSA9IHByb3BzLnJlY2xhaW1Qb2xpY3kgPz8gUGVyc2lzdGVudFZvbHVtZVJlY2xhaW1Qb2xpY3kuUkVUQUlOO1xuXG4gICAgaWYgKHByb3BzLmNsYWltKSB7XG4gICAgICB0aGlzLmJpbmQocHJvcHMuY2xhaW0pO1xuICAgIH1cblxuICAgIHRoaXMuYXBpT2JqZWN0ID0gbmV3IGs4cy5LdWJlUGVyc2lzdGVudFZvbHVtZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBtZXRhZGF0YTogcHJvcHMubWV0YWRhdGEsXG4gICAgICBzcGVjOiBMYXp5LmFueSh7IHByb2R1Y2U6ICgpID0+IHRoaXMuX3RvS3ViZSgpIH0pLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFjY2VzcyBtb2RlcyByZXF1aXJlbWVudCBvZiB0aGlzIGNsYWltLlxuICAgKi9cbiAgcHVibGljIGdldCBhY2Nlc3NNb2RlcygpOiBwdmMuUGVyc2lzdGVudFZvbHVtZUFjY2Vzc01vZGVbXSB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuX2FjY2Vzc01vZGVzID8gWy4uLnRoaXMuX2FjY2Vzc01vZGVzXSA6IHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBQVkMgdGhpcyB2b2x1bWUgaXMgYm91bmQgdG8uIFVuZGVmaW5lZCBtZWFucyB0aGlzIHZvbHVtZSBpcyBub3QgeWV0XG4gICAqIGNsYWltZWQgYnkgYW55IFBWQy5cbiAgICovXG4gIHB1YmxpYyBnZXQgY2xhaW0oKTogcHZjLklQZXJzaXN0ZW50Vm9sdW1lQ2xhaW0gfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLl9jbGFpbTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXNlcnZlIGEgYFBlcnNpc3RlbnRWb2x1bWVgIGJ5IGNyZWF0aW5nIGEgYFBlcnNpc3RlbnRWb2x1bWVDbGFpbWBcbiAgICogdGhhdCBpcyB3aXJlZCB0byBjbGFpbSB0aGlzIHZvbHVtZS5cbiAgICpcbiAgICogTm90ZSB0aGF0IHRoaXMgbWV0aG9kIHdpbGwgdGhyb3cgaW4gY2FzZSB0aGUgdm9sdW1lIGlzIGFscmVhZHkgY2xhaW1lZC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3BlcnNpc3RlbnQtdm9sdW1lcy8jcmVzZXJ2aW5nLWEtcGVyc2lzdGVudHZvbHVtZVxuICAgKi9cbiAgcHVibGljIHJlc2VydmUoKTogcHZjLlBlcnNpc3RlbnRWb2x1bWVDbGFpbSB7XG4gICAgY29uc3QgY2xhaW0gPSBuZXcgcHZjLlBlcnNpc3RlbnRWb2x1bWVDbGFpbSh0aGlzLCBgJHt0aGlzLm5hbWV9UFZDYCwge1xuICAgICAgbWV0YWRhdGE6IHsgbmFtZTogYHB2Yy0ke3RoaXMubmFtZX1gLCBuYW1lc3BhY2U6IHRoaXMubWV0YWRhdGEubmFtZXNwYWNlIH0sXG5cbiAgICAgIC8vIHRoZSBzdG9yYWdlIGNsYXNzZXMgbXVzdCBtYXRjaCwgb3RoZXJ3aXNlIHRoZSBjbGFpbVxuICAgICAgLy8gd2lsbCB1c2UgdGhlIGRlZmF1bHQgY2xhc3MgKG9yIG5vIGNsYXNzIGF0IGFsbCksIHdoaWNoIG1heSBiZSBkaWZmZXJlbnQgdGhhbiB0aGUgY2xhc3NcbiAgICAgIC8vIG9mIHRoaXMgdm9sdW1lLiBub3RlIHRoYXQgb3RoZXIgcmVxdWlyZW1lbnRzIGFyZSBub3QgbmVlZGVkIHNpbmNlXG4gICAgICAvLyB3aGVuIHRoZXkgYXJlIG5vdCBkZWZpbmVkLCBhbnkgdm9sdW1lIHNhdGlzaWZpZXMgdGhlbS5cbiAgICAgIHN0b3JhZ2VDbGFzc05hbWU6IHRoaXMuc3RvcmFnZUNsYXNzTmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMuYmluZChjbGFpbSk7XG4gICAgY2xhaW0uYmluZCh0aGlzKTtcblxuICAgIHJldHVybiBjbGFpbTtcbiAgfVxuXG4gIC8qKlxuICAgKiBCaW5kIGEgdm9sdW1lIHRvIGEgc3BlY2lmaWMgY2xhaW0uXG4gICAqIE5vdGUgdGhhdCB5b3UgbXVzdCBhbHNvIGJpbmQgdGhlIGNsYWltIHRvIHRoZSB2b2x1bWUuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS9wZXJzaXN0ZW50LXZvbHVtZXMvI2JpbmRpbmdcbiAgICpcbiAgICogQHBhcmFtIGNsYWltIFRoZSBQVkMgdG8gYmluZCB0by5cbiAgICovXG4gIHB1YmxpYyBiaW5kKGNsYWltOiBwdmMuSVBlcnNpc3RlbnRWb2x1bWVDbGFpbSkge1xuICAgIGlmICh0aGlzLl9jbGFpbSAmJiB0aGlzLl9jbGFpbS5uYW1lICE9PSBjbGFpbS5uYW1lKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBiaW5kIHZvbHVtZSAnJHt0aGlzLm5hbWV9JyB0byBjbGFpbSAnJHtjbGFpbS5uYW1lfScgc2luY2UgaXQgaXMgYWxyZWFkeSBib3VuZCB0byBjbGFpbSAnJHt0aGlzLl9jbGFpbS5uYW1lfSdgKTtcbiAgICB9XG4gICAgdGhpcy5fY2xhaW0gPSBjbGFpbTtcbiAgfVxuXG4gIHB1YmxpYyBhc1ZvbHVtZSgpOiB2b2x1bWUuVm9sdW1lIHtcbiAgICBjb25zdCBjbGFpbSA9IHRoaXMucmVzZXJ2ZSgpO1xuICAgIHJldHVybiB2b2x1bWUuVm9sdW1lLmZyb21QZXJzaXN0ZW50Vm9sdW1lQ2xhaW0oY2xhaW0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLlBlcnNpc3RlbnRWb2x1bWVTcGVjIHtcbiAgICBjb25zdCBzdG9yYWdlID0gdGhpcy5zdG9yYWdlID8gazhzLlF1YW50aXR5LmZyb21TdHJpbmcodGhpcy5zdG9yYWdlLnRvR2liaWJ5dGVzKCkgKyAnR2knKSA6IHVuZGVmaW5lZDtcbiAgICByZXR1cm4ge1xuICAgICAgY2xhaW1SZWY6IHRoaXMuX2NsYWltID8geyBuYW1lOiB0aGlzLl9jbGFpbT8ubmFtZSB9IDogdW5kZWZpbmVkLFxuICAgICAgYWNjZXNzTW9kZXM6IHRoaXMuYWNjZXNzTW9kZXM/Lm1hcChhID0+IGEudG9TdHJpbmcoKSksXG4gICAgICBjYXBhY2l0eTogc3RvcmFnZSA/IHsgc3RvcmFnZSB9IDogdW5kZWZpbmVkLFxuICAgICAgbW91bnRPcHRpb25zOiB0aGlzLm1vdW50T3B0aW9ucz8ubWFwKG8gPT4gbyksXG4gICAgICBzdG9yYWdlQ2xhc3NOYW1lOiB0aGlzLnN0b3JhZ2VDbGFzc05hbWUsXG4gICAgICBwZXJzaXN0ZW50Vm9sdW1lUmVjbGFpbVBvbGljeTogdGhpcy5yZWNsYWltUG9saWN5LFxuICAgICAgdm9sdW1lTW9kZTogdGhpcy5tb2RlLFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBSZWNsYWltIFBvbGljaWVzLlxuICovXG5leHBvcnQgZW51bSBQZXJzaXN0ZW50Vm9sdW1lUmVjbGFpbVBvbGljeSB7XG5cbiAgLyoqXG4gICAqIFRoZSBSZXRhaW4gcmVjbGFpbSBwb2xpY3kgYWxsb3dzIGZvciBtYW51YWwgcmVjbGFtYXRpb24gb2YgdGhlIHJlc291cmNlLlxuICAgKiBXaGVuIHRoZSBQZXJzaXN0ZW50Vm9sdW1lQ2xhaW0gaXMgZGVsZXRlZCwgdGhlIFBlcnNpc3RlbnRWb2x1bWUgc3RpbGwgZXhpc3RzIGFuZCB0aGVcbiAgICogdm9sdW1lIGlzIGNvbnNpZGVyZWQgXCJyZWxlYXNlZFwiLiBCdXQgaXQgaXMgbm90IHlldCBhdmFpbGFibGUgZm9yIGFub3RoZXIgY2xhaW1cbiAgICogYmVjYXVzZSB0aGUgcHJldmlvdXMgY2xhaW1hbnQncyBkYXRhIHJlbWFpbnMgb24gdGhlIHZvbHVtZS5cbiAgICogQW4gYWRtaW5pc3RyYXRvciBjYW4gbWFudWFsbHkgcmVjbGFpbSB0aGUgdm9sdW1lIHdpdGggdGhlIGZvbGxvd2luZyBzdGVwczpcbiAgICpcbiAgICogMS4gRGVsZXRlIHRoZSBQZXJzaXN0ZW50Vm9sdW1lLiBUaGUgYXNzb2NpYXRlZCBzdG9yYWdlIGFzc2V0IGluIGV4dGVybmFsXG4gICAqICAgIGluZnJhc3RydWN0dXJlIChzdWNoIGFzIGFuIEFXUyBFQlMsIEdDRSBQRCwgQXp1cmUgRGlzaywgb3IgQ2luZGVyIHZvbHVtZSkgc3RpbGwgZXhpc3RzIGFmdGVyIHRoZSBQViBpcyBkZWxldGVkLlxuICAgKiAyLiBNYW51YWxseSBjbGVhbiB1cCB0aGUgZGF0YSBvbiB0aGUgYXNzb2NpYXRlZCBzdG9yYWdlIGFzc2V0IGFjY29yZGluZ2x5LlxuICAgKiAzLiBNYW51YWxseSBkZWxldGUgdGhlIGFzc29jaWF0ZWQgc3RvcmFnZSBhc3NldC5cbiAgICpcbiAgICogSWYgeW91IHdhbnQgdG8gcmV1c2UgdGhlIHNhbWUgc3RvcmFnZSBhc3NldCwgY3JlYXRlIGEgbmV3IFBlcnNpc3RlbnRWb2x1bWVcbiAgICogd2l0aCB0aGUgc2FtZSBzdG9yYWdlIGFzc2V0IGRlZmluaXRpb24uXG4gICAqL1xuICBSRVRBSU4gPSAnUmV0YWluJyxcblxuICAvKipcbiAgICogRm9yIHZvbHVtZSBwbHVnaW5zIHRoYXQgc3VwcG9ydCB0aGUgRGVsZXRlIHJlY2xhaW0gcG9saWN5LCBkZWxldGlvbiByZW1vdmVzIGJvdGggdGhlXG4gICAqIFBlcnNpc3RlbnRWb2x1bWUgb2JqZWN0IGZyb20gS3ViZXJuZXRlcywgYXMgd2VsbCBhcyB0aGUgYXNzb2NpYXRlZCBzdG9yYWdlIGFzc2V0IGluXG4gICAqIHRoZSBleHRlcm5hbCBpbmZyYXN0cnVjdHVyZSwgc3VjaCBhcyBhbiBBV1MgRUJTLCBHQ0UgUEQsIEF6dXJlIERpc2ssIG9yIENpbmRlciB2b2x1bWUuXG4gICAqIFZvbHVtZXMgdGhhdCB3ZXJlIGR5bmFtaWNhbGx5IHByb3Zpc2lvbmVkIGluaGVyaXQgdGhlIHJlY2xhaW0gcG9saWN5IG9mIHRoZWlyIFN0b3JhZ2VDbGFzcywgd2hpY2ggZGVmYXVsdHMgdG8gRGVsZXRlLlxuICAgKiBUaGUgYWRtaW5pc3RyYXRvciBzaG91bGQgY29uZmlndXJlIHRoZSBTdG9yYWdlQ2xhc3MgYWNjb3JkaW5nIHRvIHVzZXJzJyBleHBlY3RhdGlvbnM7IG90aGVyd2lzZSxcbiAgICogdGhlIFBWIG11c3QgYmUgZWRpdGVkIG9yIHBhdGNoZWQgYWZ0ZXIgaXQgaXMgY3JlYXRlZFxuICAgKi9cbiAgREVMRVRFID0gJ0RlbGV0ZScsXG5cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBgQXdzRWxhc3RpY0Jsb2NrU3RvcmVQZXJzaXN0ZW50Vm9sdW1lYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBd3NFbGFzdGljQmxvY2tTdG9yZVBlcnNpc3RlbnRWb2x1bWVQcm9wcyBleHRlbmRzIFBlcnNpc3RlbnRWb2x1bWVQcm9wcyB7XG5cbiAgLyoqXG4gICAqIFVuaXF1ZSBJRCBvZiB0aGUgcGVyc2lzdGVudCBkaXNrIHJlc291cmNlIGluIEFXUyAoQW1hem9uIEVCUyB2b2x1bWUpLiBNb3JlIGluZm86IGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2Uvdm9sdW1lcyNhd3NlbGFzdGljYmxvY2tzdG9yZVxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2Uvdm9sdW1lcyNhd3NlbGFzdGljYmxvY2tzdG9yZVxuICAgKi9cbiAgcmVhZG9ubHkgdm9sdW1lSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogRmlsZXN5c3RlbSB0eXBlIG9mIHRoZSB2b2x1bWUgdGhhdCB5b3Ugd2FudCB0byBtb3VudC5cbiAgICogVGlwOiBFbnN1cmUgdGhhdCB0aGUgZmlsZXN5c3RlbSB0eXBlIGlzIHN1cHBvcnRlZCBieSB0aGUgaG9zdCBvcGVyYXRpbmcgc3lzdGVtLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2Uvdm9sdW1lcyNhd3NlbGFzdGljYmxvY2tzdG9yZVxuICAgKiBAZGVmYXVsdCAnZXh0NCdcbiAgICovXG4gIHJlYWRvbmx5IGZzVHlwZT86IHN0cmluZztcblxuICAvKipcbiAgICAqIFRoZSBwYXJ0aXRpb24gaW4gdGhlIHZvbHVtZSB0aGF0IHlvdSB3YW50IHRvIG1vdW50LiBJZiBvbWl0dGVkLCB0aGUgZGVmYXVsdCBpcyB0byBtb3VudCBieSB2b2x1bWUgbmFtZS5cbiAgICAqIEV4YW1wbGVzOiBGb3Igdm9sdW1lIC9kZXYvc2RhMSwgeW91IHNwZWNpZnkgdGhlIHBhcnRpdGlvbiBhcyBcIjFcIi5cbiAgICAqIFNpbWlsYXJseSwgdGhlIHZvbHVtZSBwYXJ0aXRpb24gZm9yIC9kZXYvc2RhIGlzIFwiMFwiIChvciB5b3UgY2FuIGxlYXZlIHRoZSBwcm9wZXJ0eSBlbXB0eSkuXG4gICAgKlxuICAgICogQGRlZmF1bHQgLSBObyBwYXJ0aXRpb24uXG4gICAgKi9cbiAgcmVhZG9ubHkgcGFydGl0aW9uPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgICogU3BlY2lmeSBcInRydWVcIiB0byBmb3JjZSBhbmQgc2V0IHRoZSBSZWFkT25seSBwcm9wZXJ0eSBpbiBWb2x1bWVNb3VudHMgdG8gXCJ0cnVlXCIuXG4gICAgKlxuICAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3ZvbHVtZXMjYXdzZWxhc3RpY2Jsb2Nrc3RvcmVcbiAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgKi9cbiAgcmVhZG9ubHkgcmVhZE9ubHk/OiBib29sZWFuO1xuXG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhbiBBV1MgRGlzayByZXNvdXJjZSB0aGF0IGlzIGF0dGFjaGVkIHRvIGEga3ViZWxldCdzIGhvc3QgbWFjaGluZSBhbmRcbiAqIHRoZW4gZXhwb3NlZCB0byB0aGUgcG9kLlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS92b2x1bWVzI2F3c2VsYXN0aWNibG9ja3N0b3JlXG4gKi9cbmV4cG9ydCBjbGFzcyBBd3NFbGFzdGljQmxvY2tTdG9yZVBlcnNpc3RlbnRWb2x1bWUgZXh0ZW5kcyBQZXJzaXN0ZW50Vm9sdW1lIHtcblxuICAvKipcbiAgICogVm9sdW1lIGlkIG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZvbHVtZUlkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IGl0IGlzIG1vdW50ZWQgYXMgYSByZWFkLW9ubHkgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHJlYWRPbmx5OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBGaWxlIHN5c3RlbSB0eXBlIG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGZzVHlwZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQYXJ0aXRpb24gb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcGFydGl0aW9uPzogbnVtYmVyO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBd3NFbGFzdGljQmxvY2tTdG9yZVBlcnNpc3RlbnRWb2x1bWVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuXG4gICAgdGhpcy52b2x1bWVJZCA9IHByb3BzLnZvbHVtZUlkO1xuICAgIHRoaXMucmVhZE9ubHkgPSBwcm9wcy5yZWFkT25seSA/PyBmYWxzZTtcbiAgICB0aGlzLmZzVHlwZSA9IHByb3BzLmZzVHlwZSA/PyAnZXh0NCc7XG4gICAgdGhpcy5wYXJ0aXRpb24gPSBwcm9wcy5wYXJ0aXRpb247XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiBrOHMuUGVyc2lzdGVudFZvbHVtZVNwZWMge1xuICAgIGNvbnN0IHNwZWMgPSBzdXBlci5fdG9LdWJlKCk7XG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLnNwZWMsXG4gICAgICBhd3NFbGFzdGljQmxvY2tTdG9yZToge1xuICAgICAgICB2b2x1bWVJZDogdGhpcy52b2x1bWVJZCxcbiAgICAgICAgZnNUeXBlOiB0aGlzLmZzVHlwZSxcbiAgICAgICAgcGFydGl0aW9uOiB0aGlzLnBhcnRpdGlvbixcbiAgICAgICAgcmVhZE9ubHk6IHRoaXMucmVhZE9ubHksXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBgQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZVByb3BzIGV4dGVuZHMgUGVyc2lzdGVudFZvbHVtZVByb3BzIHtcblxuICAvKipcbiAgICogVGhlIE5hbWUgb2YgdGhlIGRhdGEgZGlzayBpbiB0aGUgYmxvYiBzdG9yYWdlXG4gICAqL1xuICByZWFkb25seSBkaXNrTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgVVJJIHRoZSBkYXRhIGRpc2sgaW4gdGhlIGJsb2Igc3RvcmFnZVxuICAgKi9cbiAgcmVhZG9ubHkgZGlza1VyaTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBIb3N0IENhY2hpbmcgbW9kZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lQ2FjaGluZ01vZGUuTk9ORS5cbiAgICovXG4gIHJlYWRvbmx5IGNhY2hpbmdNb2RlPzogdm9sdW1lLkF6dXJlRGlza1BlcnNpc3RlbnRWb2x1bWVDYWNoaW5nTW9kZTtcblxuICAvKipcbiAgICAqIEZpbGVzeXN0ZW0gdHlwZSB0byBtb3VudC4gTXVzdCBiZSBhIGZpbGVzeXN0ZW0gdHlwZSBzdXBwb3J0ZWQgYnkgdGhlIGhvc3Qgb3BlcmF0aW5nIHN5c3RlbS5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCAnZXh0NCdcbiAgICAqL1xuICByZWFkb25seSBmc1R5cGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAgKiBLaW5kIG9mIGRpc2suXG4gICAgKlxuICAgICogQGRlZmF1bHQgQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZUtpbmQuU0hBUkVEXG4gICAgKi9cbiAgcmVhZG9ubHkga2luZD86IHZvbHVtZS5BenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lS2luZDtcblxuICAvKipcbiAgICAqIEZvcmNlIHRoZSBSZWFkT25seSBzZXR0aW5nIGluIFZvbHVtZU1vdW50cy5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgICovXG4gIHJlYWRvbmx5IHJlYWRPbmx5PzogYm9vbGVhbjtcblxufVxuXG4vKipcbiAqIEF6dXJlRGlzayByZXByZXNlbnRzIGFuIEF6dXJlIERhdGEgRGlzayBtb3VudCBvbiB0aGUgaG9zdCBhbmQgYmluZCBtb3VudCB0byB0aGUgcG9kLlxuICovXG5leHBvcnQgY2xhc3MgQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZSBleHRlbmRzIFBlcnNpc3RlbnRWb2x1bWUge1xuXG4gIC8qKlxuICAgKiBEaXNrIG5hbWUgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGlza05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogRGlzayBVUkkgb2YgdGhpcyB2b2x1bWUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGlza1VyaTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCBpdCBpcyBtb3VudGVkIGFzIGEgcmVhZC1vbmx5IHZvbHVtZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZWFkT25seTogYm9vbGVhbjtcblxuICAvKipcbiAgICogQ2FjaGluZyBtb2RlIG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNhY2hpbmdNb2RlOiB2b2x1bWUuQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZUNhY2hpbmdNb2RlO1xuXG4gIC8qKlxuICAgKiBGaWxlIHN5c3RlbSB0eXBlIG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGZzVHlwZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBenVyZSBraW5kIG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGF6dXJlS2luZDogdm9sdW1lLkF6dXJlRGlza1BlcnNpc3RlbnRWb2x1bWVLaW5kO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBenVyZURpc2tQZXJzaXN0ZW50Vm9sdW1lUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIHRoaXMuZGlza05hbWUgPSBwcm9wcy5kaXNrTmFtZTtcbiAgICB0aGlzLmRpc2tVcmkgPSBwcm9wcy5kaXNrVXJpO1xuICAgIHRoaXMuY2FjaGluZ01vZGUgPSBwcm9wcy5jYWNoaW5nTW9kZSA/PyB2b2x1bWUuQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZUNhY2hpbmdNb2RlLk5PTkU7XG4gICAgdGhpcy5mc1R5cGUgPSBwcm9wcy5mc1R5cGUgPz8gJ2V4dDQnO1xuICAgIHRoaXMuYXp1cmVLaW5kID0gcHJvcHMua2luZCA/PyB2b2x1bWUuQXp1cmVEaXNrUGVyc2lzdGVudFZvbHVtZUtpbmQuU0hBUkVEO1xuICAgIHRoaXMucmVhZE9ubHkgPSBwcm9wcy5yZWFkT25seSA/PyBmYWxzZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICpcbiAgICogQHNlZSBodHRwczovL2dpdGh1Yi5jb20va3ViZXJuZXRlcy9leGFtcGxlcy9ibG9iL21hc3Rlci9zdGFnaW5nL3ZvbHVtZXMvYXp1cmVfZGlzay9SRUFETUUubWRcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5QZXJzaXN0ZW50Vm9sdW1lU3BlYyB7XG4gICAgY29uc3Qgc3BlYyA9IHN1cGVyLl90b0t1YmUoKTtcbiAgICByZXR1cm4ge1xuICAgICAgLi4uc3BlYyxcbiAgICAgIGF6dXJlRGlzazoge1xuICAgICAgICBkaXNrTmFtZTogdGhpcy5kaXNrTmFtZSxcbiAgICAgICAgZGlza1VyaTogdGhpcy5kaXNrVXJpLFxuICAgICAgICBjYWNoaW5nTW9kZTogdGhpcy5jYWNoaW5nTW9kZSxcbiAgICAgICAgZnNUeXBlOiB0aGlzLmZzVHlwZSxcbiAgICAgICAga2luZDogdGhpcy5henVyZUtpbmQsXG4gICAgICAgIHJlYWRPbmx5OiB0aGlzLnJlYWRPbmx5LFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYEdDRVBlcnNpc3RlbnREaXNrUGVyc2lzdGVudFZvbHVtZWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgR0NFUGVyc2lzdGVudERpc2tQZXJzaXN0ZW50Vm9sdW1lUHJvcHMgZXh0ZW5kcyBQZXJzaXN0ZW50Vm9sdW1lUHJvcHMge1xuXG4gIC8qKlxuICAgKiBVbmlxdWUgbmFtZSBvZiB0aGUgUEQgcmVzb3VyY2UgaW4gR0NFLiBVc2VkIHRvIGlkZW50aWZ5IHRoZSBkaXNrIGluIEdDRS5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zdG9yYWdlL3ZvbHVtZXMjZ2NlcGVyc2lzdGVudGRpc2tcbiAgICovXG4gIHJlYWRvbmx5IHBkTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBGaWxlc3lzdGVtIHR5cGUgb2YgdGhlIHZvbHVtZSB0aGF0IHlvdSB3YW50IHRvIG1vdW50LlxuICAgKiBUaXA6IEVuc3VyZSB0aGF0IHRoZSBmaWxlc3lzdGVtIHR5cGUgaXMgc3VwcG9ydGVkIGJ5IHRoZSBob3N0IG9wZXJhdGluZyBzeXN0ZW0uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS92b2x1bWVzI2F3c2VsYXN0aWNibG9ja3N0b3JlXG4gICAqIEBkZWZhdWx0ICdleHQ0J1xuICAgKi9cbiAgcmVhZG9ubHkgZnNUeXBlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGFydGl0aW9uIGluIHRoZSB2b2x1bWUgdGhhdCB5b3Ugd2FudCB0byBtb3VudC4gSWYgb21pdHRlZCwgdGhlIGRlZmF1bHQgaXMgdG8gbW91bnQgYnkgdm9sdW1lIG5hbWUuXG4gICAqIEV4YW1wbGVzOiBGb3Igdm9sdW1lIC9kZXYvc2RhMSwgeW91IHNwZWNpZnkgdGhlIHBhcnRpdGlvbiBhcyBcIjFcIi5cbiAgICogU2ltaWxhcmx5LCB0aGUgdm9sdW1lIHBhcnRpdGlvbiBmb3IgL2Rldi9zZGEgaXMgXCIwXCIgKG9yIHlvdSBjYW4gbGVhdmUgdGhlIHByb3BlcnR5IGVtcHR5KS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwYXJ0aXRpb24uXG4gICAqL1xuICByZWFkb25seSBwYXJ0aXRpb24/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgXCJ0cnVlXCIgdG8gZm9yY2UgYW5kIHNldCB0aGUgUmVhZE9ubHkgcHJvcGVydHkgaW4gVm9sdW1lTW91bnRzIHRvIFwidHJ1ZVwiLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL3N0b3JhZ2Uvdm9sdW1lcyNhd3NlbGFzdGljYmxvY2tzdG9yZVxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVhZE9ubHk/OiBib29sZWFuO1xuXG59XG5cbi8qKlxuICogR0NFUGVyc2lzdGVudERpc2sgcmVwcmVzZW50cyBhIEdDRSBEaXNrIHJlc291cmNlIHRoYXQgaXMgYXR0YWNoZWQgdG8gYSBrdWJlbGV0J3MgaG9zdCBtYWNoaW5lXG4gKiBhbmQgdGhlbiBleHBvc2VkIHRvIHRoZSBwb2QuIFByb3Zpc2lvbmVkIGJ5IGFuIGFkbWluLlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc3RvcmFnZS92b2x1bWVzI2djZXBlcnNpc3RlbnRkaXNrXG4gKi9cbmV4cG9ydCBjbGFzcyBHQ0VQZXJzaXN0ZW50RGlza1BlcnNpc3RlbnRWb2x1bWUgZXh0ZW5kcyBQZXJzaXN0ZW50Vm9sdW1lIHtcblxuICAvKipcbiAgICogUEQgcmVzb3VyY2UgaW4gR0NFIG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBkTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCBpdCBpcyBtb3VudGVkIGFzIGEgcmVhZC1vbmx5IHZvbHVtZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZWFkT25seTogYm9vbGVhbjtcblxuICAvKipcbiAgICogRmlsZSBzeXN0ZW0gdHlwZSBvZiB0aGlzIHZvbHVtZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBmc1R5cGU6IHN0cmluZztcblxuICAvKipcbiAgICogUGFydGl0aW9uIG9mIHRoaXMgdm9sdW1lLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBhcnRpdGlvbj86IG51bWJlcjtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogR0NFUGVyc2lzdGVudERpc2tQZXJzaXN0ZW50Vm9sdW1lUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIHRoaXMucGROYW1lID0gcHJvcHMucGROYW1lO1xuICAgIHRoaXMucmVhZE9ubHkgPSBwcm9wcy5yZWFkT25seSA/PyBmYWxzZTtcbiAgICB0aGlzLmZzVHlwZSA9IHByb3BzLmZzVHlwZSA/PyAnZXh0NCc7XG4gICAgdGhpcy5wYXJ0aXRpb24gPSBwcm9wcy5wYXJ0aXRpb247XG5cbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICpcbiAgICogQHNlZSBodHRwczovL2dpdGh1Yi5jb20va3ViZXJuZXRlcy9leGFtcGxlcy9ibG9iL21hc3Rlci9zdGFnaW5nL3ZvbHVtZXMvYXp1cmVfZGlzay9SRUFETUUubWRcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5QZXJzaXN0ZW50Vm9sdW1lU3BlYyB7XG4gICAgY29uc3Qgc3BlYyA9IHN1cGVyLl90b0t1YmUoKTtcbiAgICByZXR1cm4ge1xuICAgICAgLi4uc3BlYyxcbiAgICAgIGdjZVBlcnNpc3RlbnREaXNrOiB7XG4gICAgICAgIHBkTmFtZTogdGhpcy5wZE5hbWUsXG4gICAgICAgIGZzVHlwZTogdGhpcy5mc1R5cGUsXG4gICAgICAgIHBhcnRpdGlvbjogdGhpcy5wYXJ0aXRpb24sXG4gICAgICAgIHJlYWRPbmx5OiB0aGlzLnJlYWRPbmx5LFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59Il19