"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        var _e;
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = (_e = props.automountServiceAccountToken) !== null && _e !== void 0 ? _e : true;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(cont) {
        var _e;
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: (_e = cont.name) !== null && _e !== void 0 ? _e : `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @internal
     */
    _toPodSpec() {
        var _e;
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy,
            serviceAccountName: (_e = this.serviceAccount) === null || _e === void 0 ? void 0 : _e.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
        };
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-22.AbstractPod", version: "2.0.0-beta.4" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return this._toPodSpec();
    }
}
exports.Pod = Pod;
_b = JSII_RTTI_SYMBOL_1;
Pod[_b] = { fqn: "cdk8s-plus-22.Pod", version: "2.0.0-beta.4" };
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        var _e, _f, _g, _h, _j;
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = (_e = props.policy) !== null && _e !== void 0 ? _e : DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = (_f = props.hostnameAsFQDN) !== null && _f !== void 0 ? _f : false;
        this._nameservers = (_g = props.nameservers) !== null && _g !== void 0 ? _g : [];
        this._searches = (_h = props.searches) !== null && _h !== void 0 ? _h : [];
        this._options = (_j = props.options) !== null && _j !== void 0 ? _j : [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this.policy,
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
}
exports.PodDns = PodDns;
_c = JSII_RTTI_SYMBOL_1;
PodDns[_c] = { fqn: "cdk8s-plus-22.PodDns", version: "2.0.0-beta.4" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        var _e, _f, _g;
        this._sysctls = [];
        this.ensureNonRoot = (_e = props.ensureNonRoot) !== null && _e !== void 0 ? _e : false;
        this.fsGroupChangePolicy = (_f = props.fsGroupChangePolicy) !== null && _f !== void 0 ? _f : FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of (_g = props.sysctls) !== null && _g !== void 0 ? _g : []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_d = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_d] = { fqn: "cdk8s-plus-22.PodSecurityContext", version: "2.0.0-beta.4" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
//# sourceMappingURL=data:application/json;base64,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