"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateModel = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * (experimental) A Step Functions Task to create a SageMaker model.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html
 * @experimental
 */
class SageMakerCreateModel extends sfn.TaskStateBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * (experimental) Allows specify security group connections for instances of this fleet.
         *
         * @experimental
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateModel.SUPPORTED_INTEGRATION_PATTERNS);
        // add the security groups to the connections object
        if (props.vpc) {
            this.vpc = props.vpc;
            this.subnets = props.subnetSelection ? this.vpc.selectSubnets(props.subnetSelection).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.role = this.props.role || this.createSagemakerRole();
        this.grantPrincipal = this.role;
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * (experimental) Add the security group to all instances via the launch configuration security groups array.
     *
     * @param securityGroup : The security group to add.
     * @experimental
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createModel', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        var _b, _c;
        return {
            EnableNetworkIsolation: this.props.enableNetworkIsolation,
            ExecutionRoleArn: this.role.roleArn,
            ModelName: this.props.modelName,
            Tags: (_b = this.props.tags) === null || _b === void 0 ? void 0 : _b.value,
            PrimaryContainer: this.props.primaryContainer.bind(this).parameters,
            Containers: (_c = this.props.containers) === null || _c === void 0 ? void 0 : _c.map(container => (container.bind(this))),
            ...this.renderVpcConfig(),
        };
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        return [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateModel'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'model',
                        // If the model name comes from input, we cannot target the policy to a particular ARN prefix reliably.
                        // SageMaker uses lowercase for resource name in the arn
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.modelName) ? '*' : `${this.props.modelName.toLowerCase()}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
    }
    createSagemakerRole() {
        // https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-roles.html
        const role = new iam.Role(this, 'SagemakerRole', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            inlinePolicies: {
                CreateModel: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatch.html
                                'cloudwatch:PutMetricData',
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchlogs.html
                                'logs:CreateLogStream',
                                'logs:CreateLogGroup',
                                'logs:PutLogEvents',
                                'logs:DescribeLogStreams',
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerregistry.html
                                'ecr:GetAuthorizationToken',
                            ],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        if (this.props.vpc) {
            role.addToPrincipalPolicy(
            // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            new iam.PolicyStatement({
                actions: [
                    'ec2:CreateNetworkInterface',
                    'ec2:CreateNetworkInterfacePermission',
                    'ec2:DeleteNetworkInterface',
                    'ec2:DeleteNetworkInterfacePermission',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DescribeVpcs',
                    'ec2:DescribeDhcpOptions',
                    'ec2:DescribeSubnets',
                    'ec2:DescribeSecurityGroups',
                ],
                resources: ['*'],
            }));
        }
        return role;
    }
    renderVpcConfig() {
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'ModelSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        return this.vpc
            ? {
                VpcConfig: {
                    SecurityGroupIds: cdk.Lazy.list({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
}
exports.SageMakerCreateModel = SageMakerCreateModel;
_a = JSII_RTTI_SYMBOL_1;
SageMakerCreateModel[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SageMakerCreateModel", version: "1.95.1" };
SageMakerCreateModel.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlLW1vZGVsLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY3JlYXRlLW1vZGVsLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4QyxrREFBa0Q7QUFDbEQscUNBQXFDO0FBRXJDLHNEQUF5Rjs7Ozs7OztBQW9FekYsTUFBYSxvQkFBcUIsU0FBUSxHQUFHLENBQUMsYUFBYTs7OztJQXFCekQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBbUIsS0FBZ0M7UUFDekYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFEaUMsVUFBSyxHQUFMLEtBQUssQ0FBMkI7Ozs7OztRQWQzRSxnQkFBVyxHQUFvQixJQUFJLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQVVwRCxtQkFBYyxHQUF5QixFQUFFLENBQUM7UUFNekQsSUFBSSxDQUFDLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxHQUFHLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCLENBQUM7UUFDOUYscUNBQXdCLENBQUMsSUFBSSxDQUFDLGtCQUFrQixFQUFFLG9CQUFvQixDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFFdkcsb0RBQW9EO1FBQ3BELElBQUksS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUNiLElBQUksQ0FBQyxHQUFHLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQztZQUNyQixJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxlQUFlLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsYUFBYSxFQUFFLENBQUMsU0FBUyxDQUFDO1NBQ3JJO1FBRUQsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsbUJBQW1CLEVBQUUsQ0FBQztRQUMxRCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7UUFDaEMsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztJQUNsRCxDQUFDOzs7Ozs7O0lBUU0sZ0JBQWdCLENBQUMsYUFBaUM7UUFDdkQsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOztPQUVHO0lBQ08sV0FBVztRQUNuQixPQUFPO1lBQ0wsUUFBUSxFQUFFLG1DQUFzQixDQUFDLFdBQVcsRUFBRSxhQUFhLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDO1lBQ3JGLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztTQUNqRSxDQUFDO0lBQ0osQ0FBQztJQUVPLGdCQUFnQjs7UUFDdEIsT0FBTztZQUNMLHNCQUFzQixFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsc0JBQXNCO1lBQ3pELGdCQUFnQixFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTztZQUNuQyxTQUFTLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTO1lBQy9CLElBQUksUUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksMENBQUUsS0FBSztZQUM1QixnQkFBZ0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVO1lBQ25FLFVBQVUsUUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsMENBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDM0UsR0FBRyxJQUFJLENBQUMsZUFBZSxFQUFFO1NBQzFCLENBQUM7SUFDSixDQUFDO0lBRU8sb0JBQW9CO1FBQzFCLE1BQU0sS0FBSyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pDLE9BQU87WUFDTCxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3RCLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO2dCQUNsQyxTQUFTLEVBQUU7b0JBQ1QsS0FBSyxDQUFDLFNBQVMsQ0FBQzt3QkFDZCxPQUFPLEVBQUUsV0FBVzt3QkFDcEIsUUFBUSxFQUFFLE9BQU87d0JBQ2pCLHVHQUF1Rzt3QkFDdkcsd0RBQXdEO3dCQUN4RCxZQUFZLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsV0FBVyxFQUFFLEdBQUc7cUJBQ3BILENBQUM7aUJBQ0g7YUFDRixDQUFDO1lBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQztnQkFDL0IsMEVBQTBFO2dCQUMxRSxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7YUFDakIsQ0FBQztZQUNGLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDdEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO2dCQUN6QixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQztnQkFDOUIsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRSxFQUFFLHFCQUFxQixFQUFFLHlCQUF5QixFQUFFO2lCQUNuRTthQUNGLENBQUM7U0FDSCxDQUFDO0lBQ0osQ0FBQztJQUVPLG1CQUFtQjtRQUN6Qix1RUFBdUU7UUFDdkUsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7WUFDL0MsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixDQUFDO1lBQzlELGNBQWMsRUFBRTtnQkFDZCxXQUFXLEVBQUUsSUFBSSxHQUFHLENBQUMsY0FBYyxDQUFDO29CQUNsQyxVQUFVLEVBQUU7d0JBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDOzRCQUN0QixPQUFPLEVBQUU7Z0NBQ1AsOEVBQThFO2dDQUM5RSwwQkFBMEI7Z0NBQzFCLGtGQUFrRjtnQ0FDbEYsc0JBQXNCO2dDQUN0QixxQkFBcUI7Z0NBQ3JCLG1CQUFtQjtnQ0FDbkIseUJBQXlCO2dDQUN6Qiw0RkFBNEY7Z0NBQzVGLDJCQUEyQjs2QkFDNUI7NEJBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO3lCQUNqQixDQUFDO3FCQUNIO2lCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUU7WUFDbEIsSUFBSSxDQUFDLG9CQUFvQjtZQUN2Qix1RUFBdUU7WUFDdkUsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUU7b0JBQ1AsNEJBQTRCO29CQUM1QixzQ0FBc0M7b0JBQ3RDLDRCQUE0QjtvQkFDNUIsc0NBQXNDO29CQUN0QywrQkFBK0I7b0JBQy9CLGtCQUFrQjtvQkFDbEIseUJBQXlCO29CQUN6QixxQkFBcUI7b0JBQ3JCLDRCQUE0QjtpQkFDN0I7Z0JBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2FBQ2pCLENBQUMsQ0FDSCxDQUFDO1NBQ0g7UUFDRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFTyxlQUFlO1FBQ3JCLHlDQUF5QztRQUN6QyxJQUFJLElBQUksQ0FBQyxHQUFHLElBQUksSUFBSSxDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDaEQsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLEdBQUcsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO2dCQUNyRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7YUFDZCxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsV0FBVyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUN0RCxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDOUM7UUFDRCxPQUFPLElBQUksQ0FBQyxHQUFHO1lBQ2IsQ0FBQyxDQUFDO2dCQUNBLFNBQVMsRUFBRTtvQkFDVCxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxFQUFFLENBQUM7b0JBQ3ZHLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztpQkFDdEI7YUFDRjtZQUNELENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDVCxDQUFDOztBQW5LSCxvREFvS0M7OztBQW5LeUIsbURBQThCLEdBQTZCO0lBQ2pGLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0I7Q0FDeEMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IGludGVncmF0aW9uUmVzb3VyY2VBcm4sIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCB9IGZyb20gJy4uL3ByaXZhdGUvdGFzay11dGlscyc7XG5pbXBvcnQgeyBJQ29udGFpbmVyRGVmaW5pdGlvbiB9IGZyb20gJy4vYmFzZS10eXBlcyc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFNhZ2VNYWtlckNyZWF0ZU1vZGVsUHJvcHMgZXh0ZW5kcyBzZm4uVGFza1N0YXRlQmFzZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtb2RlbE5hbWU6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcmltYXJ5Q29udGFpbmVyOiBJQ29udGFpbmVyRGVmaW5pdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY29udGFpbmVycz86IElDb250YWluZXJEZWZpbml0aW9uW107XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuYWJsZU5ldHdvcmtJc29sYXRpb24/OiBib29sZWFuO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHN1Ym5ldFNlbGVjdGlvbj86IGVjMi5TdWJuZXRTZWxlY3Rpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0YWdzPzogc2ZuLlRhc2tJbnB1dDtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBTYWdlTWFrZXJDcmVhdGVNb2RlbCBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlIGltcGxlbWVudHMgaWFtLklHcmFudGFibGUsIGVjMi5JQ29ubmVjdGFibGUge1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBTVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlM6IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm5bXSA9IFtcbiAgICBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJFUVVFU1RfUkVTUE9OU0UsXG4gIF07XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGNvbm5lY3Rpb25zOiBlYzIuQ29ubmVjdGlvbnMgPSBuZXcgZWMyLkNvbm5lY3Rpb25zKCk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgcm9sZTogaWFtLklSb2xlO1xuICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IGlhbS5JUHJpbmNpcGFsO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgdGFza01ldHJpY3M/OiBzZm4uVGFza01ldHJpY3NDb25maWc7XG4gIHByb3RlY3RlZCByZWFkb25seSB0YXNrUG9saWNpZXM/OiBpYW0uUG9saWN5U3RhdGVtZW50W107XG4gIHByaXZhdGUgcmVhZG9ubHkgdnBjPzogZWMyLklWcGM7XG4gIHByaXZhdGUgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcbiAgcHJpdmF0ZSByZWFkb25seSBzZWN1cml0eUdyb3VwczogZWMyLklTZWN1cml0eUdyb3VwW10gPSBbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBzdWJuZXRzPzogc3RyaW5nW107XG4gIHByaXZhdGUgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IFNhZ2VNYWtlckNyZWF0ZU1vZGVsUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcbiAgICB0aGlzLmludGVncmF0aW9uUGF0dGVybiA9IHByb3BzLmludGVncmF0aW9uUGF0dGVybiB8fCBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJFUVVFU1RfUkVTUE9OU0U7XG4gICAgdmFsaWRhdGVQYXR0ZXJuU3VwcG9ydGVkKHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuLCBTYWdlTWFrZXJDcmVhdGVNb2RlbC5TVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlMpO1xuXG4gICAgLy8gYWRkIHRoZSBzZWN1cml0eSBncm91cHMgdG8gdGhlIGNvbm5lY3Rpb25zIG9iamVjdFxuICAgIGlmIChwcm9wcy52cGMpIHtcbiAgICAgIHRoaXMudnBjID0gcHJvcHMudnBjO1xuICAgICAgdGhpcy5zdWJuZXRzID0gcHJvcHMuc3VibmV0U2VsZWN0aW9uID8gdGhpcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy5zdWJuZXRTZWxlY3Rpb24pLnN1Ym5ldElkcyA6IHRoaXMudnBjLnNlbGVjdFN1Ym5ldHMoKS5zdWJuZXRJZHM7XG4gICAgfVxuXG4gICAgdGhpcy5yb2xlID0gdGhpcy5wcm9wcy5yb2xlIHx8IHRoaXMuY3JlYXRlU2FnZW1ha2VyUm9sZSgpO1xuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSB0aGlzLnJvbGU7XG4gICAgdGhpcy50YXNrUG9saWNpZXMgPSB0aGlzLm1ha2VQb2xpY3lTdGF0ZW1lbnRzKCk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBhZGRTZWN1cml0eUdyb3VwKHNlY3VyaXR5R3JvdXA6IGVjMi5JU2VjdXJpdHlHcm91cCk6IHZvaWQge1xuICAgIHRoaXMuc2VjdXJpdHlHcm91cHMucHVzaChzZWN1cml0eUdyb3VwKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHByb3RlY3RlZCBfcmVuZGVyVGFzaygpOiBhbnkge1xuICAgIHJldHVybiB7XG4gICAgICBSZXNvdXJjZTogaW50ZWdyYXRpb25SZXNvdXJjZUFybignc2FnZW1ha2VyJywgJ2NyZWF0ZU1vZGVsJywgdGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4pLFxuICAgICAgUGFyYW1ldGVyczogc2ZuLkZpZWxkVXRpbHMucmVuZGVyT2JqZWN0KHRoaXMucmVuZGVyUGFyYW1ldGVycygpKSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJQYXJhbWV0ZXJzKCk6IHsgW2tleTogc3RyaW5nXTogYW55IH0ge1xuICAgIHJldHVybiB7XG4gICAgICBFbmFibGVOZXR3b3JrSXNvbGF0aW9uOiB0aGlzLnByb3BzLmVuYWJsZU5ldHdvcmtJc29sYXRpb24sXG4gICAgICBFeGVjdXRpb25Sb2xlQXJuOiB0aGlzLnJvbGUucm9sZUFybixcbiAgICAgIE1vZGVsTmFtZTogdGhpcy5wcm9wcy5tb2RlbE5hbWUsXG4gICAgICBUYWdzOiB0aGlzLnByb3BzLnRhZ3M/LnZhbHVlLFxuICAgICAgUHJpbWFyeUNvbnRhaW5lcjogdGhpcy5wcm9wcy5wcmltYXJ5Q29udGFpbmVyLmJpbmQodGhpcykucGFyYW1ldGVycyxcbiAgICAgIENvbnRhaW5lcnM6IHRoaXMucHJvcHMuY29udGFpbmVycz8ubWFwKGNvbnRhaW5lciA9PiAoY29udGFpbmVyLmJpbmQodGhpcykpKSxcbiAgICAgIC4uLnRoaXMucmVuZGVyVnBjQ29uZmlnKCksXG4gICAgfTtcbiAgfVxuXG4gIHByaXZhdGUgbWFrZVBvbGljeVN0YXRlbWVudHMoKTogaWFtLlBvbGljeVN0YXRlbWVudFtdIHtcbiAgICBjb25zdCBzdGFjayA9IGNkay5TdGFjay5vZih0aGlzKTtcbiAgICByZXR1cm4gW1xuICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ3NhZ2VtYWtlcjpDcmVhdGVNb2RlbCddLFxuICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICBzdGFjay5mb3JtYXRBcm4oe1xuICAgICAgICAgICAgc2VydmljZTogJ3NhZ2VtYWtlcicsXG4gICAgICAgICAgICByZXNvdXJjZTogJ21vZGVsJyxcbiAgICAgICAgICAgIC8vIElmIHRoZSBtb2RlbCBuYW1lIGNvbWVzIGZyb20gaW5wdXQsIHdlIGNhbm5vdCB0YXJnZXQgdGhlIHBvbGljeSB0byBhIHBhcnRpY3VsYXIgQVJOIHByZWZpeCByZWxpYWJseS5cbiAgICAgICAgICAgIC8vIFNhZ2VNYWtlciB1c2VzIGxvd2VyY2FzZSBmb3IgcmVzb3VyY2UgbmFtZSBpbiB0aGUgYXJuXG4gICAgICAgICAgICByZXNvdXJjZU5hbWU6IHNmbi5Kc29uUGF0aC5pc0VuY29kZWRKc29uUGF0aCh0aGlzLnByb3BzLm1vZGVsTmFtZSkgPyAnKicgOiBgJHt0aGlzLnByb3BzLm1vZGVsTmFtZS50b0xvd2VyQ2FzZSgpfSpgLFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgfSksXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFsnc2FnZW1ha2VyOkxpc3RUYWdzJ10sXG4gICAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvc2FnZW1ha2VyLWlhbS5odG1sXG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICB9KSxcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogWydpYW06UGFzc1JvbGUnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5yb2xlLnJvbGVBcm5dLFxuICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgU3RyaW5nRXF1YWxzOiB7ICdpYW06UGFzc2VkVG9TZXJ2aWNlJzogJ3NhZ2VtYWtlci5hbWF6b25hd3MuY29tJyB9LFxuICAgICAgICB9LFxuICAgICAgfSksXG4gICAgXTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlU2FnZW1ha2VyUm9sZSgpIHtcbiAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2FnZW1ha2VyL2xhdGVzdC9kZy9zYWdlbWFrZXItcm9sZXMuaHRtbFxuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ1NhZ2VtYWtlclJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnc2FnZW1ha2VyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIENyZWF0ZU1vZGVsOiBuZXcgaWFtLlBvbGljeURvY3VtZW50KHtcbiAgICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jbG91ZHdhdGNoLmh0bWxcbiAgICAgICAgICAgICAgICAnY2xvdWR3YXRjaDpQdXRNZXRyaWNEYXRhJyxcbiAgICAgICAgICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hbWF6b25jbG91ZHdhdGNobG9ncy5odG1sXG4gICAgICAgICAgICAgICAgJ2xvZ3M6Q3JlYXRlTG9nU3RyZWFtJyxcbiAgICAgICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dHcm91cCcsXG4gICAgICAgICAgICAgICAgJ2xvZ3M6UHV0TG9nRXZlbnRzJyxcbiAgICAgICAgICAgICAgICAnbG9nczpEZXNjcmliZUxvZ1N0cmVhbXMnLFxuICAgICAgICAgICAgICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2FtYXpvbmVsYXN0aWNjb250YWluZXJyZWdpc3RyeS5odG1sXG4gICAgICAgICAgICAgICAgJ2VjcjpHZXRBdXRob3JpemF0aW9uVG9rZW4nLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgICAgfSksXG4gICAgICAgICAgXSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGlmICh0aGlzLnByb3BzLnZwYykge1xuICAgICAgcm9sZS5hZGRUb1ByaW5jaXBhbFBvbGljeShcbiAgICAgICAgLy8gaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYW1hem9uZWMyLmh0bWxcbiAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICdlYzI6Q3JlYXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICAgICAnZWMyOkNyZWF0ZU5ldHdvcmtJbnRlcmZhY2VQZXJtaXNzaW9uJyxcbiAgICAgICAgICAgICdlYzI6RGVsZXRlTmV0d29ya0ludGVyZmFjZScsXG4gICAgICAgICAgICAnZWMyOkRlbGV0ZU5ldHdvcmtJbnRlcmZhY2VQZXJtaXNzaW9uJyxcbiAgICAgICAgICAgICdlYzI6RGVzY3JpYmVOZXR3b3JrSW50ZXJmYWNlcycsXG4gICAgICAgICAgICAnZWMyOkRlc2NyaWJlVnBjcycsXG4gICAgICAgICAgICAnZWMyOkRlc2NyaWJlRGhjcE9wdGlvbnMnLFxuICAgICAgICAgICAgJ2VjMjpEZXNjcmliZVN1Ym5ldHMnLFxuICAgICAgICAgICAgJ2VjMjpEZXNjcmliZVNlY3VyaXR5R3JvdXBzJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgIH0pLFxuICAgICAgKTtcbiAgICB9XG4gICAgcmV0dXJuIHJvbGU7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlclZwY0NvbmZpZygpOiB7IFtrZXk6IHN0cmluZ106IGFueSB9IHtcbiAgICAvLyBjcmVhdGUgYSBzZWN1cml0eSBncm91cCBpZiBub3QgZGVmaW5lZFxuICAgIGlmICh0aGlzLnZwYyAmJiB0aGlzLnNlY3VyaXR5R3JvdXAgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhpcy5zZWN1cml0eUdyb3VwID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdNb2RlbFNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICB9KTtcbiAgICAgIHRoaXMuY29ubmVjdGlvbnMuYWRkU2VjdXJpdHlHcm91cCh0aGlzLnNlY3VyaXR5R3JvdXApO1xuICAgICAgdGhpcy5zZWN1cml0eUdyb3Vwcy5wdXNoKHRoaXMuc2VjdXJpdHlHcm91cCk7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLnZwY1xuICAgICAgPyB7XG4gICAgICAgIFZwY0NvbmZpZzoge1xuICAgICAgICAgIFNlY3VyaXR5R3JvdXBJZHM6IGNkay5MYXp5Lmxpc3QoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnNlY3VyaXR5R3JvdXBzLm1hcCgoc2cpID0+IHNnLnNlY3VyaXR5R3JvdXBJZCkgfSksXG4gICAgICAgICAgU3VibmV0czogdGhpcy5zdWJuZXRzLFxuICAgICAgICB9LFxuICAgICAgfVxuICAgICAgOiB7fTtcbiAgfVxufSJdfQ==