"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretRotation = exports.SecretRotationApplication = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const lambda = require("@aws-cdk/aws-lambda");
const serverless = require("@aws-cdk/aws-sam");
const core_1 = require("@aws-cdk/core");
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Secret rotation for a service or database
 */
class SecretRotation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        // Max length of 64 chars, get the last 64 chars
        const uniqueId = this.node.uniqueId;
        const rotationFunctionName = uniqueId.substring(Math.max(uniqueId.length - 64, 0), uniqueId.length);
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter,
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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