"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
test('snapshot test ApiGatewayToDynamoDB default params', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb-default', apiGatewayToDynamoDBProps);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check properties', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    const construct = new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb-default', apiGatewayToDynamoDBProps);
    expect(construct.dynamoTable !== null);
    expect(construct.apiGateway !== null);
    expect(construct.apiGatewayRole !== null);
});
test('check allow CRUD operations', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowReadOperation: true,
        allowCreateOperation: true,
        createRequestTemplate: "{}",
        allowDeleteOperation: true,
        allowUpdateOperation: true,
        updateRequestTemplate: "{}"
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:PutItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:DeleteItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4"
            }
        ]
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "PUT",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "AWS_IAM"
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{id}",
    });
});
test('check allow read and update only', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowUpdateOperation: true,
        updateRequestTemplate: "{}"
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "testapigatewaydynamodbDynamoTableEEE3F463",
                            "Arn"
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4"
            }
        ]
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM"
    });
});
test('check using custom partition key for dynamodb', () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        dynamoTableProps: {
            partitionKey: {
                name: 'page_id',
                type: aws_dynamodb_1.AttributeType.STRING
            }
        }
    };
    new lib_1.ApiGatewayToDynamoDB(stack, 'test-api-gateway-dynamodb', apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{page_id}",
    });
});
//# sourceMappingURL=data:application/json;base64,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