"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToDynamoDB = void 0;
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const core_2 = require("@aws-solutions-constructs/core");
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
class ApiGatewayToDynamoDB extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let partitionKeyName;
        // Set the default props for DynamoDB table
        if (props.dynamoTableProps) {
            const dynamoTableProps = core_2.overrideProps(defaults.DefaultTableProps, props.dynamoTableProps);
            partitionKeyName = dynamoTableProps.partitionKey.name;
            this.dynamoTable = new dynamodb.Table(this, 'DynamoTable', dynamoTableProps);
        }
        else {
            partitionKeyName = defaults.DefaultTableProps.partitionKey.name;
            this.dynamoTable = new dynamodb.Table(this, 'DynamoTable', defaults.DefaultTableProps);
        }
        // Setup the API Gateway
        this.apiGateway = defaults.GlobalRestApi(this);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway Resource
        const apiGatewayResource = this.apiGateway.root.addResource("{" + partitionKeyName + "}");
        // Setup API Gateway Method
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = props.createRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActiontoPlicy("dynamodb:PutItem");
            this.addMethod(this.apiGateway.root, createRequestTemplate, "PutItem", "POST");
        }
        // Read
        if (!props.allowReadOperation || props.allowReadOperation === true) {
            const getRequestTemplate = "{\r\n\"TableName\": \"" + this.dynamoTable.tableName + "\",\r\n \"KeyConditionExpression\": \"" + partitionKeyName + " = :v1\",\r\n    \"ExpressionAttributeValues\": {\r\n        \":v1\": {\r\n            \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n        }\r\n    }\r\n}";
            this.addActiontoPlicy("dynamodb:Query");
            this.addMethod(apiGatewayResource, getRequestTemplate, "Query", "GET");
        }
        // Update
        if (props.allowUpdateOperation && props.allowUpdateOperation === true && props.updateRequestTemplate) {
            const updateRequestTemplate = props.updateRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActiontoPlicy("dynamodb:UpdateItem");
            this.addMethod(apiGatewayResource, updateRequestTemplate, "UpdateItem", "PUT");
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            const deleteRequestTemplate = "{\r\n  \"TableName\": \"" + this.dynamoTable.tableName + "\",\r\n  \"Key\": {\r\n    \"" + partitionKeyName + "\": {\r\n      \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n    }\r\n  },\r\n  \"ConditionExpression\": \"attribute_not_exists(Replies)\",\r\n  \"ReturnValues\": \"ALL_OLD\"\r\n}";
            this.addActiontoPlicy("dynamodb:DeleteItem");
            this.addMethod(apiGatewayResource, deleteRequestTemplate, "DeleteItem", "DELETE");
        }
    }
    addActiontoPlicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.dynamoTable.tableArn
            ],
            actions: [`${action}`]
        }));
    }
    addMethod(apiResource, requestTemplate, dynamodbAction, apiMethod) {
        // Setup the API Gateway Integration
        const apiGatewayIntegration = new api.AwsIntegration({
            service: "dynamodb",
            action: dynamodbAction,
            integrationHttpMethod: "POST",
            options: {
                passthroughBehavior: api.PassthroughBehavior.NEVER,
                credentialsRole: this.apiGatewayRole,
                requestParameters: {
                    "integration.request.header.Content-Type": "'application/json'"
                },
                requestTemplates: {
                    "application/json": requestTemplate
                },
                integrationResponses: [
                    {
                        statusCode: "200"
                    },
                    {
                        statusCode: "500",
                        responseTemplates: {
                            "text/html": "Error"
                        },
                        selectionPattern: "500"
                    }
                ]
            }
        });
        // Setup the API Gateway method(s)
        apiResource.addMethod(apiMethod, apiGatewayIntegration, {
            methodResponses: [
                {
                    statusCode: "200",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    }
                },
                {
                    statusCode: "500",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    },
                }
            ]
        });
    }
}
exports.ApiGatewayToDynamoDB = ApiGatewayToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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