"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line
const DELIMITER = ":";
const SCOPE = "CLOUDFRONT";
const client = new aws_sdk_1.WAFV2({
    region: "us-east-1",
});
/**
 * Handler for creating a WAF V2 ACL in US-EAST-1.
 */
exports.onEvent = async (event) => {
    const { ID, MANAGED_RULES, CIDR_ALLOW_LIST } = event.ResourceProperties;
    const [WEB_ACL_ID, IP_SET_ID] = event.PhysicalResourceId
        ? event.PhysicalResourceId.split(DELIMITER)
        : [];
    let response = {};
    switch (event.RequestType) {
        case "Create":
            response = await createWaf(ID, MANAGED_RULES, CIDR_ALLOW_LIST);
            break;
        case "Update":
            response = await updateWaf(WEB_ACL_ID, IP_SET_ID, ID, getIpSetName(ID), MANAGED_RULES, CIDR_ALLOW_LIST);
            break;
        case "Delete":
            response = await deleteWaf(WEB_ACL_ID, IP_SET_ID, ID, getIpSetName(ID));
            break;
        default:
            throw new Error(`Invalid RequestType: ${event.RequestType}`);
    }
    return response;
};
/**
 * Generates the name of the IP Set.
 *
 * @param id param passed in.
 * @returns name of IP Set.
 */
const getIpSetName = (id) => `${id}-IPSet`;
/**
 * Returns a set of rules to apply.
 *
 * @param ipSetArn ip set arn
 * @param ipSetName  ip set name
 * @param managedRules  managed rules
 * @param cidrAllowList cidr allow list
 * @returns set of rules to apply.
 */
const getWafRules = (ipSetArn, ipSetName, managedRules, cidrAllowList) => {
    const rules = [];
    if (cidrAllowList) {
        rules.push({
            Name: ipSetName,
            Priority: 1,
            VisibilityConfig: {
                MetricName: ipSetName,
                CloudWatchMetricsEnabled: true,
                SampledRequestsEnabled: true,
            },
            Action: {
                Block: {},
            },
            Statement: {
                NotStatement: {
                    Statement: {
                        IPSetReferenceStatement: {
                            ARN: ipSetArn,
                        },
                    },
                },
            },
        });
    }
    if (managedRules) {
        rules.push(...managedRules
            .map((r) => ({ VendorName: r.vendor, Name: r.name }))
            .map((rule, Priority) => ({
            Name: `${rule.VendorName}-${rule.Name}`,
            Priority,
            Statement: { ManagedRuleGroupStatement: rule },
            OverrideAction: { None: {} },
            VisibilityConfig: {
                MetricName: `${rule.VendorName}-${rule.Name}`,
                CloudWatchMetricsEnabled: true,
                SampledRequestsEnabled: true,
            },
        })));
    }
    return rules;
};
const createWaf = async (id, managedRules, cidrAllowList) => {
    const ipSetName = getIpSetName(id);
    const createIpSetResponse = await client
        .createIPSet({
        Name: ipSetName,
        Scope: SCOPE,
        Addresses: cidrAllowList?.cidrRanges ?? [],
        IPAddressVersion: cidrAllowList?.type ?? "IPV4",
    })
        .promise();
    const createWebAclResponse = await client
        .createWebACL({
        Name: id,
        DefaultAction: { Allow: {} },
        Scope: SCOPE,
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: id,
            SampledRequestsEnabled: true,
        },
        Rules: getWafRules(createIpSetResponse.Summary.ARN, ipSetName, managedRules, cidrAllowList),
    })
        .promise();
    return {
        PhysicalResourceId: `${createWebAclResponse.Summary?.Id}${DELIMITER}${createIpSetResponse.Summary?.Id}`,
        Data: {
            WebAclArn: createWebAclResponse.Summary?.ARN,
            WebAclId: createWebAclResponse.Summary?.Id,
            IPSetArn: createIpSetResponse.Summary?.ARN,
            IPSetId: createIpSetResponse.Summary?.Id,
        },
    };
};
const updateWaf = async (webAclId, ipSetId, id, ipSetName, managedRules, cidrAllowList) => {
    const getIpSetResponse = await client
        .getIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
    })
        .promise();
    await client
        .updateIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Addresses: cidrAllowList?.cidrRanges ?? [],
        Scope: SCOPE,
        LockToken: getIpSetResponse.LockToken,
    })
        .promise();
    const getWebAclResponse = await client
        .getWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
    })
        .promise();
    await client
        .updateWebACL({
        Name: id,
        DefaultAction: { Allow: {} },
        Scope: SCOPE,
        VisibilityConfig: {
            CloudWatchMetricsEnabled: true,
            MetricName: id,
            SampledRequestsEnabled: true,
        },
        Rules: getWafRules(getIpSetResponse.IPSet?.ARN, ipSetName, managedRules, cidrAllowList),
        Id: getWebAclResponse.WebACL?.Id,
        LockToken: getWebAclResponse.LockToken,
    })
        .promise();
    return {
        Data: {
            WebAclArn: getWebAclResponse.WebACL?.ARN,
            WebAclId: getWebAclResponse.WebACL?.Id,
            IPSetArn: getIpSetResponse.IPSet?.ARN,
            IPSetId: getIpSetResponse.IPSet?.Id,
        },
    };
};
const deleteWaf = async (webAclId, ipSetId, id, ipSetName) => {
    const getWebAclResponse = await client
        .getWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
    })
        .promise();
    await client
        .deleteWebACL({
        Id: webAclId,
        Name: id,
        Scope: SCOPE,
        LockToken: getWebAclResponse.LockToken,
    })
        .promise();
    const getIpSetResponse = await client
        .getIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
    })
        .promise();
    await client
        .deleteIPSet({
        Id: ipSetId,
        Name: ipSetName,
        Scope: SCOPE,
        LockToken: getIpSetResponse.LockToken,
    })
        .promise();
    return {
        Data: {
            WebAclArn: getWebAclResponse.WebACL?.ARN,
            WebAclId: getWebAclResponse.WebACL?.Id,
            IPSetArn: getIpSetResponse.IPSet?.ARN,
            IPSetId: getIpSetResponse.IPSet?.Id,
        },
    };
};
//# sourceMappingURL=data:application/json;base64,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