"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RedshiftConnectorProfile = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const type_1 = require("./type");
const appflow_permissions_manager_1 = require("../core/appflow-permissions-manager");
const connector_profile_1 = require("../core/connectors/connector-profile");
class RedshiftConnectorProfile extends connector_profile_1.ConnectorProfileBase {
    constructor(scope, id, props) {
        super(scope, id, props, type_1.RedshiftConnectorType.instance);
        this.tryAddNodeDependency(this, props.cluster);
        this._location = props.intermediateLocation;
    }
    static fromConnectionProfileArn(scope, id, arn) {
        return this._fromConnectorProfileAttributes(scope, id, { arn });
    }
    static fromConnectionProfileName(scope, id, name) {
        return this._fromConnectorProfileAttributes(scope, id, { name });
    }
    buildConnectorProfileProperties(props) {
        const properties = props;
        const redshiftAccessRole = properties.bucketAccessRole ?? this.buildRedshiftAccessRole(this.node.id, properties.cluster, properties.intermediateLocation);
        const appflowDataApiRole = properties.dataApiRole ?? this.buildAppFlowDataApiRole(this.node.id, properties.cluster, properties.databaseName, properties.basicAuth.username);
        this.tryAddNodeDependency(this, redshiftAccessRole);
        this.tryAddNodeDependency(this, appflowDataApiRole);
        this.tryAddNodeDependency(this, properties.intermediateLocation.bucket);
        appflow_permissions_manager_1.AppFlowPermissionsManager.instance().grantBucketReadWrite(properties.intermediateLocation.bucket);
        return {
            redshift: {
                bucketName: properties.intermediateLocation.bucket.bucketName,
                bucketPrefix: properties.intermediateLocation.prefix,
                roleArn: redshiftAccessRole.roleArn,
                clusterIdentifier: properties.cluster.clusterName,
                databaseName: properties.databaseName,
                dataApiRoleArn: appflowDataApiRole.roleArn,
            },
        };
    }
    buildConnectorProfileCredentials(props) {
        const properties = props;
        return {
            redshift: properties && {
                username: properties.basicAuth.username,
                password: properties.basicAuth.password,
            },
        };
    }
    buildRedshiftAccessRole(id, cluster, location) {
        // see: https://docs.aws.amazon.com/appflow/latest/userguide/security_iam_service-role-policies.html#redshift-access-s3
        const role = new aws_iam_1.Role(this.stack, `${id}RedshiftRole`, {
            assumedBy: new aws_iam_1.ServicePrincipal('redshift.amazonaws.com'),
        });
        location.bucket.grantRead(role, location.prefix ? `${location.prefix}/*` : '*');
        const modifierId = `${id}RedshiftRoleAttach`;
        const modifier = new custom_resources_1.AwsCustomResource(this.stack, modifierId, {
            onCreate: {
                service: 'Redshift',
                action: 'modifyClusterIamRoles',
                parameters: {
                    ClusterIdentifier: cluster.clusterName,
                    AddIamRoles: [role.roleArn],
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(modifierId),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromStatements([
                new aws_iam_1.PolicyStatement({
                    actions: ['iam:PassRole'],
                    resources: [role.roleArn],
                    effect: aws_iam_1.Effect.ALLOW,
                }),
                new aws_iam_1.PolicyStatement({
                    actions: ['redshift:ModifyClusterIamRoles'],
                    resources: [
                        `arn:${aws_cdk_lib_1.Aws.PARTITION}:redshift:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:cluster:${cluster.clusterName}`,
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                }),
            ]),
        });
        this.node.addDependency(modifier);
        modifier.node.addDependency(cluster);
        return role;
    }
    buildAppFlowDataApiRole(id, cluster, databaseName, username) {
        // see: https://docs.aws.amazon.com/appflow/latest/userguide/security_iam_service-role-policies.html#access-redshift
        const role = new aws_iam_1.Role(this.stack, `${id}AppFlowDataApiRole`, {
            assumedBy: new aws_iam_1.ServicePrincipal('appflow.amazonaws.com'),
        });
        const policy = new aws_iam_1.Policy(this.stack, `${id}AppFlowDataApiRolePolicy`, {
            roles: [role],
            statements: [
                new aws_iam_1.PolicyStatement({
                    sid: 'DataAPIPermissions',
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'redshift-data:ExecuteStatement',
                        'redshift-data:GetStatementResult',
                        'redshift-data:DescribeStatement',
                    ],
                    resources: ['*'],
                }),
                new aws_iam_1.PolicyStatement({
                    sid: 'GetCredentialsForAPIUser',
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['redshift:GetClusterCredentials'],
                    resources: [
                        `arn:aws:redshift:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:dbname:${cluster.clusterName}/${databaseName}`,
                        `arn:aws:redshift:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:dbuser:${cluster.clusterName}/${username ?? '*'}`,
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    sid: 'DenyCreateAPIUser',
                    effect: aws_iam_1.Effect.DENY,
                    actions: ['redshift:CreateClusterUser'],
                    resources: [
                        `arn:aws:redshift:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:dbuser:${cluster.clusterName}/*`,
                    ],
                }),
                new aws_iam_1.PolicyStatement({
                    sid: 'ServiceLinkedRole',
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['iam:CreateServiceLinkedRole'],
                    resources: [
                        `arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/aws-service-role/redshift-data.amazonaws.com/AWSServiceRoleForRedshift`,
                    ],
                    conditions: {
                        StringLike: {
                            'iam:AWSServiceName': 'redshift-data.amazonaws.com',
                        },
                    },
                }),
            ],
        });
        this.tryAddNodeDependency(this, policy);
        return role;
    }
}
exports.RedshiftConnectorProfile = RedshiftConnectorProfile;
_a = JSII_RTTI_SYMBOL_1;
RedshiftConnectorProfile[_a] = { fqn: "@cdklabs/cdk-appflow.RedshiftConnectorProfile", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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