import { Duration, IResource, Resource } from 'aws-cdk-lib';
import { CfnFlow } from 'aws-cdk-lib/aws-appflow';
import { OnEventOptions, Rule, Schedule } from 'aws-cdk-lib/aws-events';
import { IKey } from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
import { IMapping, ITransform, IFilter, IValidation } from '../tasks';
import { IDestination } from '../vertices/destination';
import { ISource } from '../vertices/source';
export interface IFlow extends IResource {
    /**
     * The ARN of the flow.
     */
    readonly arn: string;
    /**
     * The name of the flow
     */
    readonly name: string;
    /**
     * The type of the flow.
     */
    readonly type: FlowType;
    onRunStarted(id: string, options?: OnEventOptions): Rule;
    onRunCompleted(id: string, options?: OnEventOptions): Rule;
    /**
     * @internal
     */
    _addMapping(mapping: IMapping): IFlow;
    /**
     * @internal
     */
    _addValidation(validator: IValidation): IFlow;
    /**
     * @internal
     */
    _addTransform(transform: ITransform): IFlow;
    /**
     * @internal
     */
    _addFilter(filter: IFilter): IFlow;
    /**
     * @internal
     */
    _addCatalog(metadata: CfnFlow.MetadataCatalogConfigProperty): void;
}
export declare enum FlowType {
    EVENT = "Event",
    ON_DEMAND = "OnDemand",
    SCHEDULED = "Scheduled"
}
export declare enum FlowStatus {
    ACTIVE = "Active",
    SUSPENDED = "Suspended"
}
export declare enum DataPullMode {
    COMPLETE = "Complete",
    INCREMENTAL = "Incremental"
}
export interface DataPullConfig {
    readonly mode: DataPullMode;
    /**
     * The name of the field to use as a timestamp for recurring incremental flows.
     * The default field is set per particular @see ISource.
     */
    readonly timestampField?: string;
}
export interface ScheduleProperties {
    readonly startTime?: Date;
    readonly endTime?: Date;
    readonly offset?: Duration;
    /**
     * Timestamp for the records to import from the connector in the first flow run
     * @default 30 days back from the initial frow run
     */
    readonly firstExecutionFrom?: Date;
    readonly timezone?: string;
}
export interface TriggerProperties {
    readonly dataPullConfig: DataPullConfig;
    readonly schedule: Schedule;
    readonly properties?: ScheduleProperties;
    readonly flowErrorDeactivationThreshold?: number;
}
export interface TriggerConfig {
    readonly properties?: TriggerProperties;
}
export interface FlowProps {
    readonly name?: string;
    readonly description?: string;
    readonly source: ISource;
    readonly destination: IDestination;
    readonly mappings: IMapping[];
    readonly validations?: IValidation[];
    readonly transforms?: ITransform[];
    readonly filters?: IFilter[];
    readonly key?: IKey;
}
export interface FlowBaseProps extends FlowProps {
    readonly type: FlowType;
    readonly triggerConfig?: TriggerConfig;
    readonly status?: FlowStatus;
}
export declare abstract class FlowBase extends Resource implements IFlow {
    /**
     * The ARN of the flow.
     */
    readonly arn: string;
    /**
     * The type of the flow.
     */
    readonly type: FlowType;
    /**
     * The name of the flow.
     */
    readonly name: string;
    private readonly mappings;
    private readonly validations;
    private readonly transforms;
    private readonly filters;
    private readonly source;
    private _catalogMetadata?;
    private _projectionFilter;
    constructor(scope: Construct, id: string, props: FlowBaseProps);
    private buildTriggerProperties;
    private initProjectionFilter;
    /**
       * Set the catalog definitionfor the flow
       *
       * @internal
       */
    _addCatalog(metadata: CfnFlow.MetadataCatalogConfigProperty): void;
    /**
     *
     * @param validation
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addValidation(validation: IValidation): IFlow;
    /**
     *
     * @param mapping
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addMapping(mapping: IMapping): IFlow;
    /**
     *
     * @param filter
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addFilter(filter: IFilter): IFlow;
    /**
     *
     * @param transform
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addTransform(transform: ITransform): IFlow;
    private addProjectionField;
    onEvent(id: string, options?: OnEventOptions): Rule;
    onRunStarted(id: string, options?: OnEventOptions): Rule;
    onRunCompleted(id: string, options?: OnEventOptions): Rule;
    private buildIncrementalPullConfig;
}
