from ....Internal.Core import Core
from ....Internal.CommandsGroup import CommandsGroup
from ....Internal import Conversions
from ....Internal.Utilities import trim_str_response
from .... import enums


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class StandardCls:
	"""Standard commands group definition. 2 total commands, 0 Subgroups, 2 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("standard", core, parent)

	def get_reference(self) -> str:
		"""SCPI: [SOURce<HW>]:FSIMulator:STANdard:REFerence \n
		Snippet: value: str = driver.source.fsimulator.standard.get_reference() \n
		Queries the reference in the standard for the selected test case. \n
			:return: reference: string
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:FSIMulator:STANdard:REFerence?')
		return trim_str_response(response)

	def set_reference(self, reference: str) -> None:
		"""SCPI: [SOURce<HW>]:FSIMulator:STANdard:REFerence \n
		Snippet: driver.source.fsimulator.standard.set_reference(reference = '1') \n
		Queries the reference in the standard for the selected test case. \n
			:param reference: string
		"""
		param = Conversions.value_to_quoted_str(reference)
		self._core.io.write(f'SOURce<HwInstance>:FSIMulator:STANdard:REFerence {param}')

	# noinspection PyTypeChecker
	def get_value(self) -> enums.FadStan:
		"""SCPI: [SOURce<HW>]:FSIMulator:STANdard \n
		Snippet: value: enums.FadStan = driver.source.fsimulator.standard.get_value() \n
		Selects a predefined fading simulator setting which complies with the test specifications found in the common mobile
		radio standards. For a detailed summary of all of the default settings, see 'Predefined fading settings'.
			Table Header: Standard / Test Case / <Predefined_Standard> / Description \n
			- USER / USER parameter cannot be set. A query returns this value if a user-defined Fading setting was loaded or if one of the associated settings was changed subsequent to the selection of a standard
			- CDMA See 'CDMA standards' / CDMA0 | CDMA3 | CDMA8 | CDMA30 | C1DMA30 | CDMA100 / CDMA 5 (0 km/h) , CDMA6 (3km/h) , CDMA1 (8 km/h) , CDMA2 (30 km/h) , CDMA3 (30 km/h, 1 path) , CDMA4 (100km/h)
			- GSM See 'GSM standards' / GTU1P5 | G6TU1P5 | GTU3P6 | G6TU3P6 | GTU3 | G6TU3 | GTU6 | G6TU6 | GTU50 | G6TU50 | G6TU100 | G6TU60 / GSM Typical Urban 1,5/3/3,6/6/50/60/100 km/h, 6 and 12 path
			- GHT100 | G6HT100 | GHT120 | G6HT120 | GHT200 | G6HT200 / GSM Hilly Terrain 100/120 km/h, 6 and 12 path
			- GRA130 | GRA250 | GRA300 | GRA500 / GSM Rural Area 130/250/300/500 km/h, 6 path
			- GET50 | GET60 | GET100 / GSM Equal Test 50/60/100 km/h, 6 path
			- GTI5 / GSM typical case for very small cells, 5km/h, 2 path
			- NADC See 'NADC standards' / NADC8 | NADC50 | NADC100 / NADC 8/50/100 km/h, 2 path
			- DCS1800/PCS1900 See 'PCN standards' / P6TU1 | PTU1 | P6TU50 | PTU50 / Typical Urban 1.5/50 km/h, 6 and 12 path
			- P6HT100 | PHT100 / Hilly Terrain 100 km/h, 6 and 12 path
			- PRA130 / Rural Area 130 km/h, 6 path
			- PET50 | PET100 / Equal Test 50/100 km/h, 6 path
			- TETRA See 'TETRA standards' / TTU | T6TU / TETRA Typical Urban 50km/h, 2 path and 6 path
			- TBU / TETRA 2 path
			- THT | T6HT / TETRA Hilly Terrain 200 km/h, 2 path and 6 path
			- T4ET / TETRA Equal Test 200 km/h, 4 path
			- TDR / TETRA Direct Mode Rural Propagation Model 1 path
			- TDU / TETRA Mode Urban Propagation Model 1 path
			- 3GPP FDD See '3GPP standards' / G3C1 | G3C2 | G3C3 | G3C4 / 3GPP FDD Test Case x (BS)
			- G3UEC1 | G3UEC2 | G3UEC3 | G3UEC4 | G3UEC5 | G3UEC6 / 3GPP FDD UE Test Case x (UE)
			- G3UEC7SE / 3GPP FDD UE Sector (UE)
			- G3UEC7BE / 3GPP FDD Beam (UE)
			- G3UEC8CQ / 3GPP FDD CQI (UE)
			- G3UEPA3 | G3UEPB3 / 3GPP FDD Pedestrian A 4 path / B 6 path (UE)
			- G3UEVA3 | G3UEVA30 | G3UEVA120 / 3GPP FDD Vehicular A 6 path (UE)
			- G3MBSFN3 / 3GPP MBSFN
			- G3TU3 | G3TU50 | G3TU120 / 3GPP FDD Typical Urban 20 path
			- G3HT120 / 3GPP FDD Hilly Terrain 20 path
			- G3RA120 | G3RA250 / 3GPP FDD Rural Area 10 path
			- BD1 / 3GPP Birth Death 2 path
			- Moving Propagation / MD1 / 3GPP Moving Propagation 'Ref. + Moving Channel', 2path
			- See '3GPP/LTE moving propagation' / MPLTEETU200 / 3GPP Moving Propagation scenario 1 'ETU200Hz', according to the test case 3GPP TS36.141, annex B4.
			- MPLTEPDOPP / 3GPP Moving Propagation scenario 2 'AWGN', according to the test case 3GPP TS36.141, annex B4.
			- 3GPP High Speed Train scenarios / G3HST1OS | G3HST1OSDU / 3GPP HST1 'Open space', according to the test case 3GPP TS25.141, annex D.4A and 3GPP TS36.141, annex B.3
			- See '3GPP/LTE high speed train' / G3HST2TLC | G3HST2TLCDU / 3GPP HST2 'Tunnel with leaky cable', according to the test case 3GPP TS25.141, annex D.4A
			- G3HST3TMA | G3HST3TMADU / 3GPP HST3 'Tunnel for multi-antennas', according to the test case 3GPP TS25.141, annex D.4A and 3GPP TS36.141, annex B.3
			- SCME/Geo SCME UMi 3kmh/30kmh SCME/Geo SCME UMa 3kmh/30kmh See 'SCM and SCME channel models for MIMO OTA' / G3SCMEUMA3 | G3SCMEUMA30 | G3SCMEUMI3 | G3SCMEUMI30 GEOSCMEUMA3|GEOSCMEUMA30|GEOSCMEUMI3|GEOSCMEUMI30 / SCME/Geo SCME Urban Micro/Macro-Cell Channel, 3 km/h and 30 km/h
			- WLAN See 'WLAN standards' / HL2A | HL2B | HL2C | HL2D | HL2E / WLAN HyperLan 18 path
			- DAB See 'DAB standards' / DABRA04 | DABRA06 / DAB Rural Area 4 and 6 path
			- DABTU12 | DABTU06 / DAB Typical Urban 12 and 6 path
			- DABSFN / DAB Single Frequency Network (in the VHF range) 7 path
			- WiMAX / WMITUOIPA | WMITUOIPB | WMITUVA60 | WMITUVA120 / Wimax ITU OIP-A, ITU OIP-B, ITU V-A-60, ITU V-A-120
			- See 'WIMAX standards' / WMSUI1A360P90 | WMSUI1A360P75 | WMSUI1A030P90 | WMSUI1A030P75 / SUI 1 (omi ant. 90/75%) , SUI 1 (30 ant. 90/75%)
			- WMSUI2A360P90 | WMSUI2A360P75 | WMSUI2A030P90 | WMSUI2A030P75 / SUI 2 (omi ant. 90/75%) , SUI 2 (30 ant. 90/75%)
			- WMSUI3A360P90 | WMSUI3A360P75 | WMSUI3A030P90 | WMSUI3A030P75 / SUI 3 (omi ant. 90/75%) , SUI 3 (30 ant. 90/75%)
			- WMSUI4A360P90 | WMSUI4A360P75 | WMSUI4A030P90 | WMSUI4A030P75 / SUI 4 (omi ant. 90/75%) , SUI 4 (30 ant. 90/75%)
			- WMSUI5A360P90 | WMSUI5A360P75 | WMSUI5A360P50 | WMSUI5A030P90 | WMSUI5A030P75 | WMSUI5A030P50 / SUI 5 (omi ant. 90/75/50%) , SUI 5 (30 ant. 90/75/50%)
			- WMSUI6A360P90 | WMSUI6A360P75 | WMSUI6A360P50 | WMSUI6A030P90 | WMSUI6A030P75 | WMSUI6A030P50 | / SUI 6 (omi ant. 90/75/50%) , SUI 6 (30 ant. 90/75/50%)
			- WiMAX-MIMO See 'WIMAX-MIMO standards' / WMITUPB3L | WMITUPB3M | WMITUPB3H | WMITUVA60L | WMITUVA60M | WMITUVA60H / ITU PB Low/Medium/High, ITU VA Low/Medium/High
			- LTE See 'LTE standards' / LTEEPA1 | LTEEPA5 LTEEVA5 | LTEEVA70 LTEETU1 | LTEETU5 | LTEETU30 | LTEETU70 | LTEETU200 | LTEETU300 | LTEETU600 LTEMBSFN5 / LTE EPA 1/5Hz, LTE EVA 5/70Hz, LTE ETU 1/5/30/70/200/300/600Hz LTE MBSFN 5Hz
			- LTE HST See 'HST 1 500 A/B, HST 3 500 A/B' / HST1LTE500A | HST3LTE500A | HST1LTE500B | HST3LTE500B / LTE HST 1 500 A/B LTE HST 3 500 A/B
			- LTE-MIMO See 'LTE-MIMO standards' / LMEPA1L | LMEPA1M | LMEPA1H | LMEPA5L | LMEPA5M | LMEPA5H LMEVA5L | LMEVA5M | LMEVA5H | LMEVA70L | LMEVA70M | LMEVA70H LMETU1L | LMETU1M | LMETU1H | LMETU5L | LMETU5M | LMETU5H | LMETU30L | LMETU30M | LMETU30H | LMETU70L | LMETU70M | LMETU70H | LMETU200L | LMETU200M | LMETU200H | LMETU300L | LMETU300M | LMETU300H | LMETU600L | LMETU600M | LMETU600H / LTE EPA 1/5Hz Low/Medium/High, LTE EVA 5/70Hz Low/Medium/High, LTE ETU 1/30/70/300/600Hz Low/Medium/High
			- 1xEVDO See '1xevdo standards' / EVDO1CH1 | EVDO1CH1BC5 | EVDO1CH2 | EVDO1CH2BC5 | EVDO1CH3 | EVDO1CH3BC5 | EVDO1CH4 | EVDO1CH4BC5 | EVDO1CH5 | EVDO1CH5BC5 / 1xEVDO Chan. 1/2/3/4/5
			- WATTERSON See 'Watterson standards' / WATTI1 | WATTI3 | WATTI2 / Watterson I1, I2, I3
			- 802.11n-SISO See '802.11n-SISO standards' / WLANNSMODA | WLANNSMODB | WLANNSMODC | WLANNSMODD | WLANNSMODE | WLANNSMODF / 802.11n SISO Model A/B/C/D/E/F
			- 802.11n-MIMO See '802.11n-MIMO standards' / WLANNMODA | WLANNMODB | WLANNMODC | WLANNMODD | WLANNMODE | WLANNMODF / 802.11n MIMO Model A/B/C/D/E/F
			- 802.11ac-SISO See '802.11ac-SISO standards' / WLANACSMODA|WLANACSMODB|WLANACSMODC|WLANACSMODD|WLANACSMODE|WLANACSMODF / 802.11ac SISO Model A/B/C/D/E/F
			- 802.11ac-MIMO See '802.11ac-MIMO standards' / WLANACMODA | WLANACMODB | WLANACMODC | WLANACMODD | WLANACMODE | WLANACMODF / 802.11ac MIMO Model A/B/C/D/E/F
			- 802.11p See '802.11p channel models' / WLANPRURALLOS | WLANPURBANAPPLOS | WLANPURBANCRONLOS | WLANPHIGHWAYNLOS | WLANPHIGHWAYLOS / 802.11p Channel models: Rural LOS, Urban Approaching LOS, Urban Crossing NLOS, Highway LOS, Highway NLOS
			- 5G NR See '5G NR standards'. / TDLA30D5L | TDLA30D5M | TDLA30D5MA | TDLA30D5H | TDLA30D10L | TDLA30D10M | TDLA30D10MA | TDLA30D10H | TDLB100D400L | TDLB100D400M | TDLB100D400MA | TDLB100D400H | TDLC300D100L | TDLC300D100M | TDLC300D100MA | TDLC300D100H | TDLA30D75L | TDLA30D75M | TDLA30D75MA | TDLA30D75H | TDLA30D300L | TDLA30D300M | TDLA30D300MA | TDLA30D300H |TDLA30D35L | TDLA30D35M | TDLA30D35MA | TDLA30D35H | TDLC60D300L | TDLC60D300M | TDLC60D300MA | TDLC60D300H TDLA30D10S | TDLA30D35S | TDLA30D75S | TDLA30D300S | TDLB100D400S | TDLC60D300S | TDLC300D100 | STDLC300D400S | TDLC300D600S | TDLC300D600L |TDLC300D600M | TDLC300D600H | TDLC300D1200S | TDLC300D1200L | TDLC300D1200M | TDLC300D1200H FR1CDLAUMA | FR1CDLCUMA | FR1CDLCUMA4 | FR1CDLAUMI | FR1CDLCUMI | FR1CDLCUMI2 | FR2CDLAINO | FR2CDLCUMI | FR1CDLBUMA | FR1CDLBUMI / 5G New Radio models TDLA30-5/10 Hz, TDLB100-400 Hz, TDLC300-100 Hz, TDLA30-75/300 Hz, TDLA30-35/10 Hz, TDLC60-300/10 Hz Low/Medium/Medium A/High TDLA30-5/10/35/75/300 Hz, TDLB100-400 Hz, TDLC60-300 Hz, TDLC300-100 Hz SISO, TDLC300-400 Hz SISO, TDLC300-600 Hz SISO, TDLC300-600 Hz Low/Medium/High, TDLC300-1200 Hz SISO, TDLC300-1200 Hz Low/Medium/High FR1 and FR2 CDL models
			- 5G NR High Speed Train scenarios See '5G NR high speed train'. / HST1NR35015|HST1NR35030|HST3NR35015|HST3NR35030 HST1NR50015|HST1NR50030| HST3NR50015|HST3NR50030 / HST1/HST3 350 kmh 15/30 kHz HST1/HST3 500 kmh 15/30 kHz
			- 5G NR Moving propagation See '5G NR moving propagation'. / MPX15|MPX30 MPY15|MPY30 MPZ15|MPZ30 / MP X 15/30kHz SCS MP Y 15/30kHz SCS MP Z 15/30kHz SCS
			- 5G NR MIMO OTA See '5G NR MIMO OTA channel models' / FR1CDLAUMA|FR1CDLBUMA|FR1CDLCUMA FR1CDLCUMA4 FR1CDLAUMI|FR1CDLBUMIFR1CDLCUMI FR1CDLCUMI2 FR2CDLAINO|FR2CDLCUMI / FR1 CDL-A/-B/-C UMa 2x2 FR1 CDL-C UMa 4x4 FR1 CDL-A/-B/-C UMi 4x4 FR1 CDL-C UMi 2x2 FR2 CDL-A InO, FR2 CDL-C UMi \n
			:return: standard: USER| CDMA8| CDMA30| C1DMA30| CDMA100| CDMA0| CDMA3| G6TU3| GTU3| G6TU50| GTU50| G6HT100| GHT100| GRA250| GET50| GET100| HL2A| HL2B| HL2C| HL2D| HL2E| NADC8| NADC50| NADC100| P6TU1| PTU1| P6TU50| PTU50| P6HT100| PHT100| PRA130| PET50| PET100| TTU| TBU| THT| T4ET| G3C1| G3C2| G3C3| G3C4| G3UEC4| G3UEC5| G3UEC6| G3UEC7SE| G3UEC7BE| G3UEC8CQ| G3UEPA3| G3UEPB3| G3UEVA30| G3UEVA120| G3TU3| G3TU50| G3TU120| G3HT120| G3RA120| G3RA250| BD1| MP1| DABRA04| DABRA06| DABTU12| DABTU06| DABSFN| WMSUI1A360P90| WMSUI1A360P75| WMSUI1A030P90| WMSUI1A030P75| WMSUI2A360P90| WMSUI2A360P75| WMSUI2A030P90| WMSUI2A030P75| WMSUI3A360P90| WMSUI3A360P75| WMSUI3A030P90| WMSUI3A030P75| WMSUI4A360P90| WMSUI4A360P75| WMSUI4A030P90| WMSUI4A030P75| WMSUI5A360P90| WMSUI5A360P75| WMSUI5A360P50| WMSUI5A030P90| WMSUI5A030P75| WMSUI5A030P50| WMSUI6A360P90| WMSUI6A360P75| WMSUI6A360P50| WMSUI6A030P90| WMSUI6A030P75| WMSUI6A030P50| WMITUOIPA| WMITUOIPB| WMITUVA60| TDU| TDR| WMITUVA120| GET60| G6HT120| G6HT200| GRA130| GRA300| GRA500| G6TU1P5| G6TU3P6| G6TU6| G6TU60| G6TU100| GHT120| GHT200| GTU1P5| GTU3P6| GTU6| GTU60| GTU100| LMEPA5L| LMEPA5M| LMEPA5H| LMEVA5L| LMEVA5M| LMEVA5H| LMEVA70L| LMEVA70M| LMEVA70H| LMETU70L| LMETU70M| LMETU70H| LMETU300L| LMETU300M| LMETU300H| WMITUPB3L| WMITUPB3M| WMITUPB3H| WMITUVA60L| WMITUVA60M| WMITUVA60H| EVDO1CH1| EVDO1CH1BC5| EVDO1CH2| EVDO1CH2BC5| EVDO1CH3| EVDO1CH3BC5| EVDO1CH4| EVDO1CH4BC5| EVDO1CH5| EVDO1CH5BC5| G3HST1OS| G3HST2TLC| G3HST3TMA| MPLTEETU200| MPLTEPDOPP| T6TU| T6HT| LTEEPA5| LTEEVA5| LTEEVA70| LTEETU70| LTEETU300| G3UEC1| G3UEC2| G3UEC3| G3UEVA3| G3MBSFN3| WATTI1| WATTI3| WATTI2| GTI5| G3HST1OSDU| G3HST2TLCDU| G3HST3TMADU| LTEMBSFN5| LTECQI5| LTEETU30| LMETU30L| LMETU30M| LMETU30H| WLANNMODA| WLANNMODB| WLANNMODC| WLANNMODD| WLANNMODE| WLANNMODF| WLANACMODA| WLANACMODB| WLANACMODC| WLANACMODD| WLANACMODE| WLANACMODF| WLANNSMODA| WLANNSMODB| WLANNSMODC| WLANNSMODD| WLANNSMODE| WLANNSMODF| WLANACSMODA| WLANACSMODB| WLANACSMODC| WLANACSMODD| WLANACSMODE| WLANACSMODF| G3SCMEUMA3| G3SCMEUMA30| G3SCMEUMI3| G3SCMEUMI30| WLANPRURALLOS| WLANPURBANAPPLOS| WLANPURBANCRONLOS| WLANPHIGHWAYNLOS| WLANPHIGHWAYLOS| LTEEPA1| LTEETU1| LTEETU600| LMEPA1L| LMEPA1M| LMEPA1H| LMETU1L| LMETU1M| LMETU1H| LMETU600L| LMETU600M| LMETU600H| GEOSCMEUMA3| GEOSCMEUMA30| GEOSCMEUMI3| GEOSCMEUMI30| TDLA30D5L| TDLA30D5M| TDLA30D5H| TDLA30D10L| TDLA30D10M| TDLA30D10H| TDLB100D400L| TDLB100D400M| TDLB100D400H| TDLC300D100L| TDLC300D100M| TDLC300D100H| TDLA30D75L| TDLA30D75M| TDLA30D75H| TDLA30D300L| TDLA30D300M| TDLA30D300H| TDLA30D5MA| TDLA30D10MA| TDLB100D400MA| TDLC300D100MA| TDLA30D75MA| TDLA30D300MA| TDLA30D5S| TDLA30D10S| TDLB100D400S| TDLC300D100S| TDLA30D75S| TDLA30D300S| TDLA30D35S| TDLC60D300S| TDLA30D35L| TDLA30D35M| TDLA30D35MA| TDLA30D35H| TDLC60D300L| TDLC60D300M| TDLC60D300MA| TDLC60D300H| HST1NR35015| HST1NR35030| HST3NR35015| HST3NR35030| MPY15| MPY30| FR1CDLAUMA| FR1CDLCUMA| FR1CDLAUMI| FR1CDLCUMI| FR2CDLAINO| FR2CDLCUMI| FR1CDLBUMA| FR1CDLBUMI| HST1NR50015| HST1NR50030| HST3NR50015| HST3NR50030| HST1LTE500A| HST3LTE500A| HST1LTE500B| HST3LTE500B| LTEETU5| LTEETU200| LMETU5L| LMETU5M| LMETU5H| LMETU200L| LMETU200M| LMETU200H| TDLC300D600L| TDLC300D600M| TDLC300D600H| TDLC300D1200L| TDLC300D1200M| TDLC300D1200H| TDLC300D400S| TDLC300D600S| TDLC300D1200S| MPX15| MPX30| MPZ15| MPZ30| FR1CDLCUMI2| FR1CDLCUMA4
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:FSIMulator:STANdard?')
		return Conversions.str_to_scalar_enum(response, enums.FadStan)

	def set_value(self, standard: enums.FadStan) -> None:
		"""SCPI: [SOURce<HW>]:FSIMulator:STANdard \n
		Snippet: driver.source.fsimulator.standard.set_value(standard = enums.FadStan.BD1) \n
		Selects a predefined fading simulator setting which complies with the test specifications found in the common mobile
		radio standards. For a detailed summary of all of the default settings, see 'Predefined fading settings'.
			Table Header: Standard / Test Case / <Predefined_Standard> / Description \n
			- USER / USER parameter cannot be set. A query returns this value if a user-defined Fading setting was loaded or if one of the associated settings was changed subsequent to the selection of a standard
			- CDMA See 'CDMA standards' / CDMA0 | CDMA3 | CDMA8 | CDMA30 | C1DMA30 | CDMA100 / CDMA 5 (0 km/h) , CDMA6 (3km/h) , CDMA1 (8 km/h) , CDMA2 (30 km/h) , CDMA3 (30 km/h, 1 path) , CDMA4 (100km/h)
			- GSM See 'GSM standards' / GTU1P5 | G6TU1P5 | GTU3P6 | G6TU3P6 | GTU3 | G6TU3 | GTU6 | G6TU6 | GTU50 | G6TU50 | G6TU100 | G6TU60 / GSM Typical Urban 1,5/3/3,6/6/50/60/100 km/h, 6 and 12 path
			- GHT100 | G6HT100 | GHT120 | G6HT120 | GHT200 | G6HT200 / GSM Hilly Terrain 100/120 km/h, 6 and 12 path
			- GRA130 | GRA250 | GRA300 | GRA500 / GSM Rural Area 130/250/300/500 km/h, 6 path
			- GET50 | GET60 | GET100 / GSM Equal Test 50/60/100 km/h, 6 path
			- GTI5 / GSM typical case for very small cells, 5km/h, 2 path
			- NADC See 'NADC standards' / NADC8 | NADC50 | NADC100 / NADC 8/50/100 km/h, 2 path
			- DCS1800/PCS1900 See 'PCN standards' / P6TU1 | PTU1 | P6TU50 | PTU50 / Typical Urban 1.5/50 km/h, 6 and 12 path
			- P6HT100 | PHT100 / Hilly Terrain 100 km/h, 6 and 12 path
			- PRA130 / Rural Area 130 km/h, 6 path
			- PET50 | PET100 / Equal Test 50/100 km/h, 6 path
			- TETRA See 'TETRA standards' / TTU | T6TU / TETRA Typical Urban 50km/h, 2 path and 6 path
			- TBU / TETRA 2 path
			- THT | T6HT / TETRA Hilly Terrain 200 km/h, 2 path and 6 path
			- T4ET / TETRA Equal Test 200 km/h, 4 path
			- TDR / TETRA Direct Mode Rural Propagation Model 1 path
			- TDU / TETRA Mode Urban Propagation Model 1 path
			- 3GPP FDD See '3GPP standards' / G3C1 | G3C2 | G3C3 | G3C4 / 3GPP FDD Test Case x (BS)
			- G3UEC1 | G3UEC2 | G3UEC3 | G3UEC4 | G3UEC5 | G3UEC6 / 3GPP FDD UE Test Case x (UE)
			- G3UEC7SE / 3GPP FDD UE Sector (UE)
			- G3UEC7BE / 3GPP FDD Beam (UE)
			- G3UEC8CQ / 3GPP FDD CQI (UE)
			- G3UEPA3 | G3UEPB3 / 3GPP FDD Pedestrian A 4 path / B 6 path (UE)
			- G3UEVA3 | G3UEVA30 | G3UEVA120 / 3GPP FDD Vehicular A 6 path (UE)
			- G3MBSFN3 / 3GPP MBSFN
			- G3TU3 | G3TU50 | G3TU120 / 3GPP FDD Typical Urban 20 path
			- G3HT120 / 3GPP FDD Hilly Terrain 20 path
			- G3RA120 | G3RA250 / 3GPP FDD Rural Area 10 path
			- BD1 / 3GPP Birth Death 2 path
			- Moving Propagation / MD1 / 3GPP Moving Propagation 'Ref. + Moving Channel', 2path
			- See '3GPP/LTE moving propagation' / MPLTEETU200 / 3GPP Moving Propagation scenario 1 'ETU200Hz', according to the test case 3GPP TS36.141, annex B4.
			- MPLTEPDOPP / 3GPP Moving Propagation scenario 2 'AWGN', according to the test case 3GPP TS36.141, annex B4.
			- 3GPP High Speed Train scenarios / G3HST1OS | G3HST1OSDU / 3GPP HST1 'Open space', according to the test case 3GPP TS25.141, annex D.4A and 3GPP TS36.141, annex B.3
			- See '3GPP/LTE high speed train' / G3HST2TLC | G3HST2TLCDU / 3GPP HST2 'Tunnel with leaky cable', according to the test case 3GPP TS25.141, annex D.4A
			- G3HST3TMA | G3HST3TMADU / 3GPP HST3 'Tunnel for multi-antennas', according to the test case 3GPP TS25.141, annex D.4A and 3GPP TS36.141, annex B.3
			- SCME/Geo SCME UMi 3kmh/30kmh SCME/Geo SCME UMa 3kmh/30kmh See 'SCM and SCME channel models for MIMO OTA' / G3SCMEUMA3 | G3SCMEUMA30 | G3SCMEUMI3 | G3SCMEUMI30 GEOSCMEUMA3|GEOSCMEUMA30|GEOSCMEUMI3|GEOSCMEUMI30 / SCME/Geo SCME Urban Micro/Macro-Cell Channel, 3 km/h and 30 km/h
			- WLAN See 'WLAN standards' / HL2A | HL2B | HL2C | HL2D | HL2E / WLAN HyperLan 18 path
			- DAB See 'DAB standards' / DABRA04 | DABRA06 / DAB Rural Area 4 and 6 path
			- DABTU12 | DABTU06 / DAB Typical Urban 12 and 6 path
			- DABSFN / DAB Single Frequency Network (in the VHF range) 7 path
			- WiMAX / WMITUOIPA | WMITUOIPB | WMITUVA60 | WMITUVA120 / Wimax ITU OIP-A, ITU OIP-B, ITU V-A-60, ITU V-A-120
			- See 'WIMAX standards' / WMSUI1A360P90 | WMSUI1A360P75 | WMSUI1A030P90 | WMSUI1A030P75 / SUI 1 (omi ant. 90/75%) , SUI 1 (30 ant. 90/75%)
			- WMSUI2A360P90 | WMSUI2A360P75 | WMSUI2A030P90 | WMSUI2A030P75 / SUI 2 (omi ant. 90/75%) , SUI 2 (30 ant. 90/75%)
			- WMSUI3A360P90 | WMSUI3A360P75 | WMSUI3A030P90 | WMSUI3A030P75 / SUI 3 (omi ant. 90/75%) , SUI 3 (30 ant. 90/75%)
			- WMSUI4A360P90 | WMSUI4A360P75 | WMSUI4A030P90 | WMSUI4A030P75 / SUI 4 (omi ant. 90/75%) , SUI 4 (30 ant. 90/75%)
			- WMSUI5A360P90 | WMSUI5A360P75 | WMSUI5A360P50 | WMSUI5A030P90 | WMSUI5A030P75 | WMSUI5A030P50 / SUI 5 (omi ant. 90/75/50%) , SUI 5 (30 ant. 90/75/50%)
			- WMSUI6A360P90 | WMSUI6A360P75 | WMSUI6A360P50 | WMSUI6A030P90 | WMSUI6A030P75 | WMSUI6A030P50 | / SUI 6 (omi ant. 90/75/50%) , SUI 6 (30 ant. 90/75/50%)
			- WiMAX-MIMO See 'WIMAX-MIMO standards' / WMITUPB3L | WMITUPB3M | WMITUPB3H | WMITUVA60L | WMITUVA60M | WMITUVA60H / ITU PB Low/Medium/High, ITU VA Low/Medium/High
			- LTE See 'LTE standards' / LTEEPA1 | LTEEPA5 LTEEVA5 | LTEEVA70 LTEETU1 | LTEETU5 | LTEETU30 | LTEETU70 | LTEETU200 | LTEETU300 | LTEETU600 LTEMBSFN5 / LTE EPA 1/5Hz, LTE EVA 5/70Hz, LTE ETU 1/5/30/70/200/300/600Hz LTE MBSFN 5Hz
			- LTE HST See 'HST 1 500 A/B, HST 3 500 A/B' / HST1LTE500A | HST3LTE500A | HST1LTE500B | HST3LTE500B / LTE HST 1 500 A/B LTE HST 3 500 A/B
			- LTE-MIMO See 'LTE-MIMO standards' / LMEPA1L | LMEPA1M | LMEPA1H | LMEPA5L | LMEPA5M | LMEPA5H LMEVA5L | LMEVA5M | LMEVA5H | LMEVA70L | LMEVA70M | LMEVA70H LMETU1L | LMETU1M | LMETU1H | LMETU5L | LMETU5M | LMETU5H | LMETU30L | LMETU30M | LMETU30H | LMETU70L | LMETU70M | LMETU70H | LMETU200L | LMETU200M | LMETU200H | LMETU300L | LMETU300M | LMETU300H | LMETU600L | LMETU600M | LMETU600H / LTE EPA 1/5Hz Low/Medium/High, LTE EVA 5/70Hz Low/Medium/High, LTE ETU 1/30/70/300/600Hz Low/Medium/High
			- 1xEVDO See '1xevdo standards' / EVDO1CH1 | EVDO1CH1BC5 | EVDO1CH2 | EVDO1CH2BC5 | EVDO1CH3 | EVDO1CH3BC5 | EVDO1CH4 | EVDO1CH4BC5 | EVDO1CH5 | EVDO1CH5BC5 / 1xEVDO Chan. 1/2/3/4/5
			- WATTERSON See 'Watterson standards' / WATTI1 | WATTI3 | WATTI2 / Watterson I1, I2, I3
			- 802.11n-SISO See '802.11n-SISO standards' / WLANNSMODA | WLANNSMODB | WLANNSMODC | WLANNSMODD | WLANNSMODE | WLANNSMODF / 802.11n SISO Model A/B/C/D/E/F
			- 802.11n-MIMO See '802.11n-MIMO standards' / WLANNMODA | WLANNMODB | WLANNMODC | WLANNMODD | WLANNMODE | WLANNMODF / 802.11n MIMO Model A/B/C/D/E/F
			- 802.11ac-SISO See '802.11ac-SISO standards' / WLANACSMODA|WLANACSMODB|WLANACSMODC|WLANACSMODD|WLANACSMODE|WLANACSMODF / 802.11ac SISO Model A/B/C/D/E/F
			- 802.11ac-MIMO See '802.11ac-MIMO standards' / WLANACMODA | WLANACMODB | WLANACMODC | WLANACMODD | WLANACMODE | WLANACMODF / 802.11ac MIMO Model A/B/C/D/E/F
			- 802.11p See '802.11p channel models' / WLANPRURALLOS | WLANPURBANAPPLOS | WLANPURBANCRONLOS | WLANPHIGHWAYNLOS | WLANPHIGHWAYLOS / 802.11p Channel models: Rural LOS, Urban Approaching LOS, Urban Crossing NLOS, Highway LOS, Highway NLOS
			- 5G NR See '5G NR standards'. / TDLA30D5L | TDLA30D5M | TDLA30D5MA | TDLA30D5H | TDLA30D10L | TDLA30D10M | TDLA30D10MA | TDLA30D10H | TDLB100D400L | TDLB100D400M | TDLB100D400MA | TDLB100D400H | TDLC300D100L | TDLC300D100M | TDLC300D100MA | TDLC300D100H | TDLA30D75L | TDLA30D75M | TDLA30D75MA | TDLA30D75H | TDLA30D300L | TDLA30D300M | TDLA30D300MA | TDLA30D300H |TDLA30D35L | TDLA30D35M | TDLA30D35MA | TDLA30D35H | TDLC60D300L | TDLC60D300M | TDLC60D300MA | TDLC60D300H TDLA30D10S | TDLA30D35S | TDLA30D75S | TDLA30D300S | TDLB100D400S | TDLC60D300S | TDLC300D100 | STDLC300D400S | TDLC300D600S | TDLC300D600L |TDLC300D600M | TDLC300D600H | TDLC300D1200S | TDLC300D1200L | TDLC300D1200M | TDLC300D1200H FR1CDLAUMA | FR1CDLCUMA | FR1CDLCUMA4 | FR1CDLAUMI | FR1CDLCUMI | FR1CDLCUMI2 | FR2CDLAINO | FR2CDLCUMI | FR1CDLBUMA | FR1CDLBUMI / 5G New Radio models TDLA30-5/10 Hz, TDLB100-400 Hz, TDLC300-100 Hz, TDLA30-75/300 Hz, TDLA30-35/10 Hz, TDLC60-300/10 Hz Low/Medium/Medium A/High TDLA30-5/10/35/75/300 Hz, TDLB100-400 Hz, TDLC60-300 Hz, TDLC300-100 Hz SISO, TDLC300-400 Hz SISO, TDLC300-600 Hz SISO, TDLC300-600 Hz Low/Medium/High, TDLC300-1200 Hz SISO, TDLC300-1200 Hz Low/Medium/High FR1 and FR2 CDL models
			- 5G NR High Speed Train scenarios See '5G NR high speed train'. / HST1NR35015|HST1NR35030|HST3NR35015|HST3NR35030 HST1NR50015|HST1NR50030| HST3NR50015|HST3NR50030 / HST1/HST3 350 kmh 15/30 kHz HST1/HST3 500 kmh 15/30 kHz
			- 5G NR Moving propagation See '5G NR moving propagation'. / MPX15|MPX30 MPY15|MPY30 MPZ15|MPZ30 / MP X 15/30kHz SCS MP Y 15/30kHz SCS MP Z 15/30kHz SCS
			- 5G NR MIMO OTA See '5G NR MIMO OTA channel models' / FR1CDLAUMA|FR1CDLBUMA|FR1CDLCUMA FR1CDLCUMA4 FR1CDLAUMI|FR1CDLBUMIFR1CDLCUMI FR1CDLCUMI2 FR2CDLAINO|FR2CDLCUMI / FR1 CDL-A/-B/-C UMa 2x2 FR1 CDL-C UMa 4x4 FR1 CDL-A/-B/-C UMi 4x4 FR1 CDL-C UMi 2x2 FR2 CDL-A InO, FR2 CDL-C UMi \n
			:param standard: USER| CDMA8| CDMA30| C1DMA30| CDMA100| CDMA0| CDMA3| G6TU3| GTU3| G6TU50| GTU50| G6HT100| GHT100| GRA250| GET50| GET100| HL2A| HL2B| HL2C| HL2D| HL2E| NADC8| NADC50| NADC100| P6TU1| PTU1| P6TU50| PTU50| P6HT100| PHT100| PRA130| PET50| PET100| TTU| TBU| THT| T4ET| G3C1| G3C2| G3C3| G3C4| G3UEC4| G3UEC5| G3UEC6| G3UEC7SE| G3UEC7BE| G3UEC8CQ| G3UEPA3| G3UEPB3| G3UEVA30| G3UEVA120| G3TU3| G3TU50| G3TU120| G3HT120| G3RA120| G3RA250| BD1| MP1| DABRA04| DABRA06| DABTU12| DABTU06| DABSFN| WMSUI1A360P90| WMSUI1A360P75| WMSUI1A030P90| WMSUI1A030P75| WMSUI2A360P90| WMSUI2A360P75| WMSUI2A030P90| WMSUI2A030P75| WMSUI3A360P90| WMSUI3A360P75| WMSUI3A030P90| WMSUI3A030P75| WMSUI4A360P90| WMSUI4A360P75| WMSUI4A030P90| WMSUI4A030P75| WMSUI5A360P90| WMSUI5A360P75| WMSUI5A360P50| WMSUI5A030P90| WMSUI5A030P75| WMSUI5A030P50| WMSUI6A360P90| WMSUI6A360P75| WMSUI6A360P50| WMSUI6A030P90| WMSUI6A030P75| WMSUI6A030P50| WMITUOIPA| WMITUOIPB| WMITUVA60| TDU| TDR| WMITUVA120| GET60| G6HT120| G6HT200| GRA130| GRA300| GRA500| G6TU1P5| G6TU3P6| G6TU6| G6TU60| G6TU100| GHT120| GHT200| GTU1P5| GTU3P6| GTU6| GTU60| GTU100| LMEPA5L| LMEPA5M| LMEPA5H| LMEVA5L| LMEVA5M| LMEVA5H| LMEVA70L| LMEVA70M| LMEVA70H| LMETU70L| LMETU70M| LMETU70H| LMETU300L| LMETU300M| LMETU300H| WMITUPB3L| WMITUPB3M| WMITUPB3H| WMITUVA60L| WMITUVA60M| WMITUVA60H| EVDO1CH1| EVDO1CH1BC5| EVDO1CH2| EVDO1CH2BC5| EVDO1CH3| EVDO1CH3BC5| EVDO1CH4| EVDO1CH4BC5| EVDO1CH5| EVDO1CH5BC5| G3HST1OS| G3HST2TLC| G3HST3TMA| MPLTEETU200| MPLTEPDOPP| T6TU| T6HT| LTEEPA5| LTEEVA5| LTEEVA70| LTEETU70| LTEETU300| G3UEC1| G3UEC2| G3UEC3| G3UEVA3| G3MBSFN3| WATTI1| WATTI3| WATTI2| GTI5| G3HST1OSDU| G3HST2TLCDU| G3HST3TMADU| LTEMBSFN5| LTECQI5| LTEETU30| LMETU30L| LMETU30M| LMETU30H| WLANNMODA| WLANNMODB| WLANNMODC| WLANNMODD| WLANNMODE| WLANNMODF| WLANACMODA| WLANACMODB| WLANACMODC| WLANACMODD| WLANACMODE| WLANACMODF| WLANNSMODA| WLANNSMODB| WLANNSMODC| WLANNSMODD| WLANNSMODE| WLANNSMODF| WLANACSMODA| WLANACSMODB| WLANACSMODC| WLANACSMODD| WLANACSMODE| WLANACSMODF| G3SCMEUMA3| G3SCMEUMA30| G3SCMEUMI3| G3SCMEUMI30| WLANPRURALLOS| WLANPURBANAPPLOS| WLANPURBANCRONLOS| WLANPHIGHWAYNLOS| WLANPHIGHWAYLOS| LTEEPA1| LTEETU1| LTEETU600| LMEPA1L| LMEPA1M| LMEPA1H| LMETU1L| LMETU1M| LMETU1H| LMETU600L| LMETU600M| LMETU600H| GEOSCMEUMA3| GEOSCMEUMA30| GEOSCMEUMI3| GEOSCMEUMI30| TDLA30D5L| TDLA30D5M| TDLA30D5H| TDLA30D10L| TDLA30D10M| TDLA30D10H| TDLB100D400L| TDLB100D400M| TDLB100D400H| TDLC300D100L| TDLC300D100M| TDLC300D100H| TDLA30D75L| TDLA30D75M| TDLA30D75H| TDLA30D300L| TDLA30D300M| TDLA30D300H| TDLA30D5MA| TDLA30D10MA| TDLB100D400MA| TDLC300D100MA| TDLA30D75MA| TDLA30D300MA| TDLA30D5S| TDLA30D10S| TDLB100D400S| TDLC300D100S| TDLA30D75S| TDLA30D300S| TDLA30D35S| TDLC60D300S| TDLA30D35L| TDLA30D35M| TDLA30D35MA| TDLA30D35H| TDLC60D300L| TDLC60D300M| TDLC60D300MA| TDLC60D300H| HST1NR35015| HST1NR35030| HST3NR35015| HST3NR35030| MPY15| MPY30| FR1CDLAUMA| FR1CDLCUMA| FR1CDLAUMI| FR1CDLCUMI| FR2CDLAINO| FR2CDLCUMI| FR1CDLBUMA| FR1CDLBUMI| HST1NR50015| HST1NR50030| HST3NR50015| HST3NR50030| HST1LTE500A| HST3LTE500A| HST1LTE500B| HST3LTE500B| LTEETU5| LTEETU200| LMETU5L| LMETU5M| LMETU5H| LMETU200L| LMETU200M| LMETU200H| TDLC300D600L| TDLC300D600M| TDLC300D600H| TDLC300D1200L| TDLC300D1200M| TDLC300D1200H| TDLC300D400S| TDLC300D600S| TDLC300D1200S| MPX15| MPX30| MPZ15| MPZ30| FR1CDLCUMI2| FR1CDLCUMA4
		"""
		param = Conversions.enum_scalar_to_str(standard, enums.FadStan)
		self._core.io.write(f'SOURce<HwInstance>:FSIMulator:STANdard {param}')
