from ......Internal.Core import Core
from ......Internal.CommandsGroup import CommandsGroup
from ......Internal import Conversions
from ...... import enums


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class TriggerCls:
	"""Trigger commands group definition. 26 total commands, 5 Subgroups, 5 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("trigger", core, parent)

	@property
	def arm(self):
		"""arm commands group. 1 Sub-classes, 0 commands."""
		if not hasattr(self, '_arm'):
			from .Arm import ArmCls
			self._arm = ArmCls(self._core, self._cmd_group)
		return self._arm

	@property
	def execute(self):
		"""execute commands group. 0 Sub-classes, 1 commands."""
		if not hasattr(self, '_execute'):
			from .Execute import ExecuteCls
			self._execute = ExecuteCls(self._core, self._cmd_group)
		return self._execute

	@property
	def external(self):
		"""external commands group. 1 Sub-classes, 2 commands."""
		if not hasattr(self, '_external'):
			from .External import ExternalCls
			self._external = ExternalCls(self._core, self._cmd_group)
		return self._external

	@property
	def obaseband(self):
		"""obaseband commands group. 0 Sub-classes, 2 commands."""
		if not hasattr(self, '_obaseband'):
			from .Obaseband import ObasebandCls
			self._obaseband = ObasebandCls(self._core, self._cmd_group)
		return self._obaseband

	@property
	def output(self):
		"""output commands group. 10 Sub-classes, 0 commands."""
		if not hasattr(self, '_output'):
			from .Output import OutputCls
			self._output = OutputCls(self._core, self._cmd_group)
		return self._output

	# noinspection PyTypeChecker
	def get_rmode(self) -> enums.TrigRunMode:
		"""SCPI: [SOURce<HW>]:BB:WLNN:TRIGger:RMODe \n
		Snippet: value: enums.TrigRunMode = driver.source.bb.wlnn.trigger.get_rmode() \n
		The command queries the current status of signal generation for all trigger modes with IEEE 802.11 WLAN modulation on. \n
			:return: rmode: RUN| STOP RUN the signal is generated. A trigger event occurred in the triggered mode. STOP the signal is not generated. A trigger event did not occur in the triggered modes, or signal generation was stopped by the command :BB:WLNN:TRIG:ARM:EXECute (armed trigger modes only) .
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:WLNN:TRIGger:RMODe?')
		return Conversions.str_to_scalar_enum(response, enums.TrigRunMode)

	def get_slength(self) -> int:
		"""SCPI: [SOURce<HW>]:BB:WLNN:TRIGger:SLENgth \n
		Snippet: value: int = driver.source.bb.wlnn.trigger.get_slength() \n
		The command defines the length of the signal sequence to be output in the 'Single' trigger mode
		([:SOURce<hw>]:BB:WLNN[:TRIGger]:SEQuence is set to SING) . The input is made in terms of samples. It is possible to
		output deliberately just part of the frame, an exact sequence of the frame, or a defined number of repetitions of the
		frame. \n
			:return: slength: integer Range: 1 to (2^32) - 1, Unit: sample
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:WLNN:TRIGger:SLENgth?')
		return Conversions.str_to_int(response)

	def set_slength(self, slength: int) -> None:
		"""SCPI: [SOURce<HW>]:BB:WLNN:TRIGger:SLENgth \n
		Snippet: driver.source.bb.wlnn.trigger.set_slength(slength = 1) \n
		The command defines the length of the signal sequence to be output in the 'Single' trigger mode
		([:SOURce<hw>]:BB:WLNN[:TRIGger]:SEQuence is set to SING) . The input is made in terms of samples. It is possible to
		output deliberately just part of the frame, an exact sequence of the frame, or a defined number of repetitions of the
		frame. \n
			:param slength: integer Range: 1 to (2^32) - 1, Unit: sample
		"""
		param = Conversions.decimal_value_to_str(slength)
		self._core.io.write(f'SOURce<HwInstance>:BB:WLNN:TRIGger:SLENgth {param}')

	# noinspection PyTypeChecker
	def get_sl_unit(self) -> enums.UnitSlB:
		"""SCPI: [SOURce<HW>]:BB:WLNN:TRIGger:SLUNit \n
		Snippet: value: enums.UnitSlB = driver.source.bb.wlnn.trigger.get_sl_unit() \n
		Defines the unit for the entry of the length of the signal sequence ([:SOURce<hw>]:BB:WLNN:TRIGger:SLENgth) to be output
		in the single trigger mode ([:SOURce<hw>]:BB:WLNN[:TRIGger]:SEQuence is set to SINGle) . \n
			:return: sl_unit: SAMPle| SEQuence SAMPle Unit Sample. A single sample is generated after a trigger event. SEQuence Unit Sequence Length. A single sequence is generated after a trigger event.
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:WLNN:TRIGger:SLUNit?')
		return Conversions.str_to_scalar_enum(response, enums.UnitSlB)

	def set_sl_unit(self, sl_unit: enums.UnitSlB) -> None:
		"""SCPI: [SOURce<HW>]:BB:WLNN:TRIGger:SLUNit \n
		Snippet: driver.source.bb.wlnn.trigger.set_sl_unit(sl_unit = enums.UnitSlB.SAMPle) \n
		Defines the unit for the entry of the length of the signal sequence ([:SOURce<hw>]:BB:WLNN:TRIGger:SLENgth) to be output
		in the single trigger mode ([:SOURce<hw>]:BB:WLNN[:TRIGger]:SEQuence is set to SINGle) . \n
			:param sl_unit: SAMPle| SEQuence SAMPle Unit Sample. A single sample is generated after a trigger event. SEQuence Unit Sequence Length. A single sequence is generated after a trigger event.
		"""
		param = Conversions.enum_scalar_to_str(sl_unit, enums.UnitSlB)
		self._core.io.write(f'SOURce<HwInstance>:BB:WLNN:TRIGger:SLUNit {param}')

	# noinspection PyTypeChecker
	def get_source(self) -> enums.TriggerSourceC:
		"""SCPI: [SOURce<HW>]:BB:WLNN:TRIGger:SOURce \n
		Snippet: value: enums.TriggerSourceC = driver.source.bb.wlnn.trigger.get_source() \n
			INTRO_CMD_HELP: Selects the trigger signal source and determines the way the triggering is executed. Provided are: \n
			- Internal triggering by a command (INTernal)
			- External trigger signal via one of the local or global connectors
			Table Header:  \n
			- EGT1|EGT2: External global trigger
			- EGC1|EGC2: External global clock
			- ELTRigger: External local trigger
			- ELCLock: External local clock
			- Internal triggering by a signal from the other basebands (INTA|INTB)
			- In primary-secondary instrument mode, the external baseband synchronization
		signal (BBSY)
			- OBASeband|BEXTernal|EXTernal: Setting only Provided only for backward
		compatibility with other Rohde & Schwarz signal generators. The R&S SMW accepts these values and maps them automatically
		as follows: EXTernal = EGT1, BEXTernal = EGT2, OBASeband = INTA or INTB (depending on the current baseband) \n
			:return: source: INTB| INTernal| OBASeband| EGT1| EGT2| EGC1| EGC2| ELTRigger| INTA| ELCLock| BEXTernal| EXTernal | BBSY
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:WLNN:TRIGger:SOURce?')
		return Conversions.str_to_scalar_enum(response, enums.TriggerSourceC)

	def set_source(self, source: enums.TriggerSourceC) -> None:
		"""SCPI: [SOURce<HW>]:BB:WLNN:TRIGger:SOURce \n
		Snippet: driver.source.bb.wlnn.trigger.set_source(source = enums.TriggerSourceC.BBSY) \n
			INTRO_CMD_HELP: Selects the trigger signal source and determines the way the triggering is executed. Provided are: \n
			- Internal triggering by a command (INTernal)
			- External trigger signal via one of the local or global connectors
			Table Header:  \n
			- EGT1|EGT2: External global trigger
			- EGC1|EGC2: External global clock
			- ELTRigger: External local trigger
			- ELCLock: External local clock
			- Internal triggering by a signal from the other basebands (INTA|INTB)
			- In primary-secondary instrument mode, the external baseband synchronization
		signal (BBSY)
			- OBASeband|BEXTernal|EXTernal: Setting only Provided only for backward
		compatibility with other Rohde & Schwarz signal generators. The R&S SMW accepts these values and maps them automatically
		as follows: EXTernal = EGT1, BEXTernal = EGT2, OBASeband = INTA or INTB (depending on the current baseband) \n
			:param source: INTB| INTernal| OBASeband| EGT1| EGT2| EGC1| EGC2| ELTRigger| INTA| ELCLock| BEXTernal| EXTernal | BBSY
		"""
		param = Conversions.enum_scalar_to_str(source, enums.TriggerSourceC)
		self._core.io.write(f'SOURce<HwInstance>:BB:WLNN:TRIGger:SOURce {param}')

	# noinspection PyTypeChecker
	def get_sequence(self) -> enums.DmTrigMode:
		"""SCPI: [SOURce<HW>]:BB:WLNN:[TRIGger]:SEQuence \n
		Snippet: value: enums.DmTrigMode = driver.source.bb.wlnn.trigger.get_sequence() \n
			INTRO_CMD_HELP: Selects the trigger mode: \n
			- AUTO = auto
			- RETRigger = retrigger
			- AAUTo = armed auto
			- ARETrigger = armed retrigger
			- SINGle = single \n
			:return: sequence: AUTO| RETRigger| AAUTo| ARETrigger| SINGle
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:WLNN:TRIGger:SEQuence?')
		return Conversions.str_to_scalar_enum(response, enums.DmTrigMode)

	def set_sequence(self, sequence: enums.DmTrigMode) -> None:
		"""SCPI: [SOURce<HW>]:BB:WLNN:[TRIGger]:SEQuence \n
		Snippet: driver.source.bb.wlnn.trigger.set_sequence(sequence = enums.DmTrigMode.AAUTo) \n
			INTRO_CMD_HELP: Selects the trigger mode: \n
			- AUTO = auto
			- RETRigger = retrigger
			- AAUTo = armed auto
			- ARETrigger = armed retrigger
			- SINGle = single \n
			:param sequence: AUTO| RETRigger| AAUTo| ARETrigger| SINGle
		"""
		param = Conversions.enum_scalar_to_str(sequence, enums.DmTrigMode)
		self._core.io.write(f'SOURce<HwInstance>:BB:WLNN:TRIGger:SEQuence {param}')

	def clone(self) -> 'TriggerCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = TriggerCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
