from typing import List

from ......Internal.Core import Core
from ......Internal.CommandsGroup import CommandsGroup
from ......Internal import Conversions


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class SettingCls:
	"""Setting commands group definition. 16 total commands, 2 Subgroups, 4 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("setting", core, parent)

	@property
	def tmodel(self):
		"""tmodel commands group. 2 Sub-classes, 0 commands."""
		if not hasattr(self, '_tmodel'):
			from .Tmodel import TmodelCls
			self._tmodel = TmodelCls(self._core, self._cmd_group)
		return self._tmodel

	@property
	def tsetup(self):
		"""tsetup commands group. 2 Sub-classes, 0 commands."""
		if not hasattr(self, '_tsetup'):
			from .Tsetup import TsetupCls
			self._tsetup = TsetupCls(self._core, self._cmd_group)
		return self._tsetup

	def get_catalog(self) -> List[str]:
		"""SCPI: [SOURce<HW>]:BB:W3GPp:SETTing:CATalog \n
		Snippet: value: List[str] = driver.source.bb.w3Gpp.setting.get_catalog() \n
		This command reads out the files with 3GPP FDD settings in the default directory. The default directory is set using
		command method RsSmw.MassMemory.currentDirectory. Only files with the file extension *.3g are listed. \n
			:return: catalog: string
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:W3GPp:SETTing:CATalog?')
		return Conversions.str_to_str_list(response)

	def delete(self, filename: str) -> None:
		"""SCPI: [SOURce<HW>]:BB:W3GPp:SETTing:DELete \n
		Snippet: driver.source.bb.w3Gpp.setting.delete(filename = '1') \n
		This command deletes the selected file with 3GPP FDD settings. The directory is set using command method RsSmw.MassMemory.
		currentDirectory. A path can also be specified, in which case the files in the specified directory are read. The file
		extension can be omitted. Only files with the file extension *.3g are deleted. \n
			:param filename: file_name
		"""
		param = Conversions.value_to_quoted_str(filename)
		self._core.io.write(f'SOURce<HwInstance>:BB:W3GPp:SETTing:DELete {param}')

	def load(self, filename: str) -> None:
		"""SCPI: [SOURce<HW>]:BB:W3GPp:SETTing:LOAD \n
		Snippet: driver.source.bb.w3Gpp.setting.load(filename = '1') \n
		This command loads the selected file with 3GPP FDD settings. The directory is set using command method RsSmw.MassMemory.
		currentDirectory. A path can also be specified, in which case the files in the specified directory are read. The file
		extension can be omitted. Only files with the file extension *.3g are loaded. \n
			:param filename: file_name
		"""
		param = Conversions.value_to_quoted_str(filename)
		self._core.io.write(f'SOURce<HwInstance>:BB:W3GPp:SETTing:LOAD {param}')

	def set_store(self, filename: str) -> None:
		"""SCPI: [SOURce<HW>]:BB:W3GPp:SETTing:STORe \n
		Snippet: driver.source.bb.w3Gpp.setting.set_store(filename = '1') \n
		This command stores the current 3GPP FDD settings into the selected file. The directory is set using command method RsSmw.
		MassMemory.currentDirectory. A path can also be specified, in which case the files in the specified directory are read.
		Only enter the file name. 3GPP FDD settings are stored as files with the specific file extensions *.3g. \n
			:param filename: string
		"""
		param = Conversions.value_to_quoted_str(filename)
		self._core.io.write(f'SOURce<HwInstance>:BB:W3GPp:SETTing:STORe {param}')

	def clone(self) -> 'SettingCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = SettingCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
