from .......Internal.Core import Core
from .......Internal.CommandsGroup import CommandsGroup
from .......Internal import Conversions
from .......Internal.RepeatedCapability import RepeatedCapability
from ....... import enums
from ....... import repcap


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class BurstCls:
	"""Burst commands group definition. 1 total commands, 0 Subgroups, 1 group commands
	Repeated Capability: TwoStreams, default value after init: TwoStreams.Nr1"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("burst", core, parent)
		self._cmd_group.rep_cap = RepeatedCapability(self._cmd_group.group_name, 'repcap_twoStreams_get', 'repcap_twoStreams_set', repcap.TwoStreams.Nr1)

	def repcap_twoStreams_set(self, twoStreams: repcap.TwoStreams) -> None:
		"""Repeated Capability default value numeric suffix.
		This value is used, if you do not explicitely set it in the child set/get methods, or if you leave it to TwoStreams.Default
		Default value after init: TwoStreams.Nr1"""
		self._cmd_group.set_repcap_enum_value(twoStreams)

	def repcap_twoStreams_get(self) -> repcap.TwoStreams:
		"""Returns the current default repeated capability for the child set/get methods"""
		# noinspection PyTypeChecker
		return self._cmd_group.get_repcap_enum_value()

	def set(self, burst_length: enums.WcdmaUlDtxBurstLen, twoStreams=repcap.TwoStreams.Default) -> None:
		"""SCPI: [SOURce<HW>]:BB:W3GPp:MSTation:UDTX:BURSt<CH> \n
		Snippet: driver.source.bb.w3Gpp.mstation.udtx.burst.set(burst_length = enums.WcdmaUlDtxBurstLen._1, twoStreams = repcap.TwoStreams.Default) \n
		Determines the uplink DPCCH burst length in subframes without the preamble and postamble, when the corresponding UE-DTX
		cycle is applied. \n
			:param burst_length: 1| 2| 5
			:param twoStreams: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Burst')
		"""
		param = Conversions.enum_scalar_to_str(burst_length, enums.WcdmaUlDtxBurstLen)
		twoStreams_cmd_val = self._cmd_group.get_repcap_cmd_value(twoStreams, repcap.TwoStreams)
		self._core.io.write(f'SOURce<HwInstance>:BB:W3GPp:MSTation:UDTX:BURSt{twoStreams_cmd_val} {param}')

	# noinspection PyTypeChecker
	def get(self, twoStreams=repcap.TwoStreams.Default) -> enums.WcdmaUlDtxBurstLen:
		"""SCPI: [SOURce<HW>]:BB:W3GPp:MSTation:UDTX:BURSt<CH> \n
		Snippet: value: enums.WcdmaUlDtxBurstLen = driver.source.bb.w3Gpp.mstation.udtx.burst.get(twoStreams = repcap.TwoStreams.Default) \n
		Determines the uplink DPCCH burst length in subframes without the preamble and postamble, when the corresponding UE-DTX
		cycle is applied. \n
			:param twoStreams: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Burst')
			:return: burst_length: 1| 2| 5"""
		twoStreams_cmd_val = self._cmd_group.get_repcap_cmd_value(twoStreams, repcap.TwoStreams)
		response = self._core.io.query_str(f'SOURce<HwInstance>:BB:W3GPp:MSTation:UDTX:BURSt{twoStreams_cmd_val}?')
		return Conversions.str_to_scalar_enum(response, enums.WcdmaUlDtxBurstLen)

	def clone(self) -> 'BurstCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = BurstCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
