from .......Internal.Core import Core
from .......Internal.CommandsGroup import CommandsGroup
from .......Internal import Conversions
from ....... import repcap


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class PeriodCls:
	"""Period commands group definition. 1 total commands, 0 Subgroups, 1 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("period", core, parent)

	def set(self, period: int, output=repcap.Output.Default) -> None:
		"""SCPI: [SOURce<HW>]:BB:TDSCdma:TRIGger:OUTPut<CH>:PERiod \n
		Snippet: driver.source.bb.tdscdma.trigger.output.period.set(period = 1, output = repcap.Output.Default) \n
		Sets the repetition rate for the signal at the marker outputs. *) If R&S SMW-B9 is installed, the minimum marker duration
		depends on the sample/symbol rate. See 'Marker Minimum Duration'. \n
			:param period: integer Range: 1 (R&S SMW-B10) / 1* (R&S SMW-B9) to (2^32-1) chips
			:param output: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Output')
		"""
		param = Conversions.decimal_value_to_str(period)
		output_cmd_val = self._cmd_group.get_repcap_cmd_value(output, repcap.Output)
		self._core.io.write(f'SOURce<HwInstance>:BB:TDSCdma:TRIGger:OUTPut{output_cmd_val}:PERiod {param}')

	def get(self, output=repcap.Output.Default) -> int:
		"""SCPI: [SOURce<HW>]:BB:TDSCdma:TRIGger:OUTPut<CH>:PERiod \n
		Snippet: value: int = driver.source.bb.tdscdma.trigger.output.period.get(output = repcap.Output.Default) \n
		Sets the repetition rate for the signal at the marker outputs. *) If R&S SMW-B9 is installed, the minimum marker duration
		depends on the sample/symbol rate. See 'Marker Minimum Duration'. \n
			:param output: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Output')
			:return: period: integer Range: 1 (R&S SMW-B10) / 1* (R&S SMW-B9) to (2^32-1) chips"""
		output_cmd_val = self._cmd_group.get_repcap_cmd_value(output, repcap.Output)
		response = self._core.io.query_str(f'SOURce<HwInstance>:BB:TDSCdma:TRIGger:OUTPut{output_cmd_val}:PERiod?')
		return Conversions.str_to_int(response)
