from ........Internal.Core import Core
from ........Internal.CommandsGroup import CommandsGroup
from ........Internal import Conversions
from ........ import repcap


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class ToffsetCls:
	"""Toffset commands group definition. 1 total commands, 0 Subgroups, 1 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("toffset", core, parent)

	def set(self, time_offset: float, satelliteSvid=repcap.SatelliteSvid.Default) -> None:
		"""SCPI: [SOURce<HW>]:BB:GNSS:SVID<CH>:NAVic:SDYNamics:TOFFset \n
		Snippet: driver.source.bb.gnss.svid.navic.sdynamics.toffset.set(time_offset = 1.0, satelliteSvid = repcap.SatelliteSvid.Default) \n
		Sets a time delay before the profile is applied. \n
			:param time_offset: float Range: 0 to 9E4
			:param satelliteSvid: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Svid')
		"""
		param = Conversions.decimal_value_to_str(time_offset)
		satelliteSvid_cmd_val = self._cmd_group.get_repcap_cmd_value(satelliteSvid, repcap.SatelliteSvid)
		self._core.io.write(f'SOURce<HwInstance>:BB:GNSS:SVID{satelliteSvid_cmd_val}:NAVic:SDYNamics:TOFFset {param}')

	def get(self, satelliteSvid=repcap.SatelliteSvid.Default) -> float:
		"""SCPI: [SOURce<HW>]:BB:GNSS:SVID<CH>:NAVic:SDYNamics:TOFFset \n
		Snippet: value: float = driver.source.bb.gnss.svid.navic.sdynamics.toffset.get(satelliteSvid = repcap.SatelliteSvid.Default) \n
		Sets a time delay before the profile is applied. \n
			:param satelliteSvid: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Svid')
			:return: time_offset: float Range: 0 to 9E4"""
		satelliteSvid_cmd_val = self._cmd_group.get_repcap_cmd_value(satelliteSvid, repcap.SatelliteSvid)
		response = self._core.io.query_str(f'SOURce<HwInstance>:BB:GNSS:SVID{satelliteSvid_cmd_val}:NAVic:SDYNamics:TOFFset?')
		return Conversions.str_to_float(response)
