from .......Internal.Core import Core
from .......Internal.CommandsGroup import CommandsGroup
from .......Internal import Conversions
from ....... import enums


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class NiotCls:
	"""Niot commands group definition. 2 total commands, 1 Subgroups, 1 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("niot", core, parent)

	@property
	def valid(self):
		"""valid commands group. 1 Sub-classes, 0 commands."""
		if not hasattr(self, '_valid'):
			from .Valid import ValidCls
			self._valid = ValidCls(self._core, self._cmd_group)
		return self._valid

	# noinspection PyTypeChecker
	def get_sub_config(self) -> enums.EutraNbiotInbandBitmapSfAll:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:UL:NIOT:SUBConfig \n
		Snippet: value: enums.EutraNbiotInbandBitmapSfAll = driver.source.bb.eutra.uplink.niot.get_sub_config() \n
		Sets the number of subframes in the bitmap. \n
			:return: sf_config: N10| N40
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:UL:NIOT:SUBConfig?')
		return Conversions.str_to_scalar_enum(response, enums.EutraNbiotInbandBitmapSfAll)

	def set_sub_config(self, sf_config: enums.EutraNbiotInbandBitmapSfAll) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:UL:NIOT:SUBConfig \n
		Snippet: driver.source.bb.eutra.uplink.niot.set_sub_config(sf_config = enums.EutraNbiotInbandBitmapSfAll.N10) \n
		Sets the number of subframes in the bitmap. \n
			:param sf_config: N10| N40
		"""
		param = Conversions.enum_scalar_to_str(sf_config, enums.EutraNbiotInbandBitmapSfAll)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:UL:NIOT:SUBConfig {param}')

	def clone(self) -> 'NiotCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = NiotCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
