from ........Internal.Core import Core
from ........Internal.CommandsGroup import CommandsGroup
from ........Internal import Conversions
from ........ import enums


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class NprsCls:
	"""Nprs commands group definition. 12 total commands, 3 Subgroups, 8 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("nprs", core, parent)

	@property
	def bmp(self):
		"""bmp commands group. 1 Sub-classes, 1 commands."""
		if not hasattr(self, '_bmp'):
			from .Bmp import BmpCls
			self._bmp = BmpCls(self._core, self._cmd_group)
		return self._bmp

	@property
	def mtia(self):
		"""mtia commands group. 0 Sub-classes, 1 commands."""
		if not hasattr(self, '_mtia'):
			from .Mtia import MtiaCls
			self._mtia = MtiaCls(self._core, self._cmd_group)
		return self._mtia

	@property
	def mtib(self):
		"""mtib commands group. 0 Sub-classes, 1 commands."""
		if not hasattr(self, '_mtib'):
			from .Mtib import MtibCls
			self._mtib = MtibCls(self._core, self._cmd_group)
		return self._mtib

	# noinspection PyTypeChecker
	def get_conf(self) -> enums.EutraNbiotNprsConfigType:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:CONF \n
		Snippet: value: enums.EutraNbiotNprsConfigType = driver.source.bb.eutra.downlink.niot.nprs.get_conf() \n
		Defines which type of NPRS is used. \n
			:return: nprs_para_cfg: PA_A| PA_B| PA_AB
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:CONF?')
		return Conversions.str_to_scalar_enum(response, enums.EutraNbiotNprsConfigType)

	def set_conf(self, nprs_para_cfg: enums.EutraNbiotNprsConfigType) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:CONF \n
		Snippet: driver.source.bb.eutra.downlink.niot.nprs.set_conf(nprs_para_cfg = enums.EutraNbiotNprsConfigType.PA_A) \n
		Defines which type of NPRS is used. \n
			:param nprs_para_cfg: PA_A| PA_B| PA_AB
		"""
		param = Conversions.enum_scalar_to_str(nprs_para_cfg, enums.EutraNbiotNprsConfigType)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:CONF {param}')

	def get_id(self) -> int:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:ID \n
		Snippet: value: int = driver.source.bb.eutra.downlink.niot.nprs.get_id() \n
		Sets the NPRS-ID used for the generation of the NPRS. \n
			:return: nprs_id: Integer Range: 0 to 4095
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:ID?')
		return Conversions.str_to_int(response)

	def set_id(self, nprs_id: int) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:ID \n
		Snippet: driver.source.bb.eutra.downlink.niot.nprs.set_id(nprs_id = 1) \n
		Sets the NPRS-ID used for the generation of the NPRS. \n
			:param nprs_id: Integer Range: 0 to 4095
		"""
		param = Conversions.decimal_value_to_str(nprs_id)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:ID {param}')

	# noinspection PyTypeChecker
	def get_period(self) -> enums.EutraNbiotNprsConfigbPeriod:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:PERiod \n
		Snippet: value: enums.EutraNbiotNprsConfigbPeriod = driver.source.bb.eutra.downlink.niot.nprs.get_period() \n
		For NPRS Part B configuration, sets the NPRS occasion period TNPRS. \n
			:return: nprs_period: PD_160| PD_320| PD_640| PD_1280
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:PERiod?')
		return Conversions.str_to_scalar_enum(response, enums.EutraNbiotNprsConfigbPeriod)

	def set_period(self, nprs_period: enums.EutraNbiotNprsConfigbPeriod) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:PERiod \n
		Snippet: driver.source.bb.eutra.downlink.niot.nprs.set_period(nprs_period = enums.EutraNbiotNprsConfigbPeriod.PD_1280) \n
		For NPRS Part B configuration, sets the NPRS occasion period TNPRS. \n
			:param nprs_period: PD_160| PD_320| PD_640| PD_1280
		"""
		param = Conversions.enum_scalar_to_str(nprs_period, enums.EutraNbiotNprsConfigbPeriod)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:PERiod {param}')

	def get_pow(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:POW \n
		Snippet: value: float = driver.source.bb.eutra.downlink.niot.nprs.get_pow() \n
		Sets the power of the narrowband positioning reference signal (NPRS) . \n
			:return: nprs_power: float Range: -80 to 10
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:POW?')
		return Conversions.str_to_float(response)

	def set_pow(self, nprs_power: float) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:POW \n
		Snippet: driver.source.bb.eutra.downlink.niot.nprs.set_pow(nprs_power = 1.0) \n
		Sets the power of the narrowband positioning reference signal (NPRS) . \n
			:param nprs_power: float Range: -80 to 10
		"""
		param = Conversions.decimal_value_to_str(nprs_power)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:POW {param}')

	def get_sein(self) -> int:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:SEIN \n
		Snippet: value: int = driver.source.bb.eutra.downlink.niot.nprs.get_sein() \n
		Specifies the index of the physical ressource block (PRB) containing the NPRS. \n
			:return: nprs_seq_info: integer Range: 0 to 174
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:SEIN?')
		return Conversions.str_to_int(response)

	def set_sein(self, nprs_seq_info: int) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:SEIN \n
		Snippet: driver.source.bb.eutra.downlink.niot.nprs.set_sein(nprs_seq_info = 1) \n
		Specifies the index of the physical ressource block (PRB) containing the NPRS. \n
			:param nprs_seq_info: integer Range: 0 to 174
		"""
		param = Conversions.decimal_value_to_str(nprs_seq_info)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:SEIN {param}')

	# noinspection PyTypeChecker
	def get_sfnm(self) -> enums.EutraNbiotNprsConfigbSfnumb:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:SFNM \n
		Snippet: value: enums.EutraNbiotNprsConfigbSfnumb = driver.source.bb.eutra.downlink.niot.nprs.get_sfnm() \n
		For NPRS Part B configuration, sets the number of consecutive DL subframes NNPRS within one NPRS positioning occasion. \n
			:return: nprs_sf_number: SFNM_10| SFNM_20| SFNM_40| SFNM_80| SFNM_160| SFNM_320| SFNM_640| SFNM_1280
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:SFNM?')
		return Conversions.str_to_scalar_enum(response, enums.EutraNbiotNprsConfigbSfnumb)

	def set_sfnm(self, nprs_sf_number: enums.EutraNbiotNprsConfigbSfnumb) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:SFNM \n
		Snippet: driver.source.bb.eutra.downlink.niot.nprs.set_sfnm(nprs_sf_number = enums.EutraNbiotNprsConfigbSfnumb.SFNM_10) \n
		For NPRS Part B configuration, sets the number of consecutive DL subframes NNPRS within one NPRS positioning occasion. \n
			:param nprs_sf_number: SFNM_10| SFNM_20| SFNM_40| SFNM_80| SFNM_160| SFNM_320| SFNM_640| SFNM_1280
		"""
		param = Conversions.enum_scalar_to_str(nprs_sf_number, enums.EutraNbiotNprsConfigbSfnumb)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:SFNM {param}')

	def get_state(self) -> bool:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:STATe \n
		Snippet: value: bool = driver.source.bb.eutra.downlink.niot.nprs.get_state() \n
		Enables the NPRS transmission. \n
			:return: nprs_state: 1| ON| 0| OFF
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:STATe?')
		return Conversions.str_to_bool(response)

	def set_state(self, nprs_state: bool) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:STATe \n
		Snippet: driver.source.bb.eutra.downlink.niot.nprs.set_state(nprs_state = False) \n
		Enables the NPRS transmission. \n
			:param nprs_state: 1| ON| 0| OFF
		"""
		param = Conversions.bool_to_str(nprs_state)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:STATe {param}')

	# noinspection PyTypeChecker
	def get_sts_frame(self) -> enums.EutraNbiotNprsConfigbStartsf:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:STSFrame \n
		Snippet: value: enums.EutraNbiotNprsConfigbStartsf = driver.source.bb.eutra.downlink.niot.nprs.get_sts_frame() \n
		For NPRS Part B configuration, sets the subframe offset ɑNPRS. \n
			:return: nprs_start_sf: STSF0_8| STSF1_8| STSF2_8| STSF3_8| STSF4_8| STSF5_8| STSF6_8| STSF7_8
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:STSFrame?')
		return Conversions.str_to_scalar_enum(response, enums.EutraNbiotNprsConfigbStartsf)

	def set_sts_frame(self, nprs_start_sf: enums.EutraNbiotNprsConfigbStartsf) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:NPRS:STSFrame \n
		Snippet: driver.source.bb.eutra.downlink.niot.nprs.set_sts_frame(nprs_start_sf = enums.EutraNbiotNprsConfigbStartsf.STSF0_8) \n
		For NPRS Part B configuration, sets the subframe offset ɑNPRS. \n
			:param nprs_start_sf: STSF0_8| STSF1_8| STSF2_8| STSF3_8| STSF4_8| STSF5_8| STSF6_8| STSF7_8
		"""
		param = Conversions.enum_scalar_to_str(nprs_start_sf, enums.EutraNbiotNprsConfigbStartsf)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:NPRS:STSFrame {param}')

	def clone(self) -> 'NprsCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = NprsCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
