from ........Internal.Core import Core
from ........Internal.CommandsGroup import CommandsGroup
from ........Internal import Conversions
from ........ import enums


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class GapCls:
	"""Gap commands group definition. 4 total commands, 2 Subgroups, 2 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("gap", core, parent)

	@property
	def config(self):
		"""config commands group. 0 Sub-classes, 1 commands."""
		if not hasattr(self, '_config'):
			from .Config import ConfigCls
			self._config = ConfigCls(self._core, self._cmd_group)
		return self._config

	@property
	def duration(self):
		"""duration commands group. 0 Sub-classes, 1 commands."""
		if not hasattr(self, '_duration'):
			from .Duration import DurationCls
			self._duration = DurationCls(self._core, self._cmd_group)
		return self._duration

	# noinspection PyTypeChecker
	def get_periodicity(self) -> enums.EutraNbiotGapPeriodicity:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:GAP:PERiodicity \n
		Snippet: value: enums.EutraNbiotGapPeriodicity = driver.source.bb.eutra.downlink.niot.gap.get_periodicity() \n
		Sets the number of subframes after that the configured gap is repeated. \n
			:return: gap_periodicity: 64| 128| 256| 512
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:GAP:PERiodicity?')
		return Conversions.str_to_scalar_enum(response, enums.EutraNbiotGapPeriodicity)

	def set_periodicity(self, gap_periodicity: enums.EutraNbiotGapPeriodicity) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:GAP:PERiodicity \n
		Snippet: driver.source.bb.eutra.downlink.niot.gap.set_periodicity(gap_periodicity = enums.EutraNbiotGapPeriodicity._128) \n
		Sets the number of subframes after that the configured gap is repeated. \n
			:param gap_periodicity: 64| 128| 256| 512
		"""
		param = Conversions.enum_scalar_to_str(gap_periodicity, enums.EutraNbiotGapPeriodicity)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:GAP:PERiodicity {param}')

	# noinspection PyTypeChecker
	def get_threshold(self) -> enums.EutraNbiotGapThreshold:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:GAP:THReshold \n
		Snippet: value: enums.EutraNbiotGapThreshold = driver.source.bb.eutra.downlink.niot.gap.get_threshold() \n
		Sets the gap threshold. \n
			:return: gap_threshold: 32| 64| 128| 256
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:EUTRa:DL:NIOT:GAP:THReshold?')
		return Conversions.str_to_scalar_enum(response, enums.EutraNbiotGapThreshold)

	def set_threshold(self, gap_threshold: enums.EutraNbiotGapThreshold) -> None:
		"""SCPI: [SOURce<HW>]:BB:EUTRa:DL:NIOT:GAP:THReshold \n
		Snippet: driver.source.bb.eutra.downlink.niot.gap.set_threshold(gap_threshold = enums.EutraNbiotGapThreshold._128) \n
		Sets the gap threshold. \n
			:param gap_threshold: 32| 64| 128| 256
		"""
		param = Conversions.enum_scalar_to_str(gap_threshold, enums.EutraNbiotGapThreshold)
		self._core.io.write(f'SOURce<HwInstance>:BB:EUTRa:DL:NIOT:GAP:THReshold {param}')

	def clone(self) -> 'GapCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = GapCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
