from .........Internal.Core import Core
from .........Internal.CommandsGroup import CommandsGroup
from .........Internal import Conversions
from ......... import repcap


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class MindexCls:
	"""Mindex commands group definition. 1 total commands, 0 Subgroups, 1 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("mindex", core, parent)

	def set(self, mindex: float, channel=repcap.Channel.Default) -> None:
		"""SCPI: [SOURce<HW>]:BB:BTOoth:DTTest:TABLe:LONG:SET<CH>:MINDex \n
		Snippet: driver.source.bb.btooth.dtTest.table.long.set.mindex.set(mindex = 1.0, channel = repcap.Channel.Default) \n
		Sets the modulation index, that specifies the frequency deviation. The modulation index h is defined as: with <img
		border='0' src='images/imga5f78ebf8bdb0adc0a00206a003a0641_1_--_--_PNG96.png' alt='' title=''>= 'symbol rate' , set with
		the command [:SOURce<hw>]:BB:BTOoth:SRATe:VARiation <img border='0'
		src='images/img388349408bdb4b210a00206a01f5817c_1_--_--_PNG96.png' alt='' title=''>= 'frequency deviation', set with the
		command [:SOURce<hw>]:BB:BTOoth:MSETtings:FDEViation According to the Bluetooth standard, the modulation index is allowed
		to vary between 0.28 and 0.35. \n
			:param mindex: float Range: 0.28 to 0.55
			:param channel: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Set')
		"""
		param = Conversions.decimal_value_to_str(mindex)
		channel_cmd_val = self._cmd_group.get_repcap_cmd_value(channel, repcap.Channel)
		self._core.io.write(f'SOURce<HwInstance>:BB:BTOoth:DTTest:TABLe:LONG:SET{channel_cmd_val}:MINDex {param}')

	def get(self, channel=repcap.Channel.Default) -> float:
		"""SCPI: [SOURce<HW>]:BB:BTOoth:DTTest:TABLe:LONG:SET<CH>:MINDex \n
		Snippet: value: float = driver.source.bb.btooth.dtTest.table.long.set.mindex.get(channel = repcap.Channel.Default) \n
		Sets the modulation index, that specifies the frequency deviation. The modulation index h is defined as: with <img
		border='0' src='images/imga5f78ebf8bdb0adc0a00206a003a0641_1_--_--_PNG96.png' alt='' title=''>= 'symbol rate' , set with
		the command [:SOURce<hw>]:BB:BTOoth:SRATe:VARiation <img border='0'
		src='images/img388349408bdb4b210a00206a01f5817c_1_--_--_PNG96.png' alt='' title=''>= 'frequency deviation', set with the
		command [:SOURce<hw>]:BB:BTOoth:MSETtings:FDEViation According to the Bluetooth standard, the modulation index is allowed
		to vary between 0.28 and 0.35. \n
			:param channel: optional repeated capability selector. Default value: Nr1 (settable in the interface 'Set')
			:return: mindex: float Range: 0.28 to 0.55"""
		channel_cmd_val = self._cmd_group.get_repcap_cmd_value(channel, repcap.Channel)
		response = self._core.io.query_str(f'SOURce<HwInstance>:BB:BTOoth:DTTest:TABLe:LONG:SET{channel_cmd_val}:MINDex?')
		return Conversions.str_to_float(response)
