from ......Internal.Core import Core
from ......Internal.CommandsGroup import CommandsGroup
from ......Internal import Conversions
from ......Internal.Utilities import trim_str_response
from ...... import enums


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class EthernetCls:
	"""Ethernet commands group definition. 12 total commands, 2 Subgroups, 2 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._cmd_group = CommandsGroup("ethernet", core, parent)

	@property
	def statistics(self):
		"""statistics commands group. 6 Sub-classes, 1 commands."""
		if not hasattr(self, '_statistics'):
			from .Statistics import StatisticsCls
			self._statistics = StatisticsCls(self._core, self._cmd_group)
		return self._statistics

	@property
	def waveform(self):
		"""waveform commands group. 1 Sub-classes, 2 commands."""
		if not hasattr(self, '_waveform'):
			from .Waveform import WaveformCls
			self._waveform = WaveformCls(self._core, self._cmd_group)
		return self._waveform

	# noinspection PyTypeChecker
	def get_mode(self) -> enums.ArbEthMode:
		"""SCPI: [SOURce<HW>]:BB:ARBitrary:ETHernet:MODE \n
		Snippet: value: enums.ArbEthMode = driver.source.bb.arbitrary.ethernet.get_mode() \n
		Sets the Ethernet mode for the waveform data upload via the Ethernet connection. \n
			:return: mode: M10G| M40G M10G 10 Gbit Ethernet mode M40G 40 Gbit Ethernet mode
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:ARBitrary:ETHernet:MODE?')
		return Conversions.str_to_scalar_enum(response, enums.ArbEthMode)

	def set_mode(self, mode: enums.ArbEthMode) -> None:
		"""SCPI: [SOURce<HW>]:BB:ARBitrary:ETHernet:MODE \n
		Snippet: driver.source.bb.arbitrary.ethernet.set_mode(mode = enums.ArbEthMode.M10G) \n
		Sets the Ethernet mode for the waveform data upload via the Ethernet connection. \n
			:param mode: M10G| M40G M10G 10 Gbit Ethernet mode M40G 40 Gbit Ethernet mode
		"""
		param = Conversions.enum_scalar_to_str(mode, enums.ArbEthMode)
		self._core.io.write(f'SOURce<HwInstance>:BB:ARBitrary:ETHernet:MODE {param}')

	def get_status(self) -> str:
		"""SCPI: [SOURce<HW>]:BB:ARBitrary:ETHernet:STATus \n
		Snippet: value: str = driver.source.bb.arbitrary.ethernet.get_status() \n
		Queries the status of the ARB Ethernet upload application. \n
			:return: status: string 'No Waveform' No waveform available as source. 'Uploading' Loads ARB waveform data into the ARB memory of the R&S SMW. The R&S SMW receives the data from the external device connected to the QSFP network. 'Waiting for Trigger' Expects a trigger signal as configured for starting ARB Ethernet upload. 'Running' Plays the waveform, that is loaded into the ARB memory of the R&S SMW.
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:ARBitrary:ETHernet:STATus?')
		return trim_str_response(response)

	def clone(self) -> 'EthernetCls':
		"""Clones the group by creating new object from it and its whole existing subgroups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = EthernetCls(self._core, self._cmd_group.parent)
		self._cmd_group.synchronize_repcaps(new_group)
		return new_group
