# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['playhdl', 'playhdl.templates']

package_data = \
{'': ['*']}

entry_points = \
{'console_scripts': ['playhdl = playhdl.cli:main']}

setup_kwargs = {
    'name': 'playhdl',
    'version': '0.2.0',
    'description': 'Tool to play with HDL (inspired by EdaPlayground)',
    'long_description': '# playhdl\n\nYou can think about `playhdl` as [EdaPlayground](https://edaplayground.com/), but which is CLI-based and uses simulators on your local machine.\n\nIt gives you ability to simulate tiny snippets of HDL code in several commands without any headache related to vast tool guides, command-line arguments and etc.:\n\n```sh\nplayhdl init sv\nplayhdl run modelsim\n```\n\n## Features\n\n`playhdl` is written in pure Python without any external dependencies, so it is easy to use it in any environment (laptop, server, etc.), where Python 3.8+ is available.\n\nIt supports various project types (HDL languages + libraries) and many simulators:\n\n|               |    verilog         |      sv            |   sv_uvm12         |     vhdl      |\n| ------------- | :----------------: | :----------------: | :----------------: | :-----------: |\n| [modelsim](https://eda.sw.siemens.com/en-US/ic/modelsim/) | :heavy_check_mark: | :heavy_check_mark: | | |\n| [xcelium](https://www.cadence.com/ko_KR/home/tools/system-design-and-verification/simulation-and-testbench-verification/xcelium-simulator.html) | :heavy_check_mark: | :heavy_check_mark: | | |\n| [verilator](https://www.veripool.org/verilator/) | :heavy_check_mark: | :heavy_check_mark: | | |\n| [icarus](http://iverilog.icarus.com/) | :heavy_check_mark: | :heavy_check_mark: | | |\n| [vcs](https://www.synopsys.com/verification/simulation/vcs.html) | :heavy_check_mark: | :heavy_check_mark: | :heavy_check_mark: | |\n| [vivado](https://www.xilinx.com/products/design-tools/vivado.html) | :heavy_check_mark: | :heavy_check_mark: | :heavy_check_mark: | |\n\n## Quick start\n\n* Install the latest stable release (Python 3.8+ is required):\n\n```sh\npython -m pip install -U playhdl\n```\n\n* Setup settings file `$HOME/.playhdl/settings.json` with a list of all automatically-discoverd simulators. Edit file manually to add undiscovered ones. This have to be done only once.\n\n```sh\nplayhdl setup\n```\n\n* Initialize project file `playhdl.json` and template testbench in the current directory. Project file contains specific commands to be executed for compilation and simulation processes. Edit it manually to tweak tool arguments if required.\n\n```sh\nplayhdl sv # this will create ready-to-simulate tb.sv\n```\n\n* Run simulation in one of the supported simulators for this project (language):\n\n```sh\nplayhdl run icarus\n# to open waves after simulation\nplayhdl run icarus --waves\n```\n\n## Tool command guide\n\nTo get general help and command list:\n\n```sh\nplayhdl -h\n```\n\nTo get help about specific command\n\n```sh\nplayhdl <command> -h\n```\n\n### `setup` command\n\nSettings of the tool are stored in the JSON file under `$HOME/.playhdl` directory.\n\nTo create `$HOME/.playhdl/settings.json` run\n\n```sh\nplayhdl setup\n```\n\nIt will try to find all supported simulators and fill the json. If you have multiple versions of simulators or some of them were not found, add them manually to your settings file.\n\nSettings file structure:\n\n```json\n{\n    "tools": {\n        "<tool_uid>": {\n            "kind": "<tool_kind>",\n            "bin_dir": "<path_to_bin>",\n            "env": {},\n            "extras": {}\n        }\n    }\n}\n```\n\nAll tools settings are located in a dictionary under `"tools"` key.\n\nEvery tool has it is own `tool_uid`, which is just a string with any unique name, e.g. `"modelsim"`, `"verilator5"`, `"my_secret_simulator"`.\n\nValid `"kind"` must be provided:\n\n* `"modelsim"`\n* `"xcelium"`\n* `"verilator"`\n* `"icarus"`\n* `"vcs"`\n* `"vivado"`\n\nOther fields:\n\n* `"bin_dir"` - a string with a path to a directory with executable files\n* `"env"` - a dictionary with additional enviroment variables (keys and values are strings)\n* `"extras"` - a dictionary with extra values for a specific simulator kind\n\nExtras for `"vcs"` kind:\n\n* `"gui"` - `"verdi"` or `"dve"` select default GUI for VCS\n\n### `init` command\n\nThis command creates JSON project file `playhdl.json` and HDL testbench in the current directory.\n\n```sh\nplayhdl init <mode>\n```\n\nWhere `<mode>` is one of the supported project modes:\n\n* `verilog` - Verilog-2001\n* `sv` - SystemVerilog-2017\n* `sv_uvm12` - SystemVerilog-2017 + UVM 1.2\n* `vhdl` - VHDL-93\n\nProject file is filled with scripts for all suitable simulators for the selected mode. It\'s internal structure:\n\n```json\n{\n    "tools": {\n        "<tool_uid>": {\n            "build": [\n                "<cmd0>",\n                "<cmd1>",\n                "<cmd2>"\n            ],\n            "sim": [\n                "<cmd>"\n            ],\n            "waves": [\n                "<cmd>"\n            ]\n        }\n    }\n}\n```\n\nThere are three categories of commands:\n\n* `"build"` - commands needed to compile and elaborate sources\n* `"sim"` - commands needed to run simulation\n* `"waves"` - commands needed to open waves for analysis\n\nAny command can be customized for specific needs.\n\n### `run` command\n\nThis command runs CLI-mode simulation in a specific simulator according to project file\n\n```sh\nplayhdl run <tool_uid>\n```\n\nArgument `--waves` can be added to open waves for analysis after simulation ends\n\n```sh\nplayhdl run <tool_uid> --waves\n```\n\n### `info` command\n\nThis command just prints some useful information:\n\n* all tools specified in your settings file\n* current compatibility table between project mode and simulator\n\n```sh\nplayhdl info\n```\n\n## Developer guide\n\nInstall `poetry`\n\n```sh\npython -m pip install -U poetry\n```\n\nSetup virtual environment\n\n```sh\nmake setup-dev\n```\n\nYou can specify Python version to use\n\n```sh\nmake setup-dev PYTHON_VERSION=3.9\n```\n\nTo run `playhdl` from sources\n\n```sh\npoetry run playhdl <args>\n```\n\n`Makefile` provides additional ways to interact with project:\n\n* `make format` - auto-format all sources\n* `make check-format` - check current formatting of sources\n* `make lint` - perform linting\n* `make type` - perform type checking\n* `make test` - run all tests\n* `make pre-commit` - shorthand for combination of `check-format`, `lint`, `type`, `test`\n\n## Miscellaneous\n\n### Offline install\n\nFor an offline install you have several options of how to get `wheel`:\n\n* build locally using [poetry](https://python-poetry.org/)\n\n```sh\npython -m pip install -U poetry\npoetry build\n```\n\n* download `.whl` from [PyPi](https://pypi.org/)\n\n```sh\npython -m pip download playhdl\n```\n\nThen you can use pip to install it on an offline machine:\n\n```sh\npython -m pip install <wheel_file_name>.whl\n```\n',
    'author': 'esynr3z',
    'author_email': 'esynr3z@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/esynr3z/playhdl',
    'packages': packages,
    'package_data': package_data,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
