"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Destination = exports.S3OutputFilePrefixType = exports.S3OutputFilePrefixFormat = exports.S3OutputFilePrefixHierarchy = exports.S3OutputFileType = exports.S3OutputAggregationType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const type_1 = require("./type");
const appflow_permissions_manager_1 = require("../core/appflow-permissions-manager");
const flows_1 = require("../core/flows");
;
;
var S3OutputAggregationType;
(function (S3OutputAggregationType) {
    S3OutputAggregationType["NONE"] = "None";
    S3OutputAggregationType["SINGLE_FILE"] = "SingleFile";
})(S3OutputAggregationType = exports.S3OutputAggregationType || (exports.S3OutputAggregationType = {}));
var S3OutputFileType;
(function (S3OutputFileType) {
    S3OutputFileType["CSV"] = "CSV";
    S3OutputFileType["JSON"] = "JSON";
    S3OutputFileType["PARQUET"] = "PARQUET";
})(S3OutputFileType = exports.S3OutputFileType || (exports.S3OutputFileType = {}));
var S3OutputFilePrefixHierarchy;
(function (S3OutputFilePrefixHierarchy) {
    S3OutputFilePrefixHierarchy["EXECUTION_ID"] = "EXECUTION_ID";
    S3OutputFilePrefixHierarchy["SCHEMA_VERSION"] = "SCHEMA_VERSION";
})(S3OutputFilePrefixHierarchy = exports.S3OutputFilePrefixHierarchy || (exports.S3OutputFilePrefixHierarchy = {}));
var S3OutputFilePrefixFormat;
(function (S3OutputFilePrefixFormat) {
    S3OutputFilePrefixFormat["DAY"] = "DAY";
    S3OutputFilePrefixFormat["HOUR"] = "HOUR";
    S3OutputFilePrefixFormat["MINUTE"] = "MINUTE";
    S3OutputFilePrefixFormat["MONTH"] = "MONTH";
    S3OutputFilePrefixFormat["YEAR"] = "YEAR";
})(S3OutputFilePrefixFormat = exports.S3OutputFilePrefixFormat || (exports.S3OutputFilePrefixFormat = {}));
var S3OutputFilePrefixType;
(function (S3OutputFilePrefixType) {
    S3OutputFilePrefixType["FILENAME"] = "FILENAME";
    S3OutputFilePrefixType["PATH"] = "PATH";
    S3OutputFilePrefixType["PATH_AND_FILE"] = "PATH_AND_FILE";
})(S3OutputFilePrefixType = exports.S3OutputFilePrefixType || (exports.S3OutputFilePrefixType = {}));
class S3Destination {
    constructor(props) {
        this.props = props;
        this.connectorType = type_1.S3ConnectorType.instance;
        this.compatibleFlows = [flows_1.FlowType.ON_DEMAND, flows_1.FlowType.SCHEDULED];
    }
    buildAggregationConfig(aggregation) {
        return aggregation && {
            aggregationType: aggregation.type,
            // TODO: make sure it should use mebibytes
            targetFileSize: aggregation.fileSize && aggregation.fileSize,
        };
    }
    buildPrefixConfig(filePrefix) {
        return filePrefix && {
            pathPrefixHierarchy: filePrefix.hierarchy,
            prefixFormat: filePrefix.format,
            prefixType: filePrefix.type,
        };
    }
    bind(flow) {
        // TODO: make sure this even makes sense
        if (!this.compatibleFlows.includes(flow.type)) {
            throw new Error(`Flow of type ${flow.type} does not support S3 as a destination`);
        }
        this.tryAddNodeDependency(flow, this.props.location.bucket);
        appflow_permissions_manager_1.AppFlowPermissionsManager.instance().grantBucketWrite(this.props.location.bucket);
        if (this.props.catalog) {
            const { role, policy } = this.buildRoleAndPolicyForCatalog(flow);
            this.tryAddNodeDependency(flow, this.props.catalog.database);
            this.tryAddNodeDependency(flow, role);
            this.tryAddNodeDependency(flow, policy);
            flow._addCatalog({
                glueDataCatalog: {
                    databaseName: this.props.catalog.database.databaseName,
                    tablePrefix: this.props.catalog.tablePrefix,
                    roleArn: role.roleArn,
                },
            });
        }
        return {
            connectorType: this.connectorType.asProfileConnectorType,
            destinationConnectorProperties: this.buildDestinationConnectorProperties(),
        };
    }
    /**
     * see: https://docs.aws.amazon.com/appflow/latest/userguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-access-gdc
     * see: https://docs.aws.amazon.com/appflow/latest/userguide/security_iam_service-role-policies.html#access-gdc
     * @param flow
     * @returns a tuple consisting of a role for cataloguing with a specified policy
     */
    buildRoleAndPolicyForCatalog(flow) {
        const role = new aws_iam_1.Role(flow.stack, `${flow.node.id}GlueAccessRole`, {
            assumedBy: new aws_iam_1.ServicePrincipal('appflow.amazonaws.com'),
        });
        const policy = new aws_iam_1.Policy(flow.stack, `${flow.node.id}GlueAccessRolePolicy`, {
            roles: [role],
            statements: [
                new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: [
                        'glue:BatchCreatePartition',
                        'glue:CreatePartitionIndex',
                        'glue:DeleteDatabase',
                        'glue:GetTableVersions',
                        'glue:GetPartitions',
                        'glue:BatchDeletePartition',
                        'glue:DeleteTableVersion',
                        'glue:UpdateTable',
                        'glue:DeleteTable',
                        'glue:DeletePartitionIndex',
                        'glue:GetTableVersion',
                        'glue:CreatePartition',
                        'glue:UntagResource',
                        'glue:UpdatePartition',
                        'glue:TagResource',
                        'glue:UpdateDatabase',
                        'glue:CreateTable',
                        'glue:BatchUpdatePartition',
                        'glue:GetTables',
                        'glue:BatchGetPartition',
                        'glue:GetDatabases',
                        'glue:GetPartitionIndexes',
                        'glue:GetTable',
                        'glue:GetDatabase',
                        'glue:GetPartition',
                        'glue:CreateDatabase',
                        'glue:BatchDeleteTableVersion',
                        'glue:BatchDeleteTable',
                        'glue:DeletePartition',
                    ],
                    resources: ['*'],
                }),
            ],
        });
        return { role, policy };
    }
    buildDestinationConnectorProperties() {
        const bucketName = this.props.location.bucket.bucketName;
        if (!bucketName) {
            throw new Error('bucketName is required');
        }
        return {
            s3: {
                bucketName: bucketName,
                bucketPrefix: this.props.location.prefix,
                s3OutputFormatConfig: this.props.formatting && {
                    aggregationConfig: this.buildAggregationConfig(this.props.formatting.aggregation),
                    fileType: this.props.formatting.fileType ?? S3OutputFileType.JSON,
                    prefixConfig: this.buildPrefixConfig(this.props.formatting.filePrefix),
                    preserveSourceDataTyping: this.props.formatting.preserveSourceDataTypes,
                },
            },
        };
    }
    tryAddNodeDependency(scope, resource) {
        if (resource && typeof resource !== 'string') {
            scope.node.addDependency(resource);
        }
    }
}
exports.S3Destination = S3Destination;
_a = JSII_RTTI_SYMBOL_1;
S3Destination[_a] = { fqn: "@cdklabs/cdk-appflow.S3Destination", version: "0.0.2" };
//# sourceMappingURL=data:application/json;base64,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