"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowBase = exports.DataPullMode = exports.FlowType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appflow_1 = require("aws-cdk-lib/aws-appflow");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const flow_time_updater_1 = require("./flow-time-updater");
var FlowType;
(function (FlowType) {
    FlowType["EVENT"] = "Event";
    FlowType["ON_DEMAND"] = "OnDemand";
    FlowType["SCHEDULED"] = "Scheduled";
})(FlowType = exports.FlowType || (exports.FlowType = {}));
var DataPullMode;
(function (DataPullMode) {
    DataPullMode["COMPLETE"] = "Complete";
    DataPullMode["INCREMENTAL"] = "Incremental";
})(DataPullMode = exports.DataPullMode || (exports.DataPullMode = {}));
class FlowBase extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.mappings = [];
        this.validations = [];
        this.transforms = [];
        this.filters = [];
        this.type = props.type;
        this._projectionFilter = this.initProjectionFilter(props.source.connectorType);
        this.name = props.name || id;
        const resource = new aws_appflow_1.CfnFlow(this, id, {
            flowName: this.name,
            triggerConfig: {
                triggerType: props.type,
                triggerProperties: props.triggerConfig
                    && props.triggerConfig.properties
                    && this.buildTriggerProperties(scope, id, props.triggerConfig.properties),
            },
            kmsArn: props.key?.keyArn,
            metadataCatalogConfig: aws_cdk_lib_1.Lazy.any({ produce: () => this._catalogMetadata }),
            description: props.description,
            sourceFlowConfig: {
                ...props.source.bind(this),
                incrementalPullConfig: this.buildIncrementalPullConfig(props.triggerConfig),
            },
            // NOTE: currently only a single destination is allowed with AppFlow
            //       it might require a change of approach in the future.
            destinationFlowConfigList: [props.destination.bind(this)],
            tasks: aws_cdk_lib_1.Lazy.any({
                produce: () => {
                    const tasks = [
                        ...this.mappings,
                        ...this.transforms,
                        ...this.filters,
                        ...this.validations,
                    ];
                    // TODO: make sure that this.filters doesn't already have a projection definition
                    if (this._projectionFilter.sourceFields.length > 0) {
                        tasks.unshift(this._projectionFilter);
                    }
                    return tasks;
                },
            }),
        });
        this.arn = resource.attrFlowArn;
        this.source = props.source;
        props.mappings.forEach(m => this._addMapping(m));
        if (props.validations) {
            props.validations.forEach(v => this._addValidation(v));
        }
        if (props.transforms) {
            props.transforms.forEach(t => this._addTransform(t));
        }
        if (props.filters) {
            props.filters.forEach(f => this._addFilter(f));
        }
        this.node.addValidation({
            validate: () => this.mappings.length === 0 ? ['A Flow must have at least one mapping'] : [],
        });
    }
    buildTriggerProperties(scope, id, props) {
        const updater = new flow_time_updater_1.FlowTimeUpdater(scope, `${id}Updater`, {
            schedule: props.schedule,
            startTime: props.properties?.startTime,
            endTime: props.properties?.endTime,
        });
        this.node.addDependency(updater);
        return {
            dataPullMode: props.dataPullConfig.mode,
            flowErrorDeactivationThreshold: props.flowErrorDeactivationThreshold,
            scheduleExpression: updater.scheduleExpression,
            firstExecutionFrom: props.properties?.firstExecutionFrom &&
                Math.floor(props.properties.firstExecutionFrom.getTime() / 1000),
            scheduleStartTime: props.properties?.startTime && updater.startTime,
            scheduleEndTime: props.properties?.endTime && updater.endTime,
        };
    }
    initProjectionFilter(sourceType) {
        const filterConnectorOperator = {};
        filterConnectorOperator[sourceType.asTaskConnectorOperatorOrigin] = 'PROJECTION';
        return {
            taskType: 'Filter',
            connectorOperator: filterConnectorOperator,
            sourceFields: [],
        };
    }
    /**
       * Set the catalog definitionfor the flow
       *
       * @internal
       */
    _addCatalog(metadata) {
        this._catalogMetadata = metadata;
    }
    /**
     *
     * @param validation
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addValidation(validation) {
        this.validations.push(...validation.bind(this, this.source));
        return this;
    }
    /**
     *
     * @param mapping
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addMapping(mapping) {
        const boundMappingTasks = mapping.bind(this, this.source);
        this.addProjectionField(boundMappingTasks);
        this.mappings.push(...boundMappingTasks);
        return this;
    }
    /**
     *
     * @param filter
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addFilter(filter) {
        const boundFilterTasks = filter.bind(this, this.source);
        this.addProjectionField(boundFilterTasks);
        this.filters.push(...boundFilterTasks);
        return this;
    }
    /**
     *
     * @param transform
     * @returns the flow to which the validation was added
     *
     * @internal
     */
    _addTransform(transform) {
        this.transforms.push(...transform.bind(this, this.source));
        return this;
    }
    addProjectionField(boundMappingTasks) {
        // TODO: test if this satisfies all the requirements.
        boundMappingTasks.forEach(boundMapping => {
            if (['Map', 'Filter'].includes(boundMapping.taskType)) {
                boundMapping.sourceFields.forEach(field => {
                    if (!this._projectionFilter.sourceFields.includes(field)) {
                        this._projectionFilter.sourceFields.push(field);
                    }
                });
            }
        });
    }
    // see: https://docs.aws.amazon.com/appflow/latest/userguide/flow-notifications.html
    onEvent(id, options = {}) {
        const rule = new aws_events_1.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.appflow'],
            resources: [this.arn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    onRunStarted(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['AppFlow Start Flow Run Report'],
        });
        return rule;
    }
    onRunCompleted(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['AppFlow End Flow Run Report'],
        });
        return rule;
    }
    buildIncrementalPullConfig(triggerConfig) {
        return triggerConfig && triggerConfig.properties && triggerConfig.properties.dataPullConfig
            && triggerConfig.properties.dataPullConfig.timestampField
            ? {
                datetimeTypeFieldName: triggerConfig.properties.dataPullConfig.timestampField,
            }
            : undefined;
    }
}
exports.FlowBase = FlowBase;
_a = JSII_RTTI_SYMBOL_1;
FlowBase[_a] = { fqn: "@cdklabs/cdk-appflow.FlowBase", version: "0.0.2" };
//# sourceMappingURL=data:application/json;base64,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